﻿namespace TinyFM8
{
    using Jacobi.Vst.Core;
    using Jacobi.Vst.Framework;
    using System;
    using KSDManager;

    /// <summary>
    /// Manages incoming midi events and sents them to the <see cref="SampleManager"/>.
    /// </summary>
    public class MidiProcessor : IVstMidiProcessor
    {
        private Plugin _plugin;
        public bool[] Notes          = new bool[255];
        public int[]  NotesTimestamp = new int[255];
        public int    LastNote;

        /// <summary>
        /// Constructs a new instance.
        /// </summary>
        /// <param name="plugin"></param>
        public MidiProcessor(Plugin plugin)
        {
            _plugin = plugin;
        }

        #region IVstMidiProcessor Members

        /// <summary>
        /// Always returns 16.
        /// </summary>
        public int ChannelCount
        {
            get { return 16; }
        }


        /// <summary>
        /// Handles and filters the incoming midi events.
        /// </summary>
        /// <param name="events">The midi events for the current cycle.</param>
        public void Process(VstEventCollection events)
        {
            // Exit if not voices are loaded
            if ((_plugin.AudioProcessor.ActiveVoices == null) || (_plugin.AudioProcessor.ActiveVoices.Count + _plugin.AudioProcessor.OffVoices.Count == 0))
                return;

            lock (_plugin.AudioProcessor.ActiveVoices)
            {
                foreach (VstEvent evnt in events)
                {
                    if (evnt.EventType == VstEventTypes.MidiEvent)
                    {
                        VstMidiEvent midiEvent = (VstMidiEvent)evnt;

                        LastNote = midiEvent.Data[1];
                        System.Diagnostics.Debug.WriteLine(string.Format("Receiving Data0: {0}  Data1: {1}", midiEvent.Data[0], midiEvent.Data[1]));

                        // Update Note velocity

                        // pass note on and note off to the sample manager
                        if ((midiEvent.Data[0] & 0xF0) == 0x80)
                        {
                            for (int i = _plugin.AudioProcessor.ActiveVoices.Count - 1; i >= 0; i--)
                            {
                                var voice = _plugin.AudioProcessor.ActiveVoices[i];
                                if (SoundRenderInterface.GetNote(voice) == midiEvent.Data[1])
                                {
                                    System.Diagnostics.Debug.WriteLine("Off");
                                    SoundRenderInterface.OffNote(voice);
                                    _plugin.AudioProcessor.ActiveVoices.Remove(voice);
                                    _plugin.AudioProcessor.OffVoices.Insert(_plugin.AudioProcessor.OffVoices.Count, voice);
                                }
                            }
                        }

                        if ((midiEvent.Data[0] & 0xF0) == 0x90)
                        {
                            // note on with velocity = 0 is a note off
                            if (midiEvent.Data[2] == 0)
                            {
                                for (int i = _plugin.AudioProcessor.ActiveVoices.Count - 1; i >= 0; i--) 
                                {
                                    var voice = _plugin.AudioProcessor.ActiveVoices[i];
                                    if (SoundRenderInterface.GetNote(voice) == midiEvent.Data[1])
                                    {
                                        System.Diagnostics.Debug.WriteLine("Off");
                                        SoundRenderInterface.OffNote(voice);
                                        _plugin.AudioProcessor.ActiveVoices.Remove(voice);
                                        _plugin.AudioProcessor.OffVoices.Insert(_plugin.AudioProcessor.OffVoices.Count, voice);
                                    }
                                }
                            }
                            else
                            {
                                IntPtr voice = _plugin.AudioProcessor.ActiveVoices.Count != 0 ? _plugin.AudioProcessor.ActiveVoices[0] : _plugin.AudioProcessor.OffVoices[0];
                                int nVoices = SoundRenderInterface.GetVoices(voice);

                                for (int i = 0; i < nVoices; i++)
                                {
                                    // Get a free voice
                                    if (_plugin.AudioProcessor.OffVoices.Count > 0)
                                    {
                                        voice = _plugin.AudioProcessor.OffVoices[0];
                                        _plugin.AudioProcessor.OffVoices.RemoveAt(0);
                                    }
                                    else
                                    {
                                        voice = _plugin.AudioProcessor.ActiveVoices[0];
                                        _plugin.AudioProcessor.ActiveVoices.RemoveAt(0);
                                    }

                                    // Queue in active voice
                                    System.Diagnostics.Debug.WriteLine("On");
                                    _plugin.AudioProcessor.ActiveVoices.Insert(_plugin.AudioProcessor.ActiveVoices.Count, voice);

                                    SoundRenderInterface.SetVelocity(voice, midiEvent.Data[2] / 127f);
                                    SoundRenderInterface.PlayNote(voice, midiEvent.Data[1], i);
                                }
                            }
                        }
                    }
                }
            }
        }

        #endregion
    }
}
