#include <math.h>

#include "FM8_AppAbstractionLayer.h"
#include "FM8_reverb.h"

void FM8_Reverb::Setup()
{
    // compute reverb time (in samples) 
    // LV_Time is time in 10ms unit. 
    int seed = 10;
    int time = LV_Time * SAMPLE_RATE / 100;

    for (int k = 0; k < COMB_FILTERS; k++) 
    {
        // compute delay line length
        length[k] = (0.25f * k / NUMDELAYLINES + 0.25f) * time;

        // compute decaying factors
        float f = 1.f; 
        c[k] = (f * FM8_expf(-length[k] * /*ln(2.0f)*/ 0.693147181f / time));

        // reset lowpass Z-1 value
        low[k] = 0;
    }

    // Set allpass filters delay lines
    length[ALLPASS_0 + 0] = (int)(142 * 44.1 / 30);
    length[ALLPASS_0 + 1] = (int)(107 * 44.1 / 30);
    length[ALLPASS_0 + 2] = (int)(379 * 44.1 / 30);
    length[ALLPASS_0 + 3] = (int)(277 * 44.1 / 30);

    // create delay lines
    for (int k = 0; k < NUMDELAYLINES; k++) 
    {
        buffers[k] = new float[length[k]];
        FM8_memset(buffers[k], 0, length[k]*4);

        pos[k] = 0;
    }
}

float& FM8_Reverb::DelayLine(int index)
{
    // get sample from delay line
    float &val = buffers[index][pos[index]];

    // Increment position
    if (++pos[index] >= length[index])
        pos[index]=0;

    return val;
}

void FM8_Reverb::Process(float& sample)
{	
    float damping         = LV_Damping     / 100.f;
    float bandwidth       = LV_BandLimiter / 100.f;
    float input_diffuse_1 = LV_InDiffuse1  / 100.f;
    float input_diffuse_2 = LV_InDiffuse2  / 100.f;

    // High-frequency attenuation  on input;  full bandwidth  =  0.9999999
    // in = sample * BW + Prev * (1 - BW)  =>
    // in = sample * BW + Prev - Prev * BW =>
    // in = BW * (sample - Prev) + Prev 
    lowpass_out += bandwidth * (sample - lowpass_out); 
    float in = lowpass_out;
    float delayOut;
    
    // Allpass #1 (Decorrelates incoming signal)
    float& val1 = DelayLine(ALLPASS_0);
    delayOut = val1;
    val1 = in - val1 * input_diffuse_1;
    in = val1 * input_diffuse_1 + delayOut;

    // Allpass #2
    float& val2 = DelayLine(ALLPASS_0+1);
    delayOut = val2;
    val2 = in - val2 * input_diffuse_1;
    in = val2 * input_diffuse_1 + delayOut;

    // Allpass #2
    float& val3 = DelayLine(ALLPASS_0+2);
    delayOut = val3;
    val3 = in - val3 * input_diffuse_2;
    in = val3 * input_diffuse_2 + delayOut;

    // Allpass #3
    float& val4 = DelayLine(ALLPASS_0+3);
    delayOut = val4;
    val4 = in - val4 * input_diffuse_2;
    in = val4 * input_diffuse_2 + delayOut;

    float out = 0;
    for (int k = 0; k < COMB_FILTERS; k++) 
    {
        // get sample from delay line
        float &val = DelayLine(k);
        
        // add to output
        out += val;

        // execute IIR
        low[k] += damping * (val - low[k]);

        // store mixed sample to delay line
        val = (low[k] * c[k]) + in;
    }

    // return output
    sample = out;
}

void FM8_ReverbRender(FM8_Reverb* pLeft, FM8_Reverb* pRight, float* pBuffer, int nBufferSize)
{
	for (int i=0; i<nBufferSize; i++)
	{
		pLeft->Process(pBuffer[i * 2 + 0]);
		pRight->Process(pBuffer[i * 2 + 1]);
	}
}

int FM8_CreateReverbs(void* pBlock, FM8_Reverb** pReverb)
{
	int nCount = LoadInterlacedStruct((UInt8*&)pBlock, (UInt8*&)pReverb[0], sizeof(FM8_Reverb), 6);

	// Run Setup
	for (int i = 0; i < nCount; i++)
	{
		pReverb[i] = &pReverb[0][i];
		pReverb[i]->Setup();
	}

	return nCount;
}