﻿// Copyright 2008 Alexandre Mutel
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
using System;
using System.Data;
using System.Linq;
using System.Windows.Forms;
using NRenoiseTools.Xrns2MidiApp;
using Song = NRenoiseTools.RenoiseSong;
using Instrument = NRenoiseTools.RenoiseInstrument;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Drawing;

namespace NRenoiseTools.Xrns2MidiApp
{
    public partial class Xrns2MidiForm : Form
    {
        // Track Labels
        private List<Label> m_TrackLabels = new List<Label>();

        // UI related
        private int  m_DragStartPos;
        private bool m_WhilePosDrag;

        // Loading related
        private string m_RenoiseFilename;
        private string m_DefaultCaption;

        // Playing related
        private SoundRenderer sr;
        private RenoiseMidiSong midiSong;
        private bool m_bAutoStartPending;
        private bool m_bPlaying;
        private bool m_bPendingBuffering;

        public Xrns2MidiForm()
        {
            InitializeComponent();

            // Store default caption
            m_DefaultCaption = Text;

            // Update UI
            timerUpdateUI_Tick(null, null);
        }

        private void Xrns2MidiForm_Load(object sender, EventArgs e)
        {
            SoundRenderInterface.BuildWaveforms();
        }

        private void StopPlaying()
        {
            // Terminate Sound renderer
            if (sr != null)
            {
                sr.Close();
                sr = null;
            }

            // Terminte playing states
            timerExecute.Enabled = false;
            m_bPlaying = false;
            m_bPendingBuffering = false;
            m_bAutoStartPending = false;
        }

        private void btnLoad_Click(object sender, EventArgs e)
        {
            // Show open dialog
            if (openFileDialogXnrs.ShowDialog() != System.Windows.Forms.DialogResult.OK)
                return;

            // Store file name
            m_RenoiseFilename = openFileDialogXnrs.FileName;
            Text = m_DefaultCaption + "[" + m_RenoiseFilename + "]";

            // Simulate "Reload" click
            btnReload_Click(null, null);
        }

        private void BuildTrackList()
        {
            // Clear current controls
            layoutTracks.Controls.Clear();
            layoutTracks.ColumnStyles.Clear();
            m_TrackLabels.Clear();

            // Count the number of tracks
            var MaxTracks = midiSong.InstrumentsHandles.Max(t => SoundRenderInterface.GetTrackIndex(t.First())) + 1;

            // Create tracks labels
            layoutTracks.ColumnCount = MaxTracks;
            for (int i = 0; i < MaxTracks; i++)
            {
                var lbl = new Label();
                lbl.Dock = DockStyle.Fill;
                lbl.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
                lbl.Text = "---";
                lbl.Tag = (int)0;
                m_TrackLabels.Add(lbl);
                layoutTracks.ColumnStyles.Add(new System.Windows.Forms.ColumnStyle(System.Windows.Forms.SizeType.Percent, 100F));
                layoutTracks.Controls.Add(lbl, i, 0);
            }
        }

        private void btnReload_Click(object sender, EventArgs e)
        {
            try
            {
                // Store currently playing song
                StopPlaying();

                // Load renoise file
                Song renoiseSong = new Song();
                renoiseSong.Load(m_RenoiseFilename);

                // Create instruments data structers
                midiSong = new RenoiseMidiSong(renoiseSong);
                midiSong.ConvertToInternal();

                // Sort song events by time
                midiSong.SongEvents = midiSong.SongEvents.OrderBy(t => t[0]).ToList();

                // Init track bar size
                trackTime.Maximum = midiSong.SongEvents.Last()[0/*Time*/];

                // Build Track UI
                BuildTrackList();

                // Start buffering...
                StartBuffering();
            }
            catch (Exception ex)
            {
                MessageBox.Show(this, "Error while loading song: " + ex.Message + "\r\n" + ex.StackTrace, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void StartBuffering()
        {
            timerExecute.Enabled = true;

            // Build SoundRenderer interface
            int LinesPerMin = midiSong.Tempo.Lpb * (int)midiSong.Tempo.Bpm;
            sr = new SoundRenderer();
            sr.m_SamplesPerLine = SoundRenderer.SAMPLE_RATE * 60 / LinesPerMin;
            sr.m_Instruments    = midiSong.InstrumentsHandles;
            sr.m_TrackEffects   = midiSong.TracksEffectChain;
            sr.m_songEvents     = midiSong.SongEvents;

            sr.m_QueuePos_Line    = trackTime.Value;
            sr.m_QueuePos_Samples = sr.m_QueuePos_Line * sr.m_SamplesPerLine;
            sr.m_PlayPos_Line     = trackTime.Value;
            sr.m_PlayPos_Samples  = sr.m_PlayPos_Line * sr.m_SamplesPerLine;

            sr.OnNoteCommandSent += Threaded_NoteCommand;

            // Check if we need to create new writer
            try
            {
                if (!string.IsNullOrEmpty(textWavefilename.Text))
                    sr.m_waveWriter = new WaveWriter(textWavefilename.Text);
            }
            catch
            {
                MessageBox.Show(null, "Fail to open " + textWavefilename.Text, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }

            // Start buffering...
            m_bPendingBuffering = true;
            sr.Setup(this);
        }

        private void Threaded_NoteCommand(int Index)
        {
            BeginInvoke((MethodInvoker)delegate
            {
                // Get Instrument track
                int TrackIdx = SoundRenderInterface.GetTrackIndex(midiSong.InstrumentsHandles[midiSong.SongEvents[Index][1/*Note*/]].First());

                // Convert index to note string
                var NoteStr = "";
                if (midiSong.SongEvents[Index][3/*Velocity*/] == 0)
                    NoteStr = "OFF";
                else
                    NoteStr = NRenoiseTools.NoteHelper.ConvertToString(midiSong.SongEvents[Index][2/*Note*/]);

                // Update Label
                m_TrackLabels[TrackIdx].Text = NoteStr;
                m_TrackLabels[TrackIdx].ForeColor = Color.White;
                m_TrackLabels[TrackIdx].BackColor = Color.DarkRed;
                m_TrackLabels[TrackIdx].Tag = sr.m_PlayPos_Line;
            });
        }


        private void timerExecute_Tick(object sender, EventArgs e)
        {
            // Exit if sound renderer is not ready
            if (sr == null)
                return;

            // Check if need to start playing
            if ((sr.m_BufferingState > 90) && (m_bPendingBuffering) && (m_bAutoStartPending))
            {
                PlayPauseCommand();
            }

            // Check if need to start buffering
            if ((sr.m_BufferingState < 10) && (m_bPlaying))
            {
                m_bAutoStartPending = true;
                m_bPendingBuffering = true;
                PlayPauseCommand();
            }

            // Report buffering state
            progressBuffering.Value = (int)sr.m_BufferingState;

            // Update track bar position
            if (!m_WhilePosDrag && (sr.m_PlayPos_Line != -1))
                trackTime.Value = Math.Min(sr.m_PlayPos_Line, trackTime.Maximum);

            // Return Track indications off
            foreach (var lbl in m_TrackLabels)
                if (sr.m_PlayPos_Line - (int)lbl.Tag > 1)
                {
                    lbl.ForeColor = ForeColor;
                    lbl.BackColor = BackColor;
                }
        }

        private void Xrns2MidiForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            StopPlaying();
        }

        private void PlayPauseCommand()
        {
            m_bPlaying = !m_bPlaying;

            if (m_bPlaying)
            {
                sr.Play();
                m_bAutoStartPending = false;
                m_bPendingBuffering = false;
            }
            else
                sr.Pause();

            // Update UI
            timerUpdateUI_Tick(null, null);
        }

        private void timerUpdateUI_Tick(object sender, EventArgs e)
        {
            btnExport.Enabled = (midiSong != null);
            btnPlay.Enabled   = (midiSong != null);
            btnPlay.Text = m_bPlaying ? "Pause" : "Play";
            btnReload.Enabled = !string.IsNullOrEmpty(m_RenoiseFilename);
            //btnWaveSave.Enabled = !m_bPlaying;
            //textWavefilename.Enabled = !m_bPlaying; 
        }

        private void btnPlay_Click(object sender, EventArgs e)
        {
            PlayPauseCommand();

            // Stop file writing if "stop" was pressed
            if ((!m_bPlaying) && (sr.m_waveWriter != null))
            {
                sr.m_waveWriter.Dispose();
                sr.m_waveWriter = null;
            }
        }

        private void trackTime_MouseDown(object sender, MouseEventArgs e)
        {
            m_WhilePosDrag = true;
            m_DragStartPos = trackTime.Value;
            System.Diagnostics.Debug.WriteLine("MouseDown");
        }

        private void trackTime_MouseUp(object sender, MouseEventArgs e)
        {
            if (m_DragStartPos != trackTime.Value)
            {
                StopPlaying();
                StartBuffering();
            }

            m_WhilePosDrag = false;
            System.Diagnostics.Debug.WriteLine("MouseUp");
        }

        private void btnExport_Click(object sender, EventArgs e)
        {
            // Open SaveDialog
            if (saveFileDialogData.ShowDialog() != System.Windows.Forms.DialogResult.OK)
                return ;

            // Save to file
            System.IO.File.WriteAllText(saveFileDialogData.FileName, midiSong.Code);
        }

        private void btnWaveSave_Click(object sender, EventArgs e)
        {
            if (saveFileWave.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                textWavefilename.Text = saveFileWave.FileName;
        }
    }
}
