# -*- coding: utf-8 -*-
import tkinter as tk
from tkinter import filedialog
from tkinter import messagebox
from tkinter.ttk import Progressbar, Combobox, Style
import tkinter.ttk as ttk
import subprocess
import os
import configparser
import webbrowser
import datetime
import threading
import json
import winsound

VERSION = "2.1.0"
BUILD_DATE = "2025-02-09"

# --- Constants ---
# --- GUI related constants ---
GUI_PADX = 15  # Standard horizontal padding for GUI elements
GUI_PADY = 8   # Standard vertical padding for GUI elements
GUI_LABEL_PADX = 3 # Horizontal padding for labels
GUI_LABEL_PADY = 3 # Vertical padding for labels
GUI_BUTTON_PADX = 15 # Horizontal padding for buttons
GUI_BUTTON_PADY = 8 # Vertical padding for buttons
GUI_ENTRY_WIDTH_FILE = 70 # Width for file path entry field
GUI_ENTRY_WIDTH_LUFS_TP = 10 # Width for LUFS and True Peak input entry fields
GUI_COMBOBOX_WIDTH_LUFS_PRESET = 28 # Width for LUFS preset combobox
GUI_COMBOBOX_WIDTH_TP_PRESET = 30 # Width for True Peak preset combobox
GUI_PROCESS_INFO_HEIGHT = 6 # Height of the process information text area
GUI_PROCESS_INFO_WIDTH = 60  # Width of the process information text area
DIALOG_WIDTH_OPTIONS = 450 # Width for the options dialog window
DIALOG_HEIGHT_OPTIONS = 220 # Height for the options dialog window
DIALOG_PADX_OPTIONS = 10  # Horizontal padding within options dialog
DIALOG_PADY_OPTIONS = 5   # Vertical padding within options dialog
DIALOG_WIDTH_INFO = 450  # Width for the info/about dialog window
DIALOG_HEIGHT_INFO = 420 # Height for the info/about dialog window
DIALOG_VERTICAL_OFFSET_INFO = -100 # Vertical offset for the info dialog position
STANDARD_LOG_FILE_SIZE_KB = 1024 # Default log file size limit in KB
MAX_LOG_FILE_SIZE_KB = 10240 # Maximum allowed log file size in KB (10 MB)

# --- File and path related constants ---
CONFIG_FILE_NAME = "options.ini" # Name of the configuration file
LOG_FILE_NAME = "normalization.log" # Name of the normalization log file
ANALYSIS_LOG_FILE_NAME = "analysis.log" # Name of the audio analysis log file

FFMPEG_EXECUTABLE_NAME = "ffmpeg.exe" # Name of the FFmpeg executable
TEMP_FILE_EXTENSION = ".temp" # Extension for temporary output files

AUDIO_FILE_EXTENSION_WAV = ".wav" # File extension for WAV audio files
AUDIO_FILE_EXTENSION_MP3 = ".mp3" # File extension for MP3 audio files
AUDIO_FILE_EXTENSION_FLAC = ".flac" # File extension for FLAC audio files
AUDIO_FILE_EXTENSION_AAC = ".aac" # File extension for AAC audio files
AUDIO_FILE_EXTENSION_OGG = ".ogg" # File extension for OGG audio files
AUDIO_FILE_EXTENSION_M4A = ".m4a" # File extension for M4A audio files (often AAC)
AUDIO_FILE_EXTENSIONS_ALL = "*.wav *.mp3 *.flac *.aac *.ogg *.m4a" # Wildcard for all supported audio files
ALL_FILES_WILDCARD = "*.*" # Generic wildcard for all files

# --- Output format related constants ---
OUTPUT_FORMAT_WAV = "WAV" # Output format option: WAV
OUTPUT_FORMAT_MP3 = "MP3" # Output format option: MP3
OUTPUT_FORMAT_FLAC = "FLAC" # Output format option: FLAC
OUTPUT_FORMAT_AAC = "AAC" # Output format option: AAC
OUTPUT_FORMAT_OGG = "OGG" # Output format option: OGG
OUTPUT_FORMATS_LIST = [OUTPUT_FORMAT_WAV, OUTPUT_FORMAT_MP3, OUTPUT_FORMAT_FLAC, OUTPUT_FORMAT_AAC, OUTPUT_FORMAT_OGG] # List of available output formats for the GUI

# --- Configuration file section and keys ---
CONFIG_SECTION_SETTINGS = "Settings" # Section in the config file for general settings
CONFIG_KEY_FFMPEG_PATH = "ffmpeg_path" # Key for storing FFmpeg executable path in config
CONFIG_KEY_LOG_FILE_SIZE = "log_file_size_kb" # Key for storing log file size limit in config
CONFIG_KEY_SINGLE_LOG_ENTRY = "single_log_entry_enabled" # Key for enabling/disabling single log entry mode
CONFIG_KEY_LANGUAGE = "language" # Key for storing the selected language code

# --- Language related constants ---
LANGUAGE_CODE_EN_US = "en_US" # Language code for English (US)
LANGUAGE_CODE_DE_DE = "de_DE" # Language code for German (Germany)
LANGUAGE_CODE_PL_PL = "pl_PL" # Language code for Polish (Poland)
LANGUAGE_CODES_LIST = [LANGUAGE_CODE_EN_US, LANGUAGE_CODE_DE_DE, LANGUAGE_CODE_PL_PL] # List of available language codes
DEFAULT_LANGUAGE_CODE = LANGUAGE_CODE_EN_US # Default language if none is set or language file is missing
LANG_FOLDER_NAME = "lang" # Folder name where language files are stored
LANG_FILE_EXTENSION = ".json" # File extension for language files

# --- GUI Style related constants ---
STYLE_THEME_NAME = 'clam' # Tkinter style theme to use
PROCESS_INFO_BACKGROUND_COLOR = "lightgrey" # Background color for the process information text area

# --- FFmpeg codec and option constants ---
CODEC_PCM_F32LE = "pcm_f32le" # FFmpeg codec for WAV output (32-bit float PCM)
CODEC_LIBMP3LAME = "libmp3lame" # FFmpeg codec for MP3 output (using libmp3lame encoder)
CODEC_FLAC = "flac" # FFmpeg codec for FLAC output
CODEC_AAC = "aac" # FFmpeg codec for AAC output
CODEC_LIBVORBIS = "libvorbis" # FFmpeg codec for OGG output (using libvorbis encoder)

FFMPEG_OPTION_BITRATE_MP3 = "-b:a" # FFmpeg option for setting audio bitrate for MP3
FFMPEG_BITRATE_320K = "320k" # Bitrate value for MP3 output (320kbps)
FFMPEG_OPTION_QSCALE_OGG = "-qscale:a" # FFmpeg option for setting quality scale for OGG (variable bitrate)
FFMPEG_QSCALE_10 = "10" # Quality scale value for OGG output (10 is highest quality, VBR)


# --- Language Data and Presets ---
language_data = {} # Dictionary to store language-specific text strings loaded from JSON files
current_language = DEFAULT_LANGUAGE_CODE # Currently selected language code, defaults to English

# --- Default Presets (English) ---
# --- LUFS (Loudness Units Full Scale) Presets ---
LUFS_PRESETS_EN = {
    "Default (-10 LUFS)": "-10", # Default target loudness level
    "Apple Music (-16 LUFS)": "-16", # Target loudness for Apple Music streaming service
    "Amazon Music (-16 LUFS)": "-16", # Target loudness for Amazon Music streaming service
    "Broadcast EBU R128 (-23 LUFS)": "-23", # Broadcast standard loudness level
    "Custom": "custom", # Option to set a custom LUFS value
    "Gaming (-20 LUFS)": "-20", # Target loudness suitable for gaming content
    "Podcast (-16 LUFS)": "-16", # Target loudness for general podcast content
    "Podcast (Speech, -19 LUFS)": "-19", # Target loudness optimized for speech-based podcasts
    "Spotify (-14 LUFS)": "-14", # Target loudness for Spotify streaming service
    "Tidal (-14 LUFS)": "-14", # Target loudness for Tidal streaming service
    "YouTube (-14 LUFS)": "-14", # Target loudness for YouTube streaming service
}
# --- True Peak (dB True Peak) Presets ---
TRUE_PEAK_PRESETS_EN = {
    "Default (-1 dBTP)": "-1", # Default true peak limit
    "Broadcast (-2 dBTP)": "-2", # True peak limit for broadcast applications
    "CD Mastering (Standard, -1 dBTP)": "-1", # True peak limit for CD mastering (standard practice)
    "CD Mastering (Strict, -0.3 dBTP)": "-0.3", # True peak limit for CD mastering (strict/conservative)
    "No Limit (0 dBTP)": "0", # No true peak limiting (allows peaks up to 0 dBTP)
    "Custom": "custom", # Option to set a custom True Peak value
}
LUFS_PRESETS = LUFS_PRESETS_EN # Currently active LUFS presets (initially English)
TRUE_PEAK_PRESETS = TRUE_PEAK_PRESETS_EN # Currently active True Peak presets (initially English)
LUFS_PRESET_NAMES = list(LUFS_PRESETS.keys()) # List of LUFS preset names for the combobox
TRUE_PEAK_PRESET_NAMES = list(TRUE_PEAK_PRESETS.keys()) # List of True Peak preset names for the combobox


class Config:
    """
    Configuration class to manage application settings.
    Handles loading settings from 'options.ini', saving settings, and ensuring valid configuration values.
    """
    def __init__(self):
        """
        Initializes the Config object.
        Sets default values, loads options from file, and ensures log size validity.
        """
        self.ffmpeg_path = FFMPEG_EXECUTABLE_NAME # Default FFmpeg path (assumes ffmpeg.exe is in the same directory)
        self.log_file_size_kb = STANDARD_LOG_FILE_SIZE_KB # Default log file size limit
        self.single_log_entry_enabled = True # Default: Single log entry mode enabled (overwrites log on each run)
        self.load_options() # Load options from 'options.ini' when the Config object is created

    def load_options(self):
        """
        Loads options from 'options.ini' file using configparser.
        If the file exists, it reads settings for ffmpeg_path, log_file_size_kb, single_log_entry_enabled, and language.
        If the file or sections/keys are missing, default values are kept.
        """
        config = configparser.ConfigParser() # Create a ConfigParser object to handle INI file operations
        if os.path.exists(CONFIG_FILE_NAME): # Check if the configuration file exists
            config.read(CONFIG_FILE_NAME) # Read the configuration file
            if CONFIG_SECTION_SETTINGS in config: # Check if the 'Settings' section exists in the config file
                if CONFIG_KEY_FFMPEG_PATH in config[CONFIG_SECTION_SETTINGS]: # Check if ffmpeg_path key exists
                    self.ffmpeg_path = config[CONFIG_SECTION_SETTINGS][CONFIG_KEY_FFMPEG_PATH] # Load ffmpeg_path from config
                if CONFIG_KEY_LOG_FILE_SIZE in config[CONFIG_SECTION_SETTINGS]: # Check if log_file_size_kb key exists
                    try:
                        self.log_file_size_kb = int(config[CONFIG_SECTION_SETTINGS][CONFIG_KEY_LOG_FILE_SIZE]) # Load log_file_size_kb and convert to integer
                    except ValueError:
                        self.log_file_size_kb = STANDARD_LOG_FILE_SIZE_KB # If conversion fails, use default log file size
                if CONFIG_KEY_SINGLE_LOG_ENTRY in config[CONFIG_SECTION_SETTINGS]: # Check if single_log_entry_enabled key exists
                    self.single_log_entry_enabled = config[CONFIG_SECTION_SETTINGS][CONFIG_KEY_SINGLE_LOG_ENTRY].lower() == "true" # Load single_log_entry_enabled and convert to boolean
                if CONFIG_KEY_LANGUAGE in config[CONFIG_SECTION_SETTINGS]: # Check if language key exists
                    global current_language # Access the global current_language variable
                    current_language = config[CONFIG_SECTION_SETTINGS][CONFIG_KEY_LANGUAGE] # Load language code from config
        self.ensure_log_size_valid() # After loading options, ensure the log file size is valid

    def save_options(self):
        """
        Saves current configuration options to 'options.ini' file.
        Writes ffmpeg_path, log_file_size_kb, single_log_entry_enabled, and current_language to the config file.
        """
        config = configparser.ConfigParser() # Create a ConfigParser object for saving options
        config[CONFIG_SECTION_SETTINGS] = {CONFIG_KEY_FFMPEG_PATH: self.ffmpeg_path, # Save ffmpeg_path
                                  CONFIG_KEY_LOG_FILE_SIZE: self.log_file_size_kb, # Save log_file_size_kb
                                  CONFIG_KEY_SINGLE_LOG_ENTRY: str(self.single_log_entry_enabled), # Save single_log_entry_enabled as string
                                  CONFIG_KEY_LANGUAGE: current_language # Save current_language
                                  }
        with open(CONFIG_FILE_NAME, "w") as configfile: # Open the config file in write mode
            config.write(configfile) # Write the configuration to the file

    def ensure_log_size_valid(self):
        """
        Ensures that the log_file_size_kb is a valid positive integer.
        If log_file_size_kb is not an integer or is not positive, it resets it to the STANDARD_LOG_FILE_SIZE_KB.
        """
        if not isinstance(self.log_file_size_kb, int) or self.log_file_size_kb <= 0: # Check if log_file_size_kb is not an integer or not positive
            self.log_file_size_kb = STANDARD_LOG_FILE_SIZE_KB # Reset to default log file size if invalid


config = Config() # Create a Config object to manage application settings
normalization_process = None # Global variable to store the normalization subprocess, allowing for cancellation


def load_language(language_code):
    """
    Loads language-specific text data from JSON files and updates LUFS/TP presets.
    This function reads a JSON file corresponding to the given language code,
    populates the language_data dictionary, and updates the global LUFS_PRESETS and TRUE_PEAK_PRESETS dictionaries
    with translated preset names if available for the selected language (currently DE and PL).
    If the language file is not found or JSON decoding fails, it falls back to the default language (English).
    """
    global language_data, LUFS_PRESET_NAMES, TRUE_PEAK_PRESET_NAMES, LUFS_PRESETS, TRUE_PEAK_PRESETS # Access global variables
    try:
        language_file_path = os.path.join(LANG_FOLDER_NAME, f"{language_code}{LANG_FILE_EXTENSION}") # Construct the path to the language JSON file
        with open(language_file_path, "r", encoding="utf-8") as f: # Open the language file in read mode with UTF-8 encoding
            language_data = json.load(f) # Load the JSON data from the file into the language_data dictionary

        if language_code == LANGUAGE_CODE_DE_DE: # If the selected language is German
            LUFS_PRESETS = { # Update LUFS presets with German names (translations from language_data)
                get_text("lufs_preset_default_long").format(lufs_value="-10"): "-10",
                get_text("lufs_preset_youtube"): "-14",
                get_text("lufs_preset_spotify"): "-14",
                get_text("lufs_preset_applemusic"): "-16",
                get_text("lufs_preset_tidal"): "-14",
                get_text("lufs_preset_amazonmusic"): "-16",
                get_text("lufs_preset_podcast"): "-16",
                get_text("lufs_preset_podcast_speech"): "-19",
                get_text("lufs_preset_gaming"): "-20",
                get_text("lufs_preset_broadcast_long").format(lufs_value="-23"): "-23",
                get_text("lufs_preset_custom"): "custom"
            }
            TRUE_PEAK_PRESETS = { # Update True Peak presets with German names (translations from language_data)
                get_text("true_peak_preset_default_long").format(tp_value="-1"): "-1",
                get_text("true_peak_preset_broadcast_long").format(tp_value="-2"): "-2",
                get_text("true_peak_preset_cdmastering"): "-1",
                get_text("true_peak_preset_cdmastering_strict"): "-0.3",
                get_text("true_peak_preset_nolimit"): "0",
                get_text("true_peak_preset_custom"): "custom"
            }
        elif language_code == LANGUAGE_CODE_PL_PL: # If the selected language is Polish
            LUFS_PRESETS = { # Update LUFS presets with Polish names (translations from language_data)
                get_text("lufs_preset_default_long").format(lufs_value="-10"): "-10",
                get_text("lufs_preset_youtube"): "-14",
                get_text("lufs_preset_spotify"): "-14",
                get_text("lufs_preset_applemusic"): "-16",
                get_text("lufs_preset_tidal"): "-14",
                get_text("lufs_preset_amazonmusic"): "-16",
                get_text("lufs_preset_podcast"): "-16",
                get_text("lufs_preset_podcast_speech"): "-19",
                get_text("lufs_preset_gaming"): "-20",
                get_text("lufs_preset_broadcast_long").format(lufs_value="-23"): "-23",
                get_text("lufs_preset_custom"): "custom"
            }
            TRUE_PEAK_PRESETS = { # Update True Peak presets with Polish names (translations from language_data)
                get_text("true_peak_preset_default_long").format(tp_value="-1"): "-1",
                get_text("true_peak_preset_broadcast_long").format(tp_value="-2"): "-2",
                get_text("true_peak_preset_cdmastering"): "-1",
                get_text("true_peak_preset_cdmastering_strict"): "-0.3",
                get_text("true_peak_preset_nolimit"): "0",
                get_text("true_peak_preset_custom"): "custom"
            }
        else: # If the selected language is English or any other language not explicitly handled, use default English presets
            LUFS_PRESETS = LUFS_PRESETS_EN # Revert to English LUFS presets
            TRUE_PEAK_PRESETS = TRUE_PEAK_PRESETS_EN # Revert to English True Peak presets

        # --- Sort Preset Names with Custom Order ---
        lufs_presets_sorted_keys = sorted(LUFS_PRESETS.keys(), key=lambda x: (x != "Default (-10 LUFS)", x != get_text("lufs_preset_default_long").format(lufs_value="-10"), x == "Custom", x == get_text("lufs_preset_custom"), x)) # Custom sort: Default on top, Custom at bottom, then alphabetical
        TRUE_PEAK_PRESETS_SORTED_KEYS = sorted(TRUE_PEAK_PRESETS.keys(), key=lambda x: (x != "Default (-1 dBTP)", x != get_text("true_peak_preset_default_long").format(tp_value="-1"), x == "Custom",  x == get_text("true_peak_preset_custom"), x)) # Custom sort: Default on top, Custom at bottom, then alphabetical

        LUFS_PRESET_NAMES = lufs_presets_sorted_keys # Update the global list of LUFS preset names with sorted names
        TRUE_PEAK_PRESET_NAMES = TRUE_PEAK_PRESETS_SORTED_KEYS # Update the global list of True Peak preset names with sorted names

    except FileNotFoundError: # Handle case where the language file is not found
        load_language(DEFAULT_LANGUAGE_CODE) # Fallback to loading the default language (English)
    except json.JSONDecodeError: # Handle case where the language file is corrupted or not valid JSON
        load_language(DEFAULT_LANGUAGE_CODE) # Fallback to loading the default language (English)


def get_text(key):
    """
    Retrieves language-specific text from language_data dictionary.
    This function takes a key as input and returns the corresponding text string from the language_data dictionary.
    If the key is not found in the dictionary, it returns a placeholder string "[key]" indicating a missing translation.
    """
    global language_data # Access the global language_data dictionary
    return language_data.get(key, f"[{key}]") # Return the text for the given key, or a placeholder if the key is not found


def apply_language(dialog_type):
    """
    Applies the currently loaded language to the main window and dialogs.
    This function updates the text elements of the main window and any open dialogs (options, info)
    based on the currently loaded language_data. It iterates through the GUI elements and updates their 'text' or 'label' configurations
    using the get_text function to retrieve translated strings.
    The dialog_type argument specifies which dialog to update ("options", "info", or "main" for the main window).
    """
    window.title(f"{get_text('app_title')} v{VERSION}") # Set the main window title using translated app title and version

    menubar = tk.Menu(window) # Create a menu bar for the main window
    filemenu = tk.Menu(menubar, tearoff=0) # Create the 'File' menu
    filemenu.add_command(label=get_text("menu_file_options"), command=show_options_dialog) # Add 'Options' command to the File menu, using translated label
    filemenu.add_separator() # Add a separator line in the File menu
    filemenu.add_command(label=get_text("menu_file_exit"), command=exit_program) # Add 'Exit' command to the File menu, using translated label
    menubar.add_cascade(label=get_text("menu_file"), menu=filemenu) # Add the File menu to the menu bar, using translated label

    infomenu = tk.Menu(menubar, tearoff=0) # Create the 'Info' menu
    infomenu.add_command(label=get_text("menu_info_about"), command=show_info_box) # Add 'About' command to the Info menu, using translated label
    menubar.add_cascade(label=get_text("menu_info"), menu=infomenu) # Add the Info menu to the menu bar, using translated label
    window.config(menu=menubar) # Configure the main window to use the created menu bar

    if dialog_type == "options": # If updating the options dialog
        options_dialog = window.children.get("!toplevel") # Get a reference to the options dialog window (assuming it's a Toplevel window)
        if options_dialog: # Check if the options dialog exists
            try: # Use try-except to handle potential KeyErrors if GUI elements are not found (e.g., during development)
                options_dialog.title(get_text("options_dialog_title")) # Set the options dialog title using translated text
                options_dialog.children["language_frame"].config(text=get_text("options_language_group")) # Update language frame group label
                options_dialog.children["ffmpeg_frame"].config(text=get_text("options_ffmpeg_path_group")) # Update ffmpeg path frame group label
                options_dialog.children["log_frame"].config(text=get_text("options_log_settings_group")) # Update log settings frame group label
                options_dialog.children["language_frame"].children["language_label"].config(text=get_text("options_language_label")) # Update language label in options
                options_dialog.children["ffmpeg_frame"].children["ffmpeg_path_label"].config(text=get_text("options_ffmpeg_path_label")) # Update ffmpeg path label in options
                options_dialog.children["log_frame"].children["single_log_check"].config(text=get_text("options_log_single_entry_check")) # Update single log entry checkbox label
                options_dialog.children["log_frame"].children["log_size_label"].config(text=get_text("options_log_size_label")) # Update log size label in options
                options_dialog.children["ffmpeg_frame"].children["browse_ffmpeg_button"].config(text=get_text("options_browse_button")) # Update browse button text in options
                options_dialog.children["save_options_button"].config(text=get_text("options_save_button")) # Update save button text in options
            except KeyError:
                pass # Ignore KeyError if a GUI element is not found (for robustness during development)

    elif dialog_type == "info": # If updating the info/about dialog
        info_dialog = window.children.get("!toplevel2") # Get a reference to the info dialog window (assuming it's the second Toplevel window)
        if info_dialog: # Check if the info dialog exists
            try: # Use try-except to handle potential KeyErrors
                info_dialog.title(get_text("menu_info_about")) # Set the info dialog title using translated text
                info_dialog.children["info_label_frame"].config(text=get_text("app_title_long")) # Update info frame group label
                info_dialog.children["info_label_frame"].children["info_text_label"].config(text=get_text("about_text").format(version=VERSION, build_date=BUILD_DATE)) # Update main about text label
                info_dialog.children["info_label_frame"].children["website_label_1"].config(text=get_text("about_website_1")) # Update website label 1 in info dialog
                info_dialog.children["info_label_frame"].children["website_label_2"].config(text=get_text("about_website_2")) # Update website label 2 in info dialog
                info_dialog.children["info_label_frame"].children["opensource_label"].config(text=get_text("about_opensource").format(year=datetime.datetime.now().year)) # Update opensource label in info dialog
                info_dialog.children["ok_info_button"].config(text=get_text("about_ok_button")) # Update OK button text in info dialog
            except KeyError:
                pass # Ignore KeyError

    elif dialog_type == "main": # If updating the main window
        file_frame_group.config(text=get_text("file_selection_group")) # Update file selection frame group label
        loudness_settings_frame_group.config(text=get_text("loudness_settings_group")) # Update loudness settings frame group label
        output_format_frame_group.config(text=get_text("output_format_group")) # Update output format frame group label
        process_information_frame_group.config(text=get_text("process_information_group")) # Update process information frame group label

        file_label.config(text=get_text("select_audio_file_label")) # Update file selection label
        browse_button.config(text=get_text("browse_file_button")) # Update browse button text
        lufs_preset_label.config(text=get_text("lufs_preset_label")) # Update LUFS preset label
        true_peak_preset_label.config(text=get_text("true_peak_preset_label")) # Update True Peak preset label
        lufs_label.config(text=get_text("target_lufs_label_custom")) # Update LUFS custom label
        true_peak_label.config(text=get_text("true_peak_label")) # Update True Peak label
        output_format_label.config(text=get_text("output_format_file_format_label")) # Update output format label
        analyze_button.config(text=get_text("analyze_audio_button")) # Update analyze button text
        start_button.config(text=get_text("start_normalization_button")) # Update start normalization button text
        cancel_button.config(text=get_text("cancel_normalization_button")) # Update cancel normalization button text

        lufs_preset_combobox.config(values=LUFS_PRESET_NAMES) # Update LUFS preset combobox values with translated preset names
        true_peak_preset_combobox.config(values=TRUE_PEAK_PRESET_NAMES) # Update True Peak preset combobox values with translated preset names
        if lufs_preset_var.get() not in LUFS_PRESET_NAMES: # Ensure the selected LUFS preset is still valid after language change
            lufs_preset_var.set(LUFS_PRESET_NAMES[0]) # If not valid, reset to the first preset in the list (usually 'Default')
        if true_peak_preset_var.get() not in TRUE_PEAK_PRESET_NAMES: # Ensure the selected True Peak preset is still valid after language change
            true_peak_preset_var.set(TRUE_PEAK_PRESET_NAMES[0]) # If not valid, reset to the first preset in the list (usually 'Default')

        update_lufs_entry_state() # Update the state of the custom LUFS entry (enable/disable based on preset selection)
        update_true_peak_entry_state() # Update the state of the custom True Peak entry (enable/disable based on preset selection)


def show_options_dialog():
    """
    Displays the options dialog window.
    Creates a Toplevel window for application options, including language selection, FFmpeg path setting,
    and log file settings.  It makes the dialog modal (grabs focus) and positions it relative to the main window.
    """
    options_window = tk.Toplevel(window) # Create a Toplevel window for options dialog, parented to the main window
    options_window.title(get_text("options_dialog_title")) # Set the title of the options dialog using translated text
    options_window.transient(window) # Set the options dialog as transient to the main window (stays on top)
    options_window.grab_set() # Make the options dialog modal, grabbing all input

    window_width = window.winfo_width() # Get width of the main window (currently unused)
    window_height = window.winfo_height() # Get height of the main window (currently unused)
    window_x = window.winfo_rootx() # Get x position of the main window on screen (currently unused)
    window_y = window.winfo_rooty() # Get y position of the main window on screen (currently unused)
    options_window.geometry(f"+{x_position}+{y_position}") # Position the options dialog at the same x,y as the main window (using global x_position, y_position - might need adjustment)

    language_frame = tk.LabelFrame(options_window, text=get_text("options_language_group"), padx=DIALOG_PADX_OPTIONS,
                                     pady=DIALOG_PADY_OPTIONS, name="language_frame") # Create a LabelFrame for language options
    language_frame.grid(row=0, column=0, columnspan=3, padx=GUI_PADX, pady=GUI_PADY,
                         sticky="ew") # Place language frame in the grid layout, spanning 3 columns, sticky east-west

    ffmpeg_frame = tk.LabelFrame(options_window, text=get_text("options_ffmpeg_path_group"), padx=DIALOG_PADX_OPTIONS,
                                 pady=DIALOG_PADY_OPTIONS, name="ffmpeg_frame") # Create a LabelFrame for FFmpeg path options
    ffmpeg_frame.grid(row=1, column=0, columnspan=3, padx=GUI_PADX, pady=GUI_PADY,
                        sticky="ew") # Place ffmpeg frame in the grid layout

    log_frame = tk.LabelFrame(options_window, text=get_text("options_log_settings_group"), padx=DIALOG_PADX_OPTIONS,
                              pady=DIALOG_PADY_OPTIONS, name="log_frame") # Create a LabelFrame for log settings options
    log_frame.grid(row=2, column=0, columnspan=3, padx=GUI_PADX, pady=GUI_PADY,
                    sticky="ew") # Place log frame in the grid layout

    language_label = tk.Label(language_frame, text=get_text("options_language_label"), name="language_label") # Create a label for language selection
    language_label.grid(row=0, column=0, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="w") # Place language label in the language frame

    language_var = tk.StringVar(value=current_language) # Create a StringVar to hold the selected language, initialized with current language
    language_combobox = Combobox(language_frame, textvariable=language_var,
                                    values=LANGUAGE_CODES_LIST, state="readonly", width=20, name="language_combobox") # Create a Combobox for language selection, readonly, with language codes as values
    language_combobox.grid(row=0, column=1, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="ew") # Place language combobox in the language frame
    language_combobox.bind("<<ComboboxSelected>>", lambda event, var=language_var: update_language_selection(event, var)) # Bind combobox selection event to update_language_selection function

    ffmpeg_path_label = tk.Label(ffmpeg_frame, text=get_text("options_ffmpeg_path_label"), name="ffmpeg_path_label") # Create a label for FFmpeg path input
    ffmpeg_path_label.grid(row=0, column=0, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="w") # Place ffmpeg path label in the ffmpeg frame
    ffmpeg_path_input = tk.Entry(ffmpeg_frame, width=50, name="ffmpeg_path_input") # Create an Entry field for FFmpeg path input
    ffmpeg_path_input.grid(row=0, column=1, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="ew") # Place ffmpeg path input field in the ffmpeg frame

    if config.ffmpeg_path == FFMPEG_EXECUTABLE_NAME: # Check if the configured FFmpeg path is the default (ffmpeg.exe in program directory)
        program_path = os.path.dirname(os.path.abspath(__file__)) # Get the directory of the current script
        ffmpeg_path_input.insert(0, program_path) # If default path, pre-fill the input field with the program directory
    else:
        ffmpeg_path_input.insert(0, config.ffmpeg_path) # Otherwise, pre-fill with the configured FFmpeg path

    single_log_check_var = tk.BooleanVar(
        value=config.single_log_entry_enabled) # Create a BooleanVar to hold the state of the single log entry checkbox, initialized with the config value
    single_log_check = tk.Checkbutton(log_frame, text=get_text("options_log_single_entry_check"),
                                      variable=single_log_check_var,
                                      command=lambda: update_log_size_state(single_log_check_var,
                                                                               log_size_input), name="single_log_check") # Create a Checkbutton for enabling/disabling single log entry mode
    single_log_check.grid(row=1, column=0, columnspan=2, padx=DIALOG_PADX_OPTIONS,
                            pady=(DIALOG_PADY_OPTIONS, DIALOG_PADY_OPTIONS),
                            sticky="w") # Place single log entry checkbox in the log frame

    log_size_label = tk.Label(log_frame, text=get_text("options_log_size_label"), name="log_size_label") # Create a label for log size input
    log_size_label.grid(row=0, column=0, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="w") # Place log size label in the log frame
    log_size_input = tk.Entry(log_frame, width=10, name="log_size_input") # Create an Entry field for log size input
    log_size_input.grid(row=0, column=1, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS,
                            sticky="w") # Place log size input field in the log frame
    log_size_input.insert(0, str(config.log_file_size_kb)) # Pre-fill log size input with the configured log file size

    def update_log_size_state(check_variable, size_input_field):
        """
        Enables or disables the log size input field based on the state of the single log entry checkbox.
        If single log entry mode is enabled, the log size input is disabled as it's not relevant.
        """
        if check_variable.get(): # If single log entry checkbox is checked (True)
            size_input_field.config(state=tk.DISABLED) # Disable the log size input field
        else:
            size_input_field.config(state=tk.NORMAL) # Otherwise, enable the log size input field

    update_log_size_state(single_log_check_var, log_size_input) # Initialize the log size input state based on the initial checkbox value

    def browse_ffmpeg_path():
        """
        Opens a directory dialog to browse for the FFmpeg executable path.
        Updates the FFmpeg path input field with the selected directory.
        """
        path = filedialog.askdirectory(title=get_text("options_ffmpeg_path_dialog_title")) # Open a directory dialog to select FFmpeg path
        if path: # If a path is selected (not canceled)
            ffmpeg_path_input.delete(0, tk.END) # Clear the current content of the FFmpeg path input field
            ffmpeg_path_input.insert(0, path) # Insert the selected path into the FFmpeg path input field

    browse_button = ttk.Button(ffmpeg_frame, text=get_text("options_browse_button"), command=browse_ffmpeg_path, name="browse_ffmpeg_button") # Create a button to browse for FFmpeg path
    browse_button.grid(row=0, column=2, padx=DIALOG_PADX_OPTIONS, pady=DIALOG_PADY_OPTIONS) # Place browse button in the ffmpeg frame

    def save_and_close_options():
        """
        Saves the options from the dialog to the configuration and closes the options dialog.
        Validates FFmpeg path, log size, and then saves the settings using the Config class.
        Updates language settings and applies language changes to all open windows.
        """
        global current_language # Access the global current_language variable
        ffmpeg_path_input_value = ffmpeg_path_input.get() # Get the value from the FFmpeg path input field
        ffmpeg_executable_path = os.path.join(ffmpeg_path_input_value, FFMPEG_EXECUTABLE_NAME) # Construct the full path to the FFmpeg executable

        if not os.path.exists(ffmpeg_executable_path): # Check if the FFmpeg executable exists at the specified path
            messagebox.showerror(get_text("options_error_invalid_ffmpeg_path_title"),
                                 get_text("options_error_invalid_ffmpeg_path_message"),
                                 parent=options_window) # Show an error message if FFmpeg executable is not found
            return # Exit the save function if FFmpeg path is invalid

        try: # Try to run FFmpeg to check if it's a valid executable
            subprocess.run([ffmpeg_executable_path, "-version"], capture_output=True,
                             check=True) # Run ffmpeg -version command, capture output, and check for errors
        except (FileNotFoundError, subprocess.CalledProcessError): # Handle cases where FFmpeg is not found or execution fails
            messagebox.showerror(get_text("options_error_ffmpeg_executable_title"),
                                 get_text("options_error_ffmpeg_executable_message"),
                                 parent=options_window) # Show an error message if FFmpeg execution fails
            return # Exit the save function if FFmpeg execution check fails

        config.ffmpeg_path = ffmpeg_path_input_value # Update the FFmpeg path in the Config object

        if not single_log_check_var.get(): # If single log entry mode is disabled (checkbox is unchecked)
            try: # Try to validate and save the log file size
                log_file_size_kb_input = log_size_input.get() # Get the log file size input value
                log_file_size_kb_value = int(log_file_size_kb_input) # Convert the input value to an integer
                if log_file_size_kb_value <= 0: # Check if the log file size is not positive
                    messagebox.showerror(get_text("options_error_invalid_log_size_title"),
                                         get_text("options_error_invalid_log_size_positive_message"),
                                         parent=options_window) # Show an error if log size is not positive
                    return # Exit the save function if log size is invalid
                if log_file_size_kb_value > MAX_LOG_FILE_SIZE_KB: # Check if the log file size exceeds the maximum limit
                    messagebox.showerror(get_text("options_error_invalid_log_size_title"),
                                         get_text("options_error_invalid_log_size_maximum_message").format(max_size_kb=MAX_LOG_FILE_SIZE_KB),
                                         parent=options_window) # Show an error if log size exceeds the maximum limit
                    return # Exit if log size is too large
                config.log_file_size_kb = log_file_size_kb_value # Update the log file size in the Config object
            except ValueError: # Handle case where the log file size input is not a valid integer
                messagebox.showerror(get_text("options_error_invalid_log_size_title"),
                                     get_text("options_error_invalid_log_size_integer_message"),
                                     parent=options_window) # Show an error if log size is not an integer
                return # Exit if log size is not a number

        config.single_log_entry_enabled = single_log_check_var.get() # Update single log entry enabled setting in Config object
        current_language = language_var.get() # Update the global current_language variable with the selected language from combobox

        config.save_options() # Save all options to the config file
        apply_language("options") # Re-apply language to the options dialog itself
        apply_language("main") # Re-apply language to the main window
        if window.children.get("!toplevel2"): # If the info dialog is open
            apply_language("info") # Re-apply language to the info dialog
        options_window.destroy() # Close the options dialog

    save_button = ttk.Button(options_window, text=get_text("options_save_button"),
                                 command=save_and_close_options, name="save_options_button") # Create a button to save options and close the dialog
    save_button.grid(row=3, column=0, columnspan=3, pady=GUI_PADY) # Place save button in the grid layout

    options_window.columnconfigure(1, weight=1) # Configure the second column of the options dialog to expand horizontally
    options_window.wait_window(options_window) # Make the options dialog modal and wait for it to be closed before continuing


def show_info_box():
    """
    Displays the information/about dialog box.
    Creates a Toplevel window to show application information, version, author, website links, and license details.
    Positions the dialog in the center of the screen with a vertical offset.
    """
    info_window = tk.Toplevel(window) # Create a Toplevel window for the info/about dialog
    info_window.title(get_text("menu_info_about")) # Set the title of the info dialog using translated text
    info_window.transient(window) # Set the info dialog as transient to the main window
    info_window.grab_set() # Make the info dialog modal

    screen_width = window.winfo_screenwidth() # Get screen width
    screen_height = window.winfo_screenheight() # Get screen height
    dialog_width = DIALOG_WIDTH_INFO # Get dialog width from constants
    dialog_height = DIALOG_HEIGHT_INFO # Get dialog height from constants
    x_position = (screen_width - dialog_width) // 2 # Calculate horizontal center position
    y_position = (screen_height - dialog_height) // 2 + DIALOG_VERTICAL_OFFSET_INFO # Calculate vertical center position with offset
    info_window.geometry(f"+{x_position}+{y_position}") # Position the info dialog in the center of the screen

    info_frame = tk.LabelFrame(info_window, text=get_text("app_title_long"), padx=GUI_PADX,
                                pady=GUI_PADY, name="info_label_frame") # Create a LabelFrame for the info content
    info_frame.pack(padx=GUI_PADX, pady=GUI_PADY, fill=tk.BOTH,
                     expand=True) # Pack the info frame to fill and expand in the info dialog

    bold_font = ("Helvetica", 9, "bold") # Define a bold font
    large_bold_font = ("Helvetica", 11, "bold") # Define a larger bold font
    normal_font = ("Helvetica", 9) # Define a normal font (currently unused)

    tk.Label(info_frame, text=get_text("app_title_long"), font=large_bold_font).pack(pady=(GUI_PADY, 2), anchor="center") # Display app title in large bold font

    tk.Label(info_frame, text=get_text("about_version").format(version=VERSION), font=bold_font).pack(anchor="center") # Display version in bold font
    tk.Label(info_frame, text=get_text("about_build_date").format(build_date=BUILD_DATE)).pack(pady=(0, GUI_PADY), anchor="center") # Display build date

    separator_line_info_1 = tk.Frame(info_frame, bg=separator_color, height=1) # Create a separator line
    separator_line_info_1.pack(fill="x", padx=GUI_PADX, pady=GUI_PADY) # Pack separator line, filling horizontally

    description_text = get_text("about_description") # Get the description text from language data
    for line in description_text.splitlines(): # Iterate through each line of the description text
        if line.strip(): # If the line is not empty after stripping whitespace
            tk.Label(info_frame, text=line, justify=tk.LEFT, anchor="w").pack(padx=GUI_PADX, pady=2, fill="x") # Display each description line as a label, left-justified

    separator_line_info_2 = tk.Frame(info_frame, bg=separator_color, height=1) # Create another separator line
    separator_line_info_2.pack(fill="x", padx=GUI_PADX, pady=GUI_PADY) # Pack separator line

    tk.Label(info_frame, text=get_text("about_author").format(author="melcom (Andreas Thomas Urban)"), anchor="w", font=bold_font).pack(padx=GUI_PADX, pady=(GUI_PADY, 2), fill="x") # Display author name in bold font
    tk.Label(info_frame, text=get_text("about_email").format(email="melcom [@] vodafonemail.de"), anchor="w").pack(padx=GUI_PADX, pady=(0, GUI_PADY), fill="x") # Display email

    tk.Label(info_frame, text=get_text("about_website_header"), anchor="w", font=bold_font).pack(padx=GUI_PADX, pady=(GUI_PADY, 2), fill="x") # Display website header in bold font

    website_label_1 = tk.Label(info_frame, text=get_text("about_website_1"), fg="blue", cursor="hand2", anchor="w") # Create website label 1, blue color, hand cursor
    website_label_1.pack(anchor="w", padx=GUI_PADX, pady=2, fill="x") # Pack website label 1
    website_label_1.bind("<Button-1>", lambda e: webbrowser.open("https://www.melcom-music.de")) # Bind click event to open website 1 in browser

    website_label_2 = tk.Label(info_frame, text=get_text("about_website_2"), fg="blue", cursor="hand2", anchor="w") # Create website label 2, blue color, hand cursor
    website_label_2.pack(anchor="w", padx=GUI_PADX, pady=2, fill="x") # Pack website label 2
    website_label_2.bind("<Button-1>", lambda e: webbrowser.open("https://scenes.at/melcom")) # Bind click event to open website 2

    tk.Label(info_frame, text=get_text("about_youtube_header"), anchor="w", font=bold_font).pack(padx=GUI_PADX, pady=(GUI_PADY, 2), fill="x") # Display YouTube header in bold font
    website_label_3 = tk.Label(info_frame, text=get_text("about_youtube_link"), fg="blue", cursor="hand2", anchor="w") # Create YouTube link label, blue color, hand cursor
    website_label_3.pack(anchor="w", padx=GUI_PADX, pady=2, fill="x") # Pack YouTube link label
    website_label_3.bind("<Button-1>", lambda e: webbrowser.open("https://youtube.com/@melcom")) # Bind click event to open YouTube channel

    website_label_4 = tk.Label(info_frame, text=get_text("about_bluesky_link"), fg="blue", cursor="hand2", anchor="w") # Create Bluesky link label, blue color, hand cursor
    website_label_4.pack(anchor="w", padx=GUI_PADX, pady=2, fill="x") # Pack Bluesky link label
    website_label_4.bind("<Button-1>", lambda e: webbrowser.open("https://melcom-music.bsky.social/")) # Bind click event to open Bluesky profile


    separator_line_info_3 = tk.Frame(info_frame, bg=separator_color, height=1) # Create another separator line
    separator_line_info_3.pack(fill="x", padx=GUI_PADX, pady=GUI_PADY) # Pack separator line

    opensource_label = tk.Label(info_frame,
                                 text=get_text("about_opensource").format(year=datetime.datetime.now().year), anchor="center") # Display opensource text
    opensource_label.pack(pady=(GUI_PADY, GUI_PADY), anchor="center") # Pack opensource label
    license_label = tk.Label(info_frame, text=get_text("about_license"), anchor="center") # Display license text
    license_label.pack(pady=(0, GUI_PADY), anchor="center") # Pack license label
    copyright_label = tk.Label(info_frame, text=get_text("about_copyright").format(year=datetime.datetime.now().year), anchor="center") # Display copyright text
    copyright_label.pack(anchor="center") # Pack copyright label


    ok_button = ttk.Button(info_frame, text=get_text("about_ok_button"), command=info_window.destroy, name="ok_info_button") # Create OK button to close info dialog
    ok_button.pack(pady=GUI_PADY, anchor="center") # Pack OK button

    info_window.wait_window(info_window) # Make the info dialog modal and wait for it to be closed
    apply_language("info") # Re-apply language to the info dialog (in case language was changed in options dialog before opening info)


def browse_file():
    """
    Opens a file dialog to select an audio file.
    Allows the user to select an audio file from their system.
    Filters the file types to common audio formats (WAV, MP3, FLAC, AAC, OGG, M4A) and "All Files".
    Updates the file input entry field with the selected file path.
    """
    file_path = filedialog.askopenfilename( # Open a file dialog for opening a file
        defaultextension=AUDIO_FILE_EXTENSION_WAV, # Default file extension if none is specified by the user
        filetypes=[ # Define file type filters for the dialog
            (get_text("file_dialog_audio_files"), AUDIO_FILE_EXTENSIONS_ALL), # Filter for all supported audio files
            (get_text("file_dialog_wav_files"), AUDIO_FILE_EXTENSION_WAV), # Filter for WAV files
            (get_text("file_dialog_mp3_files"), AUDIO_FILE_EXTENSION_MP3), # Filter for MP3 files
            (get_text("file_dialog_flac_files"), AUDIO_FILE_EXTENSION_FLAC), # Filter for FLAC files
            (get_text("file_dialog_aac_files"), f"{AUDIO_FILE_EXTENSION_AAC} {AUDIO_FILE_EXTENSION_M4A}"), # Filter for AAC and M4A files
            (get_text("file_dialog_ogg_files"), AUDIO_FILE_EXTENSION_OGG), # Filter for OGG files
            (get_text("file_dialog_all_files"), ALL_FILES_WILDCARD) # Filter for all files
        ],
        title=get_text("file_dialog_title"), # Set the title of the file dialog using translated text
        parent=window # Set the main window as the parent of the file dialog
    )
    if file_path: # If a file path is selected (not canceled)
        file_input.delete(0, tk.END) # Clear the current content of the file input entry field
        file_input.insert(0, file_path) # Insert the selected file path into the file input entry field


def analyze_audio():
    """
    Initiates audio analysis using FFmpeg in a separate thread.
    Retrieves the input file path from the GUI, validates it, and starts an audio analysis process in a background thread.
    Disables GUI elements during analysis, displays a progress bar and process information, and logs the analysis start.
    Calls audio_analysis_thread_function to execute the FFmpeg analysis command.
    """
    file = file_input.get() # Get the file path from the file input entry field

    if not file: # Check if a file path is entered
        messagebox.showerror(get_text("analysis_no_file_error_title"),
                             get_text("analysis_no_file_error_message"),
                             parent=window) # Show an error message if no file is selected
        return # Exit the analyze_audio function if no file is selected

    analyze_button.config(state=tk.DISABLED) # Disable the analyze button to prevent multiple clicks
    start_button.config(state=tk.DISABLED) # Disable the start button during analysis
    cancel_button.config(state=tk.DISABLED) # Disable the cancel button initially (not relevant for analysis, only for normalization)
    progressbar.grid(row=7, column=0, columnspan=3, sticky="ew", padx=GUI_PADX, pady=GUI_PADY) # Show the progress bar
    progressbar.start() # Start the progress bar animation

    process_info_field.config(state=tk.NORMAL) # Enable the process information text area for writing
    process_info_field.delete("1.0", tk.END) # Clear the process information text area
    process_info_field.insert(tk.END,
                             get_text("analysis_start_message").format(filename=os.path.basename(file))) # Display analysis start message in the process info area
    process_info_field.config(state=tk.DISABLED) # Disable the process information text area to prevent user editing

    log_entry_start = f"======================== {datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')} ========================\n" # Create a log entry start separator with timestamp
    log_entry_start += get_text("analysis_start_message").format(filename=file) + "\n" # Add analysis start message to the log entry
    log_entry_start += get_text("analysis_log_ffmpeg_command_generating") + "\n" # Add message indicating FFmpeg command generation to log
    append_analysis_log(log_entry_start) # Append the starting log entry to the analysis log file

    analysis_thread = threading.Thread(target=audio_analysis_thread_function,
                                      args=(file,)) # Create a new thread to run the audio_analysis_thread_function
    analysis_thread.start() # Start the analysis thread


def audio_analysis_thread_function(file):
    """
    Executes FFmpeg audio analysis command in a thread.
    This function runs in a separate thread to perform audio analysis using FFmpeg's loudnorm filter in print_format=json mode.
    It constructs the FFmpeg command, executes it using subprocess, captures the output, and parses the JSON output to extract loudness metrics.
    Calls _update_gui_on_analysis_completion to update the GUI with the analysis results or error messages.
    Handles potential errors like subprocess.CalledProcessError, FileNotFoundError, JSONDecodeError, and general Exceptions.
    """
    global normalization_process, config # Access global variables config and normalization_process (though normalization_process is not used here, might be for consistency)

    ffmpeg_analysis_command = [ # Construct the FFmpeg command for audio analysis
        os.path.join(config.ffmpeg_path, FFMPEG_EXECUTABLE_NAME), # Path to FFmpeg executable from config
        "-i", file, # Input audio file path
        "-af", "loudnorm=print_format=json", # Apply loudnorm filter with JSON output format
        "-f", "null", "-" # Output format set to null (no output file), '-' discards output
    ]
    log_entry_command = get_text("analysis_log_ffmpeg_command") + "\n" + " ".join(ffmpeg_analysis_command) + "\n\n" # Create a log entry with the FFmpeg command
    append_analysis_log(log_entry_command) # Append the FFmpeg command log entry to the analysis log

    try: # Try to execute the FFmpeg analysis command and process the output
        analysis_result = subprocess.run(ffmpeg_analysis_command, check=True, capture_output=True, text=True,
                                            creationflags=subprocess.CREATE_NO_WINDOW) # Run FFmpeg command, capture output, check for errors, hide console window
        log_output = analysis_result.stderr # Get the standard error output from FFmpeg (loudnorm outputs to stderr)
        append_analysis_log(get_text("analysis_log_ffmpeg_output") + "\n" + log_output + "\n") # Append FFmpeg output to the analysis log

        try: # Try to parse the JSON output from FFmpeg
            start_index = log_output.find("{") # Find the starting brace of the JSON output
            end_index = log_output.rfind("}") + 1 # Find the ending brace of the JSON output
            json_string = log_output[start_index:end_index] # Extract the JSON string from the FFmpeg output
            json_output = json.loads(json_string) # Parse the JSON string into a Python dictionary

            input_i = json_output.get("input_i") # Get the input integrated loudness (LUFS-I) from the JSON output
            input_tp = json_output.get("input_tp") # Get the input true peak level (dBTP) from the JSON output
            lra = json_output.get("input_lra") # Get the loudness range (LRA) from the JSON output
            analysis_results = { # Create a dictionary to store the analysis results
                "input_i": input_i,
                "input_tp": json_output.get("input_tp"),
                "lra": lra
            }
            window.after(0, _update_gui_on_analysis_completion, "Success", file, analysis_results) # Update GUI on the main thread with analysis success and results

        except json.JSONDecodeError: # Handle case where JSON parsing fails (invalid JSON output from FFmpeg)
            error_message_json = get_text("analysis_log_json_error") # Get the JSON error message
            append_analysis_log("ERROR: " + error_message_json + "\n") # Log the JSON error
            window.after(0, _update_gui_on_analysis_completion, "Error", file, error_message_json) # Update GUI with analysis error message

    except subprocess.CalledProcessError as e: # Handle case where FFmpeg command returns a non-zero exit code (error during FFmpeg execution)
        error_message_ffmpeg = get_text("analysis_ffmpeg_error_message").format(return_code=e.returncode, stderr=e.stderr) # Get the FFmpeg error message with return code and stderr
        append_log("ERROR: " + error_message_ffmpeg + "\n") # Log the FFmpeg error to the normalization log (using append_log, might need to use append_analysis_log for analysis errors)
        window.after(0, update_gui_on_completion, "Error", file, error_message_ffmpeg) # Update GUI with FFmpeg error message (using update_gui_on_completion, might need to use _update_gui_on_analysis_completion for analysis)
    except FileNotFoundError: # Handle case where FFmpeg executable is not found
        error_message_ffmpeg_path = get_text("analysis_ffmpeg_not_found_error_message") # Get the FFmpeg not found error message
        append_log("ERROR: " + error_message_ffmpeg_path + "\n") # Log FFmpeg not found error
        window.after(0, update_gui_on_completion, "FileNotFound", file, error_message_ffmpeg_path) # Update GUI with FFmpeg not found error (using update_gui_on_completion, might need to use _update_gui_on_analysis_completion for analysis)
    except Exception as e: # Handle any other unexpected exceptions during analysis
        error_message_unknown = get_text("analysis_unknown_error_message").format(error=e) # Get the unknown error message with exception details
        append_log("ERROR: " + error_message_unknown + "\n") # Log unknown error
        window.after(0, update_gui_on_completion, "UnknownError", file, error_message_unknown) # Update GUI with unknown error (using update_gui_on_completion, might need to use _update_gui_on_analysis_completion for analysis)
    finally: # Ensure normalization_process is reset to None after analysis is complete (or error occurs)
        normalization_process = None


def _update_gui_after_process(status, process_type, message_text="", error_message="", output_file=None):
    """
    Updates GUI elements after a process (analysis or normalization) completes.
    This is a helper function called by both _update_gui_on_analysis_completion and update_gui_on_completion.
    It handles common GUI updates after a process finishes, including enabling/disabling buttons, stopping progress bar,
    updating the process information text area, showing message boxes for errors or success, and playing sound beeps.
    """
    analyze_button.config(state=tk.NORMAL) # Re-enable the analyze button
    start_button.config(state=tk.NORMAL) # Re-enable the start button
    cancel_button.config(state=tk.DISABLED) # Disable the cancel button (as process is finished)
    progressbar.stop() # Stop the progress bar animation
    progressbar.grid_forget() # Hide the progress bar from the grid layout
    process_info_field.config(state=tk.NORMAL) # Enable process information text area for writing
    process_info_field.delete("1.0", tk.END) # Clear the process information text area

    if status == "Success": # If the process was successful
        process_info_field.insert(tk.END, message_text) # Insert the success message into the process info area
        winsound.Beep(1000, 200) # Play a short beep sound for success
    elif status == "Error": # If the process resulted in an error
        process_info_field.insert(tk.END, error_message) # Insert the error message into the process info area
        process_info_field.config(state=tk.DISABLED) # Disable process info area to prevent editing
        error_title_key = f"{process_type.lower()}_ffmpeg_error_title" # Construct error title key based on process type
        messagebox.showerror(get_text(error_title_key), error_message, parent=window) # Show an error message box with translated title and error message
        winsound.Beep(1500, 500) # Play a longer, higher-pitched beep sound for error
    elif status == "FileNotFound": # If FFmpeg executable was not found
        process_info_field.insert(tk.END, error_message) # Insert the file not found error message
        process_info_field.config(state=tk.DISABLED) # Disable process info area
        error_title_key = f"{process_type.lower()}_ffmpeg_not_found_error_title" # Construct error title key
        messagebox.showerror(get_text(error_title_key), error_message, parent=window) # Show file not found error message box
        winsound.Beep(1500, 500) # Play error beep sound
    elif status == "UnknownError": # If an unknown error occurred
        process_info_field.config(state=tk.NORMAL) # Enable process info area
        process_info_field.insert(tk.END, error_message) # Insert the unknown error message
        process_info_field.config(state=tk.DISABLED) # Disable process info area
        error_title_key = f"{process_type.lower()}_unknown_error_title" # Construct error title key
        messagebox.showerror(get_text(error_title_key), get_text(f"{process_type.lower()}_unknown_error_message_title_only"), parent=window) # Show unknown error message box (title only from translated text)
        winsound.Beep(1500, 500) # Play error beep sound
    elif status == "Cancel": # If the process was canceled by the user
        process_info_field.insert(tk.END, message_text) # Insert the cancel message
        process_info_field.config(state=tk.DISABLED) # Disable process info area
        winsound.Beep(500, 300) # Play a lower-pitched, shorter beep for cancellation

    process_info_field.config(state=tk.DISABLED) # Ensure process info field is disabled after update
    window.after(100, lambda: window.focus_force()) # Force focus back to the main window after message boxes
    window.after(100, lambda: window.update()) # Force GUI update to ensure immediate visual feedback


def _update_gui_on_analysis_completion(status=None, file=None, result=None):
    """
    Updates GUI after audio analysis completion.
    This function is called from the audio_analysis_thread_function using window.after to ensure GUI updates are done on the main thread.
    It formats the analysis results for display in the process information text area and calls _update_gui_after_process
    to handle the common GUI updates after analysis completion.
    """
    if status == "Success": # If analysis was successful
        analysis_results = result # Get the analysis results
        output_text = "\n" # Start building the output text
        output_text += get_text("analysis_completed_message_process_info").format(
                                     filename=os.path.basename(file)
                                 ) + "\n\n" # Add analysis completion message with filename
        if analysis_results: # If analysis results are available
            output_text += get_text("analysis_result_input_i").format(input_i=analysis_results['input_i']) + "\n" # Add input integrated loudness (LUFS-I)
            output_text += get_text("analysis_result_input_tp").format(input_tp=analysis_results['input_tp']) + "\n" # Add input true peak level (dBTP)
            output_text += get_text("analysis_result_lra").format(lra=analysis_results['lra']) + "\n\n" # Add loudness range (LRA)
        output_text += get_text("analysis_hint_log_file") # Add a hint about the analysis log file
        _update_gui_after_process(status, "Analysis", message_text=output_text) # Call _update_gui_after_process with success status, "Analysis" process type, and formatted output text
    elif status == "Error": # If analysis resulted in an error
        _update_gui_after_process(status, "Analysis", error_message=result) # Call _update_gui_after_process with error status, "Analysis" type, and error message
    elif status == "FileNotFound": # If FFmpeg was not found during analysis
        _update_gui_after_process(status, "Analysis", error_message=error_message) # Call _update_gui_after_process for FileNotFoundError
    elif status == "UnknownError": # If an unknown error occurred during analysis
        _update_gui_after_process(status, "Analysis", error_message=result) # Call _update_gui_after_process for UnknownError
    elif status == "Cancel": # If analysis was cancelled (not applicable to analysis in current implementation, but included for completeness)
        _update_gui_after_process(status, "Analysis", message_text=get_text("normalization_canceled_by_user_message")) # Call _update_gui_after_process for Cancel status


def update_gui_on_completion(status=None, output_file=None, error_message=None):
    """
    Updates GUI after normalization completion.
    This function is called from the normalize_audio_thread_function using window.after to ensure GUI updates are on the main thread.
    It formats the completion message for display in the process information text area and calls _update_gui_after_process
    to handle the common GUI updates after normalization completion.
    """
    if status == "Success": # If normalization was successful
        output_text = "\n" # Start building output text
        output_text += get_text("normalization_completed_message_process_info").format(output_filename=os.path.basename(output_file)) + "\n\n" # Add normalization completion message with output filename
        output_text += get_text("normalization_hint_log_file") # Add a hint about the normalization log file
        _update_gui_after_process(status, "Normalization", message_text=output_text, output_file=output_file) # Call _update_gui_after_process with success status, "Normalization" type, and formatted output text
    elif status == "Error": # If normalization resulted in an error
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process with error status, "Normalization" type, and error message
    elif status == "FileNotFound": # If FFmpeg was not found during normalization
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process for FileNotFoundError
    elif status == "UnknownError": # If an unknown error occurred during normalization
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process for UnknownError
    elif status == "Cancel": # If normalization was cancelled by user
        _update_gui_after_process(status, "Normalization", message_text=get_text("normalization_canceled_by_user_message")) # Call _update_gui_after_process for Cancel status


def start_normalization():
    """
    Initiates audio normalization using FFmpeg in a separate thread.
    Retrieves input parameters from the GUI (file, output format, LUFS/TP presets/custom values), validates input,
    constructs the FFmpeg normalization command, handles output file naming and overwrite confirmation,
    starts a normalization process in a background thread, and updates the GUI to reflect the ongoing process.
    Calls normalize_audio_thread_function to execute the FFmpeg normalization command.
    """
    file = file_input.get() # Get the input file path from the GUI
    output_format = output_format_var.get() # Get the selected output format from the GUI
    lufs_preset_name = lufs_preset_var.get() # Get the selected LUFS preset name from the GUI
    target_lufs_input = "" # Initialize variable to store custom LUFS input value
    true_peak_preset_name = true_peak_preset_var.get() # Get the selected True Peak preset name from the GUI
    target_true_peak_input = "" # Initialize variable to store custom True Peak input value
    target_lufs = "" # Initialize variable to store the target LUFS value (numerical)
    target_true_peak = "" # Initialize variable to store the target True Peak value (numerical)

    if not file: # Check if an input file is selected
        messagebox.showerror(get_text("normalization_no_file_error_title"),
                             get_text("normalization_no_file_error_message"),
                             parent=window) # Show an error message if no input file is selected
        return # Exit if no input file

    if lufs_preset_name == get_text("lufs_preset_custom"): # Check if the selected LUFS preset is "Custom"
        target_lufs_input = lufs_input.get() # Get the custom LUFS value from the input field
        if not target_lufs_input: # If the custom LUFS input field is empty
            target_lufs = "-10" # Default to -10 LUFS if custom input is empty
            messagebox.showinfo(get_text("normalization_custom_lufs_note_title"),
                                 get_text("normalization_custom_lufs_note_message"), parent=window) # Show an info message about using default -10 LUFS
        else: # If there is a custom LUFS input value
            try: # Try to convert the custom LUFS input to a float
                target_lufs_value = float(target_lufs_input) # Convert custom LUFS input to float
                if not -70 <= target_lufs_value <= -5: # Validate the custom LUFS value range
                    messagebox.showerror(
                        get_text("normalization_invalid_lufs_error_title"),
                        get_text("normalization_invalid_lufs_error_message"),
                        parent=window) # Show an error message if LUFS value is out of range
                    return # Exit if LUFS value is invalid
                target_lufs = str(target_lufs_value) # Convert the valid LUFS value back to string for FFmpeg command
            except ValueError: # Handle case where custom LUFS input is not a valid number
                messagebox.showerror(
                    get_text("normalization_invalid_lufs_error_title"),
                    get_text("normalization_invalid_lufs_error_message"),
                    parent=window) # Show an error message if LUFS input is not a number
                return # Exit if LUFS input is not a number

    else: # If a pre-defined LUFS preset is selected
        target_lufs = LUFS_PRESETS[
            lufs_preset_name] # Get the LUFS value from the LUFS_PRESETS dictionary based on the selected preset name

    if true_peak_preset_name == get_text("true_peak_preset_custom"): # Check if the selected True Peak preset is "Custom"
        target_true_peak_input = true_peak_input.get() # Get the custom True Peak value from the input field
        if not target_true_peak_input: # If the custom True Peak input field is empty
            target_true_peak = "-1" # Default to -1 dBTP if custom input is empty
            messagebox.showinfo(get_text("normalization_custom_tp_note_title"),
                                 get_text("normalization_custom_tp_note_message"), parent=window) # Show an info message about using default -1 dBTP
        else: # If there is a custom True Peak input value
            try: # Try to convert the custom True Peak input to a float
                target_true_peak_value = float(target_true_peak_input) # Convert custom True Peak input to float
                if not -9 <= target_true_peak_value <= 0: # Validate the custom True Peak value range
                    messagebox.showerror(get_text("normalization_invalid_tp_error_title"),
                                         get_text("normalization_invalid_tp_error_message"), parent=window) # Show an error message if True Peak value is out of range
                    return # Exit if True Peak value is invalid
                target_true_peak = str(target_true_peak_value) # Convert the valid True Peak value back to string for FFmpeg command
            except ValueError: # Handle case where custom True Peak input is not a valid number
                messagebox.showerror(
                    get_text("normalization_invalid_tp_error_title"),
                    get_text("normalization_invalid_tp_error_message"),
                    parent=window) # Show an error message if True Peak input is not a number
                return # Exit if True Peak input is not a number
    else: # If a pre-defined True Peak preset is selected
        target_true_peak = TRUE_PEAK_PRESETS[
            true_peak_preset_name] # Get the True Peak value from the TRUE_PEAK_PRESETS dictionary based on the selected preset name


    format_options = { # Dictionary mapping output formats to their settings (extension, codec, options)
        OUTPUT_FORMAT_WAV: {"extension": AUDIO_FILE_EXTENSION_WAV, "codec": CODEC_PCM_F32LE}, # WAV format settings
        OUTPUT_FORMAT_MP3: {"extension": AUDIO_FILE_EXTENSION_MP3, "codec": CODEC_LIBMP3LAME, "options": [FFMPEG_OPTION_BITRATE_MP3, FFMPEG_BITRATE_320K]}, # MP3 format settings (includes bitrate option)
        OUTPUT_FORMAT_FLAC: {"extension": AUDIO_FILE_EXTENSION_FLAC, "codec": CODEC_FLAC}, # FLAC format settings
        OUTPUT_FORMAT_AAC: {"extension": AUDIO_FILE_EXTENSION_M4A, "codec": CODEC_AAC}, # AAC format settings (output extension is M4A)
        OUTPUT_FORMAT_OGG: {"extension": AUDIO_FILE_EXTENSION_OGG, "codec": CODEC_LIBVORBIS, "options": [FFMPEG_OPTION_QSCALE_OGG, FFMPEG_QSCALE_10]} # OGG format settings (includes quality scale option)
    }
    selected_format = format_options[
        output_format] # Get the format settings for the selected output format
    output_file_name_without_extension = os.path.splitext(file)[0] + "-Normalized" # Create output filename without extension (based on input filename + "-Normalized")
    output_file = output_file_name_without_extension + selected_format[
        "extension"] # Construct the full output filename with the correct extension
    temporary_output_file = output_file_name_without_extension + TEMP_FILE_EXTENSION + selected_format[
        "extension"] # Construct a temporary output filename (used during processing, renamed on success)

    if os.path.exists(output_file): # Check if the output file already exists
        winsound.Beep(1500, 500) # Play an error beep sound to indicate potential overwrite
        confirmation = messagebox.askyesno(
             get_text("normalization_overwrite_confirmation_title"),
             get_text("normalization_overwrite_confirmation_message").format(output_filename=os.path.basename(output_file)),
             parent=window
         ) # Ask the user for confirmation before overwriting existing file
        if not confirmation: # If user cancels overwrite confirmation
             return # Exit normalization start function

    ffmpeg_command = [ # Construct the FFmpeg command for audio normalization
        os.path.join(config.ffmpeg_path, FFMPEG_EXECUTABLE_NAME), # Path to FFmpeg executable from config
        "-i", file, # Input audio file path
        "-af", f"loudnorm=I={target_lufs}:TP={target_true_peak}", # Apply loudnorm filter with target LUFS and True Peak values
        "-ar", "48000", # Set output audio sample rate to 48kHz (common standard)
        "-ac", "2", # Set output audio channels to 2 (stereo, common standard)
    ]

    if "options" in selected_format: # Check if the selected output format has additional FFmpeg options (like bitrate for MP3 or quality scale for OGG)
        ffmpeg_command.extend(selected_format["options"]) # Extend the FFmpeg command with format-specific options

    ffmpeg_command.extend(["-c:a", selected_format["codec"], temporary_output_file]) # Add audio codec and temporary output file path to the FFmpeg command

    start_button.config(state=tk.DISABLED) # Disable the start button to prevent multiple clicks
    analyze_button.config(state=tk.DISABLED) # Disable the analyze button during normalization
    cancel_button.config(state=tk.NORMAL) # Enable the cancel button to allow stopping normalization
    progressbar.grid(row=7, column=0, columnspan=3, sticky="ew", padx=GUI_PADX, pady=GUI_PADY) # Show the progress bar
    progressbar.start() # Start the progress bar animation

    process_info_field.config(state=tk.NORMAL) # Enable process information text area for writing
    process_info_field.delete("1.0", tk.END) # Clear the process information text area
    process_info_field.insert(tk.END,
                             get_text("normalization_start_message").format(
                                 filename=os.path.basename(file_input.get()),
                                 target_lufs=target_lufs,
                                 target_true_peak=target_true_peak,
                                 output_format=output_format,
                                 lufs_preset_name=lufs_preset_name,
                                 true_peak_preset_name=true_peak_preset_name
                             )) # Display normalization start message in the process info area with parameters
    process_info_field.config(state=tk.DISABLED) # Disable the process information text area to prevent user editing
    process_info_field.grid() # Ensure the process info field is visible in the grid layout

    log_entry_start = f"======================== {datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')} ========================\n" # Create a log entry start separator with timestamp
    log_entry_start += get_text("normalization_start_message").format(
                                 filename=file,
                                 target_lufs=target_lufs,
                                 output_format=output_format,
                                 target_true_peak=target_true_peak,
                                 lufs_preset_name=lufs_preset_name,
                                 true_peak_preset_name=true_peak_preset_name
                             ) + "\n" # Add normalization start message with parameters to the log entry
    log_entry_start += get_text("normalization_log_ffmpeg_command") + "\n" + " ".join(ffmpeg_command) + "\n\n" # Add FFmpeg command to the log entry
    append_log(log_entry_start) # Append the starting log entry to the normalization log file

    thread = threading.Thread(target=normalize_audio_thread_function,
                              args=(ffmpeg_command, temporary_output_file, output_file)) # Create a new thread to run the normalize_audio_thread_function
    thread.start() # Start the normalization thread


def normalize_audio_thread_function(ffmpeg_command, temporary_output_file, output_file):
    """
    Executes FFmpeg normalization command in a thread.
    This function runs in a separate thread to perform audio normalization using FFmpeg.
    It executes the provided FFmpeg command using subprocess.Popen, captures stdout and stderr, and checks the return code.
    On success (return code 0), it renames the temporary output file to the final output file.
    On error (non-zero return code) or exceptions, it handles cleanup, logs errors, and updates the GUI using window.after
    to signal completion status (success, error, cancel, file not found, unknown error).
    """
    global normalization_process # Access the global normalization_process variable

    try: # Try to execute the FFmpeg normalization command and handle potential errors
        normalization_process = subprocess.Popen(ffmpeg_command, stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                                     text=True,
                                                     creationflags=subprocess.CREATE_NO_WINDOW) # Run FFmpeg command using Popen to allow for cancellation, capture stdout and stderr, hide console window
        stdout, stderr = normalization_process.communicate() # Wait for the process to finish and get stdout and stderr
        result_code = normalization_process.returncode # Get the return code of the FFmpeg process

        log_output = stdout + "\n" + stderr # Combine stdout and stderr for logging
        append_log(log_output) # Append the FFmpeg output to the normalization log

        if result_code == 0: # If FFmpeg command was successful (return code 0)
            try: # Try to rename the temporary output file to the final output file
                if os.path.exists(output_file): # Check if the final output file already exists (shouldn't happen due to overwrite confirmation, but for robustness)
                    os.remove(output_file) # Remove existing output file if it exists
                os.rename(temporary_output_file, output_file) # Rename the temporary file to the final output file
                window.after(0, update_gui_on_completion, "Success", output_file) # Update GUI on the main thread with success status and output file path
            except OSError as e: # Handle OSError exceptions during file rename (e.g., permissions issues)
                error_message_rename = get_text("normalization_rename_error").format(error=e) # Get the file rename error message
                append_log("ERROR: " + error_message_rename + "\n") # Log the rename error
                window.after(0, update_gui_on_completion, "Error", output_file, error_message_rename) # Update GUI with rename error

        elif result_code == 1: # If FFmpeg process was terminated by user (return code 1 - based on cancel logic, might need to verify FFmpeg behavior)
            if os.path.exists(temporary_output_file): # Clean up temporary file if it exists
                os.remove(temporary_output_file) # Remove temporary file
            window.after(0, update_gui_on_completion, "Cancel",
                         output_file) # Update GUI with "Cancel" status

        else: # If FFmpeg command failed (return code other than 0 or 1)
            if os.path.exists(temporary_output_file): # Clean up temporary file if it exists
                os.remove(temporary_output_file) # Remove temporary file
            error_message = get_text("normalization_ffmpeg_error_message").format(return_code=result_code, stderr=stderr) # Get generic FFmpeg error message with return code and stderr
            append_log("ERROR: " + error_message + "\n") # Log the FFmpeg error
            window.after(0, update_gui_on_completion, "Error", output_file, error_message) # Update GUI with error status and FFmpeg error message

    except FileNotFoundError: # Handle case where FFmpeg executable is not found
        error_message = get_text("normalization_ffmpeg_not_found_error_message") # Get FFmpeg not found error message
        append_log("ERROR: " + error_message + "\n") # Log FFmpeg not found error
        window.after(0, update_gui_on_completion, "FileNotFound", output_file, error_message) # Update GUI with FileNotFoundError status
    except Exception as e: # Handle any other unexpected exceptions during normalization
        error_message = get_text("normalization_unknown_error_message").format(error=e) # Get generic unknown error message with exception details
        append_log("ERROR: " + error_message + "\n") # Log unknown error
        window.after(0, update_gui_on_completion, "UnknownError", output_file, error_message) # Update GUI with UnknownError status
    finally: # Ensure cleanup and GUI update regardless of success or error
        normalization_process = None # Reset normalization_process to None after thread finishes
        if os.path.exists(temporary_output_file) and status not in ["Success", "Cancel"]: # If temporary file exists and process was not successful or cancelled
            try: # Try to remove the temporary file
                os.remove(temporary_output_file) # Remove temporary file to clean up after errors
            except OSError: # Handle potential OSError during temporary file removal (e.g., permissions issues)
                pass # Ignore OSError during cleanup (non-critical error)
        window.after(0, update_gui_on_completion) # Call update_gui_on_completion again (might be redundant, check logic flow)


def update_gui_on_completion(status=None, output_file=None, error_message=None):
    """
    Updates GUI after normalization completion. (Redundant function - identical to the previous one. Likely a copy-paste error.)
    This function is called from the normalize_audio_thread_function using window.after to ensure GUI updates are on the main thread.
    It formats the completion message for display in the process information text area and calls _update_gui_after_process
    to handle the common GUI updates after normalization completion.
    """
    if status == "Success": # If normalization was successful
        output_text = "\n" # Start building output text
        output_text += get_text("normalization_completed_message_process_info").format(output_filename=os.path.basename(output_file)) + "\n\n" # Add normalization completion message with output filename
        output_text += get_text("normalization_hint_log_file") # Add a hint about the normalization log file
        _update_gui_after_process(status, "Normalization", message_text=output_text, output_file=output_file) # Call _update_gui_after_process with success status, "Normalization" type, and formatted output text
    elif status == "Error": # If normalization resulted in an error
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process with error status, "Normalization" type, and error message
    elif status == "FileNotFound": # If FFmpeg was not found during normalization
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process for FileNotFoundError
    elif status == "UnknownError": # If an unknown error occurred during normalization
        _update_gui_after_process(status, "Normalization", error_message=error_message) # Call _update_gui_after_process for UnknownError
    elif status == "Cancel": # If normalization was cancelled by user
        _update_gui_after_process(status, "Normalization", message_text=get_text("normalization_canceled_by_user_message")) # Call _update_gui_after_process for Cancel status


def cancel_normalization():
    """
    Terminates the normalization process if it is running.
    If a normalization process is currently running (normalization_process is not None), it terminates the process using normalization_process.terminate().
    Updates the GUI to stop the progress bar and display a cancellation message in the process information text area.
    If no normalization process is running, it shows an information message to the user.
    """
    global normalization_process # Access the global normalization_process variable

    if normalization_process: # Check if a normalization process is currently running
        normalization_process.terminate() # Terminate the FFmpeg normalization subprocess

        progressbar.stop() # Stop the progress bar animation
        progressbar.grid_forget() # Hide the progress bar

        process_info_field.config(state=tk.NORMAL) # Enable process information text area for writing
        process_info_field.delete("1.0", tk.END) # Clear the process information text area
        process_info_field.insert(tk.END,
                                 get_text("normalization_cancel_process_message")) # Display cancellation message in the process info area
        process_info_field.config(state=tk.DISABLED) # Disable the process information text area to prevent user editing


    else: # If no normalization process is running
        messagebox.showinfo(get_text("normalization_cancel_title"),
                             get_text("normalization_no_process_cancel_message"),
                             parent=window) # Show an information message that no process is running to cancel


def exit_program():
    """
    Exits the application.
    Destroys the main window, which terminates the Tkinter main loop and exits the application.
    """
    window.destroy() # Destroy the main window, which will exit the application


def update_lufs_entry_state(*args):
    """
    Updates the state of the custom LUFS input entry based on the LUFS preset combobox selection.
    If the selected preset is "Custom", it enables the LUFS input entry and updates the LUFS label text to "Target LUFS (Custom)".
    Otherwise, it disables the LUFS input entry and hides the LUFS label and input field.
    """
    if lufs_preset_var.get() == get_text("lufs_preset_custom"): # Check if the selected LUFS preset is "Custom"
        lufs_label.config(text=get_text("target_lufs_label_custom_short")) # Change LUFS label text to "Target LUFS (Custom)"
        lufs_label.grid(row=2, column=0, sticky="w", padx=(0, GUI_LABEL_PADX), pady=(GUI_LABEL_PADY, GUI_PADY)) # Re-grid the LUFS label to make it visible
        lufs_input.grid(row=2, column=1, sticky="w", padx=(0, GUI_PADX), pady=(GUI_LABEL_PADY, GUI_PADY)) # Re-grid the LUFS input field to make it visible
        lufs_input.config(state=tk.NORMAL) # Enable the LUFS input entry field
        lufs_input.delete(0, tk.END) # Clear the current content of the LUFS input field
        lufs_input.insert(0, "-10") # Insert the default value "-10" into the LUFS input field
    else: # If a pre-defined LUFS preset is selected
        lufs_label.config(text=get_text("target_lufs_label_custom")) # Reset LUFS label text to default custom label (might be redundant)
        lufs_label.grid_forget() # Hide the LUFS label from the grid layout
        lufs_input.grid_forget() # Hide the LUFS input field from the grid layout
        lufs_input.config(state=tk.DISABLED) # Disable the LUFS input entry field

def update_log_size_state(check_variable, size_input_field):
    """
    Enables or disables the log size input field based on the state of the single log entry checkbox.
    If single log entry mode is enabled, the log size input is disabled as it's not relevant because log file rolling is not used.
    """
    if check_variable.get(): # If single log entry checkbox is checked (True)
        size_input_field.config(state=tk.DISABLED) # Disable the log size input field
    else:
        size_input_field.config(state=tk.NORMAL) # Otherwise, enable the log size input field

def update_true_peak_entry_state(*args):
    """
    Updates the state of the custom True Peak input entry based on the True Peak preset combobox selection.
    If the selected preset is "Custom", it enables the True Peak input entry and makes the True Peak label and input field visible.
    Otherwise, it disables the True Peak input entry and hides the True Peak label and input field.
    """
    if true_peak_preset_var.get() == get_text("true_peak_preset_custom"): # Check if the selected True Peak preset is "Custom"
        true_peak_label.config(text=get_text("true_peak_label")) # Ensure True Peak label text is set to "True Peak" (might be redundant)
        true_peak_label.grid(row=2, column=2, sticky="w", padx=(GUI_LABEL_PADX * 2, GUI_LABEL_PADX), pady=(GUI_LABEL_PADY, 0)) # Re-grid the True Peak label to make it visible
        true_peak_input.grid(row=2, column=3, sticky="w", padx=(0, GUI_PADX), pady=(GUI_LABEL_PADY, 0)) # Re-grid the True Peak input field to make it visible
        true_peak_label.config(text=get_text("true_peak_label")) # Redundant line - already set above
        true_peak_input.config(state=tk.NORMAL) # Enable the True Peak input entry field
        true_peak_input.delete(0, tk.END) # Clear the current content of the True Peak input field
        true_peak_input.insert(0, "-1") # Insert the default value "-1" into the True Peak input field
    else: # If a pre-defined True Peak preset is selected
        true_peak_label.grid_forget() # Hide the True Peak label from the grid layout
        true_peak_input.grid_forget() # Hide the True Peak input field from the grid layout
        true_peak_input.config(state=tk.DISABLED) # Disable the True Peak input entry field


def append_log(text):
    """
    Appends text to the normalization log file.
    Calls the helper function _append_log_with_rolling to handle the actual log writing and rolling logic for the normalization log.
    """
    _append_log_with_rolling(LOG_FILE_NAME, text, config) # Call _append_log_with_rolling for normalization log


def append_analysis_log(text):
    """
    Appends text to the analysis log file.
    Calls the helper function _append_analysis_log_with_rolling to handle the actual log writing and rolling logic for the analysis log.
    """
    _append_analysis_log_with_rolling(ANALYSIS_LOG_FILE_NAME, text, config) # Call _append_analysis_log_with_rolling for analysis log


def _append_analysis_log_with_rolling(log_file_name, text, config_obj):
    """
    Helper function to append text to a log file with rolling for analysis log.
    Handles log file creation, rolling (truncating the log if it exceeds the size limit), and appending new text.
    In single log entry mode, it overwrites the log file on each run.
    In rolling log mode, it keeps the log file size within the configured limit by removing older lines.
    """
    try: # Use try-except to handle potential file writing errors
        log_file_path = log_file_name # Get the log file path
        log_size_limit_bytes = config_obj.log_file_size_kb * 1024 # Calculate the log size limit in bytes from KB

        if config_obj.single_log_entry_enabled: # Check if single log entry mode is enabled
            if os.path.exists(log_file_path): # If single log entry mode and log file exists
                with open(log_file_path, "w") as logfile_truncate: # Open the log file in write mode (truncates existing content)
                    pass # Truncate the log file by opening and closing in write mode (empty the file)
        else: # If rolling log mode is enabled
            if os.path.exists(log_file_path) and os.path.getsize(log_file_path) >= log_size_limit_bytes: # Check if log file exists and exceeds size limit
                with open(log_file_path, "r") as logfile_r: # Open the log file in read mode
                    lines = logfile_r.readlines() # Read all lines from the log file

                lines_to_write = lines[len(lines) // 2:] # Keep only the last half of the lines (rolling logic - removes older lines)

                with open(log_file_path, "w") as logfile_w: # Open the log file in write mode (overwrites with truncated content)
                    logfile_w.writelines(lines_to_write) # Write the last half of the lines back to the log file (effectively rolling the log)

        with open(log_file_path, "a") as logfile: # Open the log file in append mode
            logfile.write(text) # Append the new text to the log file

    except Exception as e: # Handle any exceptions during log file writing (e.g., permissions issues, disk full)
        print(f"Error writing to log file {log_file_name}: {e}") # Print an error message to the console if logging fails


def _append_log_with_rolling(log_file_name, text, config_obj):
    """
    Helper function to append text to a log file with rolling for normalization log.
    Identical in function to _append_analysis_log_with_rolling, but used for the normalization log file.
    Handles log file creation, rolling, and appending text, with single log entry mode and rolling log mode support.
    """
    try: # Use try-except to handle potential file writing errors
        log_file_path = log_file_name # Get the log file path
        log_file_size_bytes = config_obj.log_file_size_kb * 1024 # Calculate the log size limit in bytes from KB

        if config_obj.single_log_entry_enabled: # Check if single log entry mode is enabled
            if os.path.exists(log_file_path): # If single log entry mode and log file exists
                with open(log_file_path, "w") as logfile_truncate: # Open the log file in write mode (truncates existing content)
                    pass # Truncate the log file by opening and closing in write mode (empty the file)
        else: # If rolling log mode is enabled
            if os.path.exists(log_file_path) and os.path.getsize(log_file_path) >= log_file_size_bytes: # Check if log file exists and exceeds size limit
                with open(log_file_path, "r") as logfile_r: # Open the log file in read mode
                    lines = logfile_r.readlines() # Read all lines from the log file

                lines_to_write = lines[len(lines) // 2:] # Keep only the last half of the lines (rolling logic - removes older lines)

                with open(log_file_path, "w") as logfile_w: # Open the log file in write mode (overwrites with truncated content)
                    logfile_w.writelines(lines_to_write) # Write the last half of the lines back to the log file (effectively rolling the log)

        with open(log_file_path, "a") as logfile: # Open the log file in append mode
            logfile.write(text) # Append the new text to the log file

    except Exception as e: # Handle any exceptions during log file writing (e.g., permissions issues, disk full)
        print(f"Error writing to log file {log_file_name}: {e}") # Print an error message to the console if logging fails


def update_language_selection(event, language_variable):
    """
    Updates the application language based on combobox selection.
    Loads the selected language, applies the language changes to the GUI (main window and dialogs),
    and resets LUFS and True Peak presets to their default values after language change to ensure consistency.
    """
    global current_language # Access the global current_language variable
    current_language = language_variable.get() # Get the selected language code from the combobox variable
    load_language(current_language) # Load the language data for the selected language code
    apply_language("options") # Apply language changes to the options dialog
    apply_language("main") # Apply language changes to the main window
    if window.children.get("!toplevel2"): # If the info dialog is open
        apply_language("info") # Apply language changes to the info dialog

    # --- Reset LUFS and True Peak Presets to Default after Language Change ---
    lufs_preset_var.set(LUFS_PRESET_NAMES[0]) # Set LUFS Preset to Default (first in the list, usually "Default")
    true_peak_preset_var.set(TRUE_PEAK_PRESET_NAMES[0]) # Set True Peak Preset to Default (first in the list, usually "Default")
    update_lufs_entry_state() # Update LUFS entry state (enable/disable custom input based on preset)
    update_true_peak_entry_state() # Update True Peak entry state (enable/disable custom input based on preset)
# --- Main Window Initialization ---
window = tk.Tk() # Create the main Tkinter window
window.title(f"{get_text('app_title')} v{VERSION}") # Set the initial title of the main window, using translated app title and version
window_width = 800 # Define initial window width
window_height = 630 # Define initial window height
window.geometry(f"{window_width}x{window_height}") # Set the initial window size

screen_width = window.winfo_screenwidth() # Get the screen width
screen_height = window.winfo_screenheight() # Get the screen height
x_position = (screen_width - window_width) // 2 # Calculate horizontal center position for the window
y_position = (screen_height - window_height) // 2 # Calculate vertical center position for the window
window.geometry(f"+{x_position}+{y_position}") # Position the window in the center of the screen

style = Style() # Create a style object for ttk widgets
style.theme_use(STYLE_THEME_NAME) # Set the style theme to 'clam' (defined as constant)

background_color = "#e0e0e0" # Define background color for the GUI
text_color = "#424242" # Define text color for the GUI (currently unused)
accent_color = "#b0bec5" # Define accent color for the GUI (currently unused)
separator_color = "#cccccc" # Define separator line color for the GUI

window.configure(bg=background_color) # Set the background color of the main window

standard_font = ("Helvetica", 9) # Define a standard font for the GUI
window.option_add("*Font", standard_font) # Set the default font for all widgets in the window

style.configure("TButton", padding=4, font=('Helvetica', 9)) # Configure style for ttk Buttons
style.configure("TLabel", font=('Helvetica', 9)) # Configure style for ttk Labels
style.configure("TCombobox", font=('Helvetica', 9)) # Configure style for ttk Comboboxes
style.configure("TEntry", font=('Helvetica', 9)) # Configure style for ttk Entries
style.configure("TCheckbutton", font=('Helvetica', 9)) # Configure style for ttk Checkbuttons
style.configure("TLabelframe", font=('Helvetica', 9, 'bold')) # Configure style for ttk LabelFrames (bold font)
style.configure("TLabelframe.Label", font=('Helvetica', 9, 'bold')) # Configure style for LabelFrames labels (bold font)


menubar = tk.Menu(window) # Create the menu bar for the main window
filemenu = tk.Menu(menubar, tearoff=0) # Create the 'File' menu (tearoff=0 removes dotted line to detach menu)
filemenu.add_command(label=get_text("menu_file_options"), command=show_options_dialog) # Add 'Options' command to File menu, using translated label
filemenu.add_separator() # Add a separator line in the File menu
filemenu.add_command(label=get_text("menu_file_exit"), command=exit_program) # Add 'Exit' command to File menu, using translated label
menubar.add_cascade(label=get_text("menu_file"), menu=filemenu) # Add the File menu to the menu bar, using translated label

infomenu = tk.Menu(menubar, tearoff=0) # Create the 'Info' menu
infomenu.add_command(label=get_text("menu_info_about"), command=show_info_box) # Add 'About' command to Info menu, using translated label
menubar.add_cascade(label=get_text("menu_info"), menu=infomenu) # Add the Info menu to the menu bar, using translated label
window.config(menu=menubar) # Configure the main window to use the created menu bar

content_frame = tk.Frame(window, padx=GUI_PADX, pady=GUI_PADY,
                            bg=background_color) # Create a main content frame in the window
content_frame.grid(row=0, column=0, sticky="nsew") # Place the content frame in the grid layout, sticky to all sides (north, south, east, west) to expand

file_frame_group = tk.LabelFrame(content_frame, text=get_text("file_selection_group"), padx=GUI_PADX, pady=GUI_PADY, bg=background_color) # Create a LabelFrame for file selection group
file_frame_group.grid(row=0, column=0, columnspan=3, sticky="ew", pady=(0, GUI_PADY)) # Place file frame in the grid layout, spanning 3 columns, sticky east-west, padding bottom

file_label = tk.Label(file_frame_group, text=get_text("select_audio_file_label"), anchor="w", bg=background_color) # Create a label for file selection
file_label.grid(row=0, column=0, sticky="w", padx=(0, GUI_LABEL_PADX)) # Place file label in the file frame, sticky west, padding right
file_input = ttk.Entry(file_frame_group, width=GUI_ENTRY_WIDTH_FILE) # Create an Entry field for file path input
file_input.grid(row=0, column=1, sticky="ew") # Place file input field in the file frame, sticky east-west to expand
browse_button = ttk.Button(file_frame_group, text=get_text("browse_file_button"), command=browse_file) # Create a button to browse for audio file
browse_button.grid(row=0, column=2, padx=(GUI_LABEL_PADX, 0)) # Place browse button in the file frame, padding left

separator_line_1 = tk.Frame(content_frame, bg=separator_color, height=1) # Create a separator line
separator_line_1.grid(row=1, column=0, columnspan=3, sticky="ew", pady=GUI_PADY) # Place separator line in the grid layout, spanning 3 columns, sticky east-west, padding vertical

loudness_settings_frame_group = tk.LabelFrame(content_frame, text=get_text("loudness_settings_group"), padx=GUI_PADX, pady=GUI_PADY, bg=background_color) # Create a LabelFrame for loudness settings group
loudness_settings_frame_group.grid(row=2, column=0, columnspan=3, sticky="ew", pady=(0, GUI_PADY)) # Place loudness settings frame in the grid layout, spanning 3 columns, sticky east-west, padding bottom

lufs_preset_label = tk.Label(loudness_settings_frame_group, text=get_text("lufs_preset_label"), anchor="w", bg=background_color) # Create a label for LUFS preset selection
lufs_preset_label.grid(row=1, column=0, sticky="w", padx=(0, GUI_LABEL_PADX), pady=(0, GUI_LABEL_PADY)) # Place LUFS preset label, sticky west, padding right, bottom
lufs_preset_var = tk.StringVar() # Create a StringVar to hold the selected LUFS preset
lufs_preset_combobox = Combobox(loudness_settings_frame_group, textvariable=lufs_preset_var, values=LUFS_PRESET_NAMES,
                                  state="readonly", width=GUI_COMBOBOX_WIDTH_LUFS_PRESET) # Create a Combobox for LUFS preset selection, readonly, with preset names as values
lufs_preset_combobox.set(LUFS_PRESET_NAMES[0]) # Set the default selected LUFS preset to the first in the list (usually "Default")
lufs_preset_combobox.grid(row=1, column=1, sticky="ew", padx=(0, GUI_PADX), pady=(0, GUI_LABEL_PADY)) # Place LUFS preset combobox, sticky east-west to expand, padding right, bottom
lufs_preset_combobox.bind("<<ComboboxSelected>>", update_lufs_entry_state) # Bind combobox selection event to update_lufs_entry_state function

true_peak_preset_label = tk.Label(loudness_settings_frame_group, text=get_text("true_peak_preset_label"), anchor="w", bg=background_color) # Create a label for True Peak preset selection
true_peak_preset_label.grid(row=1, column=2, sticky="w", padx=(GUI_LABEL_PADX * 2, GUI_LABEL_PADX), pady=(0, GUI_LABEL_PADY)) # Place True Peak preset label, sticky west, padding left and right, bottom
true_peak_preset_var = tk.StringVar() # Create a StringVar to hold the selected True Peak preset
true_peak_preset_combobox = Combobox(loudness_settings_frame_group, textvariable=true_peak_preset_var,
                                     values=TRUE_PEAK_PRESET_NAMES, state="readonly",
                                     width=GUI_COMBOBOX_WIDTH_TP_PRESET) # Create a Combobox for True Peak preset selection, readonly, with preset names as values
true_peak_preset_combobox.set(TRUE_PEAK_PRESET_NAMES[0]) # Set the default selected True Peak preset to the first in the list (usually "Default")
true_peak_preset_combobox.grid(row=1, column=3, sticky="ew", padx=(0, GUI_PADX), pady=(0, GUI_LABEL_PADY)) # Place True Peak preset combobox, sticky east-west to expand, padding right, bottom
true_peak_preset_combobox.bind("<<ComboboxSelected>>", update_true_peak_entry_state) # Bind combobox selection event to update_true_peak_entry_state function

lufs_label = tk.Label(loudness_settings_frame_group, text=get_text("target_lufs_label_custom"), anchor="w", bg=background_color) # Create a label for custom LUFS input (initially hidden)
lufs_label.grid(row=2, column=0, sticky="w", padx=(0, GUI_LABEL_PADX), pady=(GUI_LABEL_PADY, GUI_PADY)) # Place LUFS custom label (initially visible but might be hidden by update_lufs_entry_state)
lufs_input = ttk.Entry(loudness_settings_frame_group, width=GUI_ENTRY_WIDTH_LUFS_TP, state=tk.DISABLED) # Create an Entry field for custom LUFS input, initially disabled
lufs_input.insert(0, "-10") # Insert the default value "-10" into the LUFS input field

true_peak_label = tk.Label(loudness_settings_frame_group, text=get_text("true_peak_label"), anchor="w", bg=background_color) # Create a label for custom True Peak input (initially hidden)
true_peak_label.grid(row=2, column=2, sticky="w", padx=(GUI_LABEL_PADX * 2, GUI_LABEL_PADX), pady=(GUI_LABEL_PADY, 0)) # Place True Peak custom label (initially visible but might be hidden by update_true_peak_entry_state)

true_peak_input = ttk.Entry(loudness_settings_frame_group, width=GUI_ENTRY_WIDTH_LUFS_TP, state=tk.DISABLED) # Create an Entry field for custom True Peak input, initially disabled
true_peak_input.insert(0, "-1") # Insert the default value "-1" into the True Peak input field


separator_line_2 = tk.Frame(content_frame, bg=separator_color, height=1) # Create a separator line
separator_line_2.grid(row=3, column=0, columnspan=3, sticky="ew", pady=GUI_PADY) # Place separator line in the grid layout, spanning 3 columns, sticky east-west, padding vertical

output_format_frame_group = tk.LabelFrame(content_frame, text=get_text("output_format_group"), padx=GUI_PADX, pady=GUI_PADY, bg=background_color) # Create a LabelFrame for output format group
output_format_frame_group.grid(row=4, column=0, columnspan=3, sticky="ew", pady=(0, GUI_PADY)) # Place output format frame in the grid layout, spanning 3 columns, sticky east-west, padding bottom

output_format_label = tk.Label(output_format_frame_group, text=get_text("output_format_file_format_label"), anchor="w", bg=background_color) # Create a label for output format selection
output_format_label.grid(row=1, column=0, sticky="w", padx=(0, GUI_LABEL_PADX), pady=(0, GUI_LABEL_PADY)) # Place output format label, sticky west, padding right, bottom
output_format_var = tk.StringVar() # Create a StringVar to hold the selected output format
output_format_combobox = Combobox(output_format_frame_group, textvariable=output_format_var,
                                   values=OUTPUT_FORMATS_LIST, state="readonly") # Create a Combobox for output format selection, readonly, with format list as values
output_format_combobox.set(OUTPUT_FORMAT_WAV) # Set the default selected output format to WAV
output_format_combobox.grid(row=1, column=1, sticky="w", padx=(0, GUI_PADX), pady=(0, GUI_LABEL_PADY)) # Place output format combobox, sticky west, padding right, bottom

separator_line_3 = tk.Frame(content_frame, bg=separator_color, height=1) # Create a separator line
separator_line_3.grid(row=5, column=0, columnspan=3, sticky="ew", pady=GUI_PADY) # Place separator line in the grid layout, spanning 3 columns, sticky east-west, padding vertical

buttons_frame = tk.Frame(content_frame, bg=background_color) # Create a frame to hold the action buttons
buttons_frame.grid(row=6, column=0, columnspan=3, pady=(GUI_PADY, GUI_PADY),
                    sticky="ew") # Place buttons frame in the grid layout, spanning 3 columns, sticky east-west, padding vertical

analyze_button = ttk.Button(buttons_frame, text=get_text("analyze_audio_button"), command=analyze_audio) # Create 'Analyze Audio' button
analyze_button.grid(row=0, column=0, sticky="ew", padx=(0, GUI_BUTTON_PADX)) # Place analyze button, sticky east-west to expand, padding right

start_button = ttk.Button(buttons_frame, text=get_text("start_normalization_button"), command=start_normalization) # Create 'Start Normalization' button
start_button.grid(row=0, column=1, sticky="ew", padx=(0, GUI_BUTTON_PADX)) # Place start button, sticky east-west to expand, padding right

cancel_button = ttk.Button(buttons_frame, text=get_text("cancel_normalization_button"), command=cancel_normalization, state=tk.DISABLED) # Create 'Cancel Normalization' button, initially disabled
cancel_button.grid(row=0, column=2, sticky="ew") # Place cancel button, sticky east-west to expand

buttons_frame.columnconfigure(0, weight=1) # Configure the first column of the buttons frame to expand
buttons_frame.columnconfigure(1, weight=1) # Configure the second column of the buttons frame to expand
buttons_frame.columnconfigure(2, weight=1) # Configure the third column of the buttons frame to expand

progressbar = Progressbar(content_frame, mode='indeterminate') # Create a progress bar in indeterminate mode (no specific end value)

process_information_frame_group = tk.LabelFrame(content_frame, text=get_text("process_information_group"), padx=GUI_PADX, pady=GUI_PADY, bg=background_color) # Create a LabelFrame for process information group
process_information_frame_group.grid(row=8, column=0, columnspan=3, sticky="nsew") # Place process information frame, spanning 3 columns, sticky to all sides to expand

process_info_field = tk.Text(process_information_frame_group, height=GUI_PROCESS_INFO_HEIGHT, width=GUI_PROCESS_INFO_WIDTH,
                              wrap=tk.WORD, state=tk.DISABLED, bg=PROCESS_INFO_BACKGROUND_COLOR, font=('Helvetica', 9)) # Create a Text widget to display process information, readonly, with word wrap
process_info_field.grid(row=1, column=0, sticky="nsew", padx=GUI_LABEL_PADX,
                        pady=(0, GUI_LABEL_PADY)) # Place process info text field, sticky to all sides to expand, padding horizontal and bottom

process_information_frame_group.columnconfigure(0, weight=1) # Configure the column of the process info frame to expand
process_information_frame_group.rowconfigure(1, weight=1) # Configure the row of the process info frame to expand

content_frame.columnconfigure(0, weight=1) # Configure the column of the content frame to expand
content_frame.rowconfigure(8, weight=1) # Configure the last row of the content frame to expand (process info frame)
window.columnconfigure(0, weight=1) # Configure the main window's column to expand
window.rowconfigure(0, weight=1) # Configure the main window's row to expand (content frame)

load_language(current_language) # Load the language data based on the current language setting
apply_language("main") # Apply the loaded language to the main window GUI elements

update_lufs_entry_state() # Initialize the state of the custom LUFS input entry (enable/disable based on default preset)
update_true_peak_entry_state() # Initialize the state of the custom True Peak input entry (enable/disable based on default preset)


menubar = tk.Menu(window) # Create the menu bar again (redundant - already created above)
filemenu = tk.Menu(menubar, tearoff=0) # Create the 'File' menu again (redundant)
filemenu.add_command(label=get_text("menu_file_options"), command=show_options_dialog) # Add 'Options' command again (redundant)
filemenu.add_separator() # Add separator again (redundant)
filemenu.add_command(label=get_text("menu_file_exit"), command=exit_program) # Add 'Exit' command again (redundant)
menubar.add_cascade(label=get_text("menu_file"), menu=filemenu) # Add File menu to menubar again (redundant)

infomenu = tk.Menu(menubar, tearoff=0) # Create the 'Info' menu again (redundant)
infomenu.add_command(label=get_text("menu_info_about"), command=show_info_box) # Add 'About' command again (redundant)
menubar.add_cascade(label=get_text("menu_info"), menu=infomenu) # Add Info menu to menubar again (redundant)
window.config(menu=menubar) # Configure window menu again (redundant)

window.mainloop() # Start the Tkinter main event loop, which runs the GUI and waits for user interactions
