/*  modplay.h - Tiny MOD Player V2.10 for Watcom C/C++ and DOS/4GW

    Module player for Sound Blaster and compatibles header file.

    Copyright 1993,94 Carlos Hasan
*/

/******************************************************************************\
 QUICK REFERENCE GUIDE:

 MODDetectCard
 
 Function:      Detect Sound Blaster card.

 Prototype:     int MODDetectCard(&Port,&IRQ,&DRQ)

 Parameters:    Port    - I/O Port address
                IRQ     - IRQ level
                DRQ     - DMA channel

 Returns:       On success, return a zero value.
                On error, return a non zero value.

 Remarks:       This function will detect the Sound Blaster card and
                will return the configuration parameters. You cannot
                call this routine while playing a module.
                Actually the routine only detect the I/O Port address
                and the IRQ level. The DMA channel 1 is always returned
                by this routine.


 MODPlayModule
 
 Function:      Start playing a Modulefile.

 Prototype:     int MODPlayModule(Song,Chans,Rate,Port,IRQ,DRQ)

 Parameters:    Song    - Address of the Module
                Chans   - Number of channels (1-8)
                Rate    - Mixing speed in Hertz (4kHz-22kHz)
                Port    - I/O Port address (220h..280h)
                IRQ     - IRQ level (2,3,5,7)
                DRQ     - DMA channel (0,1,3)

 Returns:       On success, return a zero value.
                On error, return a non zero value.

 Remarks:       This function will initialize the Sound Blaster card and
                will start playing the module file immediately. The module
                file must be a 4,6 or 8 channels Protracker or FastTracker
                music module file.
                The player supports two kinds of channels, music and sample
                audio channels. The music channels are used by the player
                to play the module, and the sample channels are used for
                sound effects and the like.
                The channels voices 0 to N-1 are used for music channels,
                where N is the number of channels of the module file.


 MODStopModule
 
 Function:      Stop playing the current Modulefile.

 Prototype:     void MODStopModule()

 Parameters:    None.

 Returns:       None.

 Remarks:       This function shut down the playing system. Must be called
                before exiting the user program.


 MODPlaySample
 
 Function:      Play instrument at specified period and volume.

 Prototype:     void MODPlaySample(Voice,Instr)

 Parameters:    Voice   - Audio channel number (0-7)
                Instr   - Instrument address

 Returns:       None.

 Remarks:       This function is useful to play samples over music. The sample
                structure holds the period, volume and the address of the 8-bit
                signed samples to be played in the channel.
                The amiga period value can be translated to hertz using the
                following formula:  Hertz = 8363*428/Period


 MODStopSample
 
 Function:      Stop a sample channel.

 Prototype:     void MODStopSample(Voice)

 Parameters:    Voice   - Audio channel number (0-7)

 Remarks:       This function will stop the specified voice setting the channel
                volume to zero. The voice should be a sample channel.


 MODSetPeriod
 
 Function:      Set the sample channel period value.

 Prototype:     void MODSetPeriod(Voice,Period)

 Parameters:    Voice   - Audio channel number (0-7)
                Period  - Amiga Period (113-856)

 Returns:       None.

 Remarks:       This function will change the current frequency of the sample
                channel. The voice should be a sample channel.


 MODSetVolume
 
 Function:      Set the sample channel volume.

 Prototype:     void MODSetVolume(Voice,Volume)

 Parameters:    Voice   - Audio channel number (0-7)
                Volume  - Volume (0-64)

 Returns:       None.

 Remarks:       This function will change the channel volume. The voice should
                be a sample channel.


 MODSetMusicVolume
 
 Function:      Set the global music channels volume.

 Prototype:     void MODSetMusicVolume(Volume)

 Parameters:    Volume  - Volume (0-255)

 Returns:       None.

 Remarks:       This function will set the global volume for all the music
                channels.


 MODSetSampleVolume
 
 Function:      Set the global sample channels volume.

 Prototype:     void MODSetSampleVolume(Volume)

 Parameters:    Volume  - Volume (0-255)

 Returns:       None.

 Remarks:       This function will set the global volume for all the sample
                channels.


 MODLoadModule
 
 Function:      Load Module file from disk.

 Prototype:     Module *MODLoadModule(Path)

 Parameters:    Path    - Module file name.

 Returns:       The module structure address or NULL of error.

 Remarks:       This function load a 4,6 or 8 channels standard Protracker
                or Fastracker module music file from disk.


 MODFreeModule
 
 Function:      Free Module file from memory.

 Prototype:     void MODFreeModule(Song)

 Parameters:    Song    - Module file address.

 Returns:       None.

 Remarks:       This function frees the module music from memory. You cannot
                free a module while it is being played.


 MODLoadSample
 
 Function:      Load sample file from disk.

 Prototype:     Sample *MODLoadSample(Path)

 Parameters:    Path    - WAV file name.

 Returns:       The sample structure address or NULL of error.

 Remarks:       This function load a RIFF/WAV sample file, the player only
                supports 8-bit mono unsigned sample file formats.


 MODFreeSample
 
 Function:      Free Sample file from memory.

 Prototype:     void MODFreeSample(Instr)

 Parameters:    Instr   - Sample file address.

 Returns:       None.

 Remarks:       This function frees the sample file from memory. You cannot
                free a sample while it is being played.
\******************************************************************************/

#ifndef __MODPLAY_H
#define __MODPLAY_H

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long dword;
typedef byte *pointer;

typedef struct {
    word    NumTracks;
    word    OrderLength;
    byte    Orders[128];
    pointer Patterns[128];
    pointer SampPtr[32];
    pointer SampEnd[32];
    pointer SampLoop[32];
    byte    SampVolume[32];
} Module;

typedef struct {
    word    Period;
    word    Volume;
    dword   Length;
    pointer Data;
} Sample;

#pragma aux MODDetectCard "_*" parm caller [];
#pragma aux MODPlayModule "_*" parm caller [];
#pragma aux MODStopModule "_*" parm caller [];
#pragma aux MODPlaySample "_*" parm caller [];
#pragma aux MODStopSample "_*" parm caller [];
#pragma aux MODSetPeriod  "_*" parm caller [];
#pragma aux MODSetVolume  "_*" parm caller [];
#pragma aux MODSetMusicVolume  "_*" parm caller [];
#pragma aux MODSetSampleVolume "_*" parm caller [];

extern int  MODDetectCard(word *Port, byte *IRQ, byte *DRQ);
extern int  MODPlayModule(Module *Modulefile,word Chans,word Rate,word Port,byte IRQ,byte DRQ);
extern void MODStopModule(void);
extern void MODPlaySample(byte Voice,Sample *Instr);
extern void MODStopSample(byte Voice);
extern void MODSetPeriod(byte Voice,word Period);
extern void MODSetVolume(byte Voice,byte Volume);
extern void MODSetMusicVolume(byte Volume);
extern void MODSetSampleVolume(byte Volume);

extern Module *MODLoadModule(char *Path);
extern void MODFreeModule(Module *Song);
extern Sample *MODLoadSample(char *Path);
extern void MODFreeSample(Sample *Instr);

#endif

