{*      MODP.PAS
 *
 * ProTracker Module Player, v1.10
 *
 * Copyright 1994 Petteri Kangaslampi and Jarno Paananen
 *
 * This file is part of the MIDAS Sound System, and may only be
 * used, modified and distributed under the terms of the MIDAS
 * Sound System license, LICENSE.TXT. By continuing to use,
 * modify or distribute this file you indicate that you have
 * read the license and understand and accept it fully.
*}


unit MODP;


interface



{****************************************************************************\
*       struct modInstHdr
*       -----------------
* Description:  Protracker module instrument header. Note that all 16-bit
*               fields are big-endian.
\****************************************************************************}

type
    modInstHdr = Record
        iname : array[0..21] of char;   { instrument name }
        slength : word;                 { sample length }
        finetune : byte;                { sample finetune value }
        volume : byte;                  { sample default volume }
        loopStart : word;               { sample loop start, in words }
        loopLength : word;              { sample loop length, in words }
    end;




{****************************************************************************\
*       struct modHeader
*       ----------------
* Description:  Protracker module file header
\****************************************************************************}

type
    modHeader = Record
        songName : array[0..19] of char;    { song name }
        instruments : array[0..30] of modInstHdr;   { instrument headers }
        songLength : byte;              { song length }
        unused : byte;                  { unused by Protracker, used to be
                                          song restart position }
        orders : array[0..127] of byte; { pattern playing orders }
        sign : array[0..3] of byte;     { module signature }
    end;




{****************************************************************************\
*       struct modChannel
*       -----------------
* Description:  Protracker module player channel data structure
\****************************************************************************}

type
    modChannel = Record
        note : byte;
        inst : byte;
        cmd : byte;
        info : byte;
        comp : byte;

        sample : byte;
        volume : byte;

        period : word;
        snote : word;
        loff : byte;
        coff : byte;
        toperi : word;
        notepsp : byte;
        retrigc : byte;

        status : byte;

        vibcmd : byte;
        vibpos : byte;

        trecmd : byte;
        trepos : byte;

        volbar : byte;
        playoff : word;
    end;


    Pinteger = ^integer;
    Pword = ^word;
    Ppointer = ^pointer;




{****************************************************************************\
*
* Function:     modLoadModule(fileName : string; SD : pointer;
*                   module : Ppointer) : integer;
*
* Description:  Loads a Protracker module into memory
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to the Sound Device which will
*                                       store the samples
*               module : Ppointer       pointer to variable which will store
*                                       the module pointer.
*
* Returns:      MIDAS error code.
*               Pointer to module structure is stored in module^.
*
\****************************************************************************}

function modLoadModule(fileName : string; SD : pointer; module : Ppointer) :
    integer;



{****************************************************************************\
*
* Function:     modFreeModule(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates a Protracker module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      MIDAS error code
*
\****************************************************************************}

function modFreeModule(module : pointer; SD : pointer) : integer;




function modConvertSample(sample : pointer; length : word) : integer;
function modConvertTrack(track : pointer; ttype : word; trackLen : Pword)
    : integer;

function modIdentify(header : pointer; recognized : Pinteger) : integer;
function modInit(SD : pointer) : integer;
function modClose : integer;
function modPlayModule(module : pointer; firstSDChannel, numSDChannels,
    loopStart, loopEnd : word) : integer;
function modStopModule : integer;
function modSetInterrupt : integer;
function modRemoveInterrupt : integer;
function modPlay : integer;
function modSetPosition(pos : word) : integer;
function modGetInformation(info : pointer) : integer;


procedure mpMOD;                        { Protracker Module Player structure }



implementation


USES  Errors, mGlobals, mMem, EMS, MPlayer, SDevice, Timer
{$IFDEF REALVUMETERS}
    ,VU
{$ENDIF}
    ;



function modConvertSample(sample : pointer; length : word) : integer;
    external;
function modConvertTrack(track : pointer; ttype : word; trackLen : Pword)
    : integer; external;

function modIdentify(header : pointer; recognized : Pinteger) : integer;
    external;
function modInit(SD : pointer) : integer; external;
function modClose : integer; external;
function modPlayModule(module : pointer; firstSDChannel, numSDChannels,
    loopStart, loopEnd : word) : integer; external;
function modStopModule : integer; external;
function modSetInterrupt : integer; external;
function modRemoveInterrupt : integer; external;
function modPlay : integer; external;
function modSetPosition(pos : word) : integer; external;
function modGetInformation(info : pointer) : integer; external;
procedure mpMOD; external;
{$L MOD.OBJ}



{****************************************************************************\
*       enum modFunctIDs
*       ----------------
* Description:  ID numbers for Protracker Module Player functions
\****************************************************************************}

const
    ID_modIdentify = ID_mod;
    ID_modInit = ID_mod + 1;
    ID_modClose = ID_mod + 2;
    ID_modLoadModule = ID_mod + 3;
    ID_modFreeModule = ID_mod + 4;
    ID_modPlayModule = ID_mod + 5;
    ID_modStopModule = ID_mod + 6;
    ID_modSetInterrupt = ID_mod + 7;
    ID_modRemoveInterrupt = ID_mod + 8;
    ID_modPlay = ID_mod + 9;
    ID_modSetPosition = ID_mod + 10;
    ID_modGetInformation = ID_mod + 11;
    ID_modConvertSample = ID_mod + 12;
    ID_modConvertTrack = ID_mod + 13;





{$I-}

{* Size of temporary memory area used for avoiding memory fragmentation
   if EMS is used *}
const
    TEMPSIZE = 8192;

type
    wordArray = array[0..8192] of word;
    longArray = array[0..8192] of longint;



{****************************************************************************\
*       Module loader buffers and file pointer. These variables are static
*       instead of local so that a separate deallocation can be used which
*       will be called before exiting in error situations
\****************************************************************************}
var
    f : file;
    fileOpened : boolean;
    mmod : PmpModule;
    pattBuf : ^longArray;
    trackBuf : ^longArray;
    smpBuf : pointer;
    tempMem : pointer;



{****************************************************************************\
*
* Function:     modFreeModule(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates a Protracker module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      MIDAS error code
*
\****************************************************************************}

function modFreeModule(module : pointer; SD : pointer) : integer;
var
    i, error : integer;
    sdev : ^SoundDevice;
begin
    mmod := module;
    sdev := SD;

    if mmod = NIL then                 { valid module? }
    begin
        mError(errUndefined, ID_modFreeModule);
        modFreeModule := errUndefined;
        exit;
    end;

    { deallocate pattern orders if allocated: }
    if mmod^.orders <> NIL then
    begin
        error := memFree(mmod^.orders);
        if error <> OK then
        begin
            mError(error, ID_modFreeModule);
            modFreeModule := error;
            exit;
        end;
    end;

    { deallocate sample used flags: }
    if mmod^.instsUsed <> NIL then
    begin
        error := memFree(mmod^.instsUsed);
        if error <> OK then
        begin
            mError(error, ID_modFreeModule);
            modFreeModule := error;
            exit;
        end;
    end;


    if mmod^.insts <> NIL then       { instruments? }
    begin
        for i := 0 to (mmod^.numInsts-1) do
        begin
            { If the instrument has been added to Sound Device, remove
               it, otherwise just deallocate the sample if allocated }
            if mmod^.insts^[i].sdInstHandle <> 0 then
            begin
                error := sdev^.RemInstrument(mmod^.insts^[i].sdInstHandle);
                if error <> OK then
                begin
                    mError(error, ID_modFreeModule);
                    modFreeModule := error;
                    exit;
                end;
            end
            else
            begin
                if mmod^.insts^[i].sample <> NIL then
                begin
                    error := memFree(mmod^.insts^[i].sample);
                    if error <> OK then
                    begin
                        mError(error, ID_modFreeModule);
                        modFreeModule := error;
                        exit;
                    end;
                end;
            end;

{$IFDEF REALVUMETERS}
            { remove VU meter information if used: }
            if realVU = 1 then
            begin
                if mmod^.insts^[i].sdInstHandle <> 0 then
                begin
                    error := vuRemove(mmod^.insts^[i].sdInstHandle);
                    if error <> OK then
                    begin
                        mError(error, ID_modFreeModule);
                        modFreeModule := error;
                        exit;
                    end;
                end;
            end;
{$ENDIF}
        end;

        { deallocate instrument structures: }
        error := memFree(mmod^.insts);
        if error <> OK then
        begin
            mError(error, ID_modFreeModule);
            modFreeModule := error;
            exit;
        end;
    end;

    if (mmod^.patterns <> NIL) and (mmod^.pattEMS <> NIL) then
    begin
        for i := 0 to (mmod^.numPatts-1) do
        begin
            { if the pattern has been allocated, deallocate it - either
                from conventional memory or from EMS }
            if mmod^.patterns^[i] <> NIL then
            begin
                if mmod^.pattEMS^[i] = 1 then
                begin
                    error := emsFree(mmod^.patterns^[i]);
                    if error <> OK then
                    begin
                        mError(error, ID_modFreeModule);
                        modFreeModule := error;
                        exit;
                    end;
                end
                else
                begin
                    error := memFree(mmod^.patterns^[i]);
                    if error <> OK then
                    begin
                        mError(error, ID_modFreeModule);
                        modFreeModule := error;
                        exit;
                    end;
                end;
            end;
        end;

        { deallocate pattern pointers: }
        error := memFree(mmod^.patterns);
        if error <> OK then
        begin
            mError(error, ID_modFreeModule);
            modFreeModule := error;
            exit;
        end;

        { deallocate pattern EMS flags: }
        error := memFree(mmod^.pattEMS);
        if error <> OK then
        begin
            mError(error, ID_modFreeModule);
            modFreeModule := error;
            exit;
        end;
    end;

    { deallocate the module: }
    error := memFree(mmod);
    if error <> OK then
    begin
        mError(error, ID_modFreeModule);
        modFreeModule := error;
        exit;
    end;

    modFreeModule := OK;
end;




{***************************************************************************\
*
* Function:     modLoadError(SD : PSoundDevice)
*
* Description:  Stops loading the module, deallocates all buffers and closes
*               the file.
*
* Input:        SD : PSoundDevice       Sound Device that has been used for
*                                       loading.
*
\***************************************************************************}

procedure modLoadError(SD : PSoundDevice);
begin
    if fileOpened then
        Close(f);                       { close file if opened }

    { Attempt to deallocate module if allocated. Do not process errors. }
    if mmod <> NIL then
        if modFreeModule(mmod, SD) <> OK then
            exit;

    { Deallocate buffers if allocated. Do not process errors. }
    if pattBuf <> NIL then
        if memFree(pattBuf) <> OK then
            exit;
    if trackBuf <> NIL then
        if memFree(trackBuf) <> OK then
            exit;
    if smpBuf <> NIL then
        if memFree(smpBuf) <> OK then
            exit;
    if tempmem <> NIL then
        if memFree(tempmem) <> OK then
            exit;
end;




{****************************************************************************\
*
* Function:     modLoadModule(fileName : string; SD : pointer;
*                   module : Ppointer) : integer;
*
* Description:  Loads a Protracker module into memory
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to the Sound Device which will
*                                       store the samples
*               module : Ppointer       pointer to variable which will store
*                                       the module pointer.
*
* Returns:      MIDAS error code.
*               Pointer to module structure is stored in module^.
*
\****************************************************************************}

function modLoadModule(fileName : string; SD : pointer; module : Ppointer) :
    integer;

    { memcpy - copies bytes from src to dest }
    procedure memcpy(var dest; var src; bytes : word); assembler;
    asm
            push    ds
            les     di,dest
            lds     si,src
            mov     cx,bytes
            cld
            rep     movsb
            pop     ds
    end;


    { SWAP16 - endianness-swap for 16-bit words }
    function SWAP16(w : word) : word; assembler;
    asm
            mov     ax,w
            xchg    al,ah
    end;


    { cmpstr - compares memory area m against string s. Returns 0 if
      identical. }
    function cmpstr(var m1; s : string) : integer;
    var
        b : ^byteArray;
        pos : byte;
    begin
        b := @m1;
        cmpstr := 0;
        for pos := 1 to ord(s[0]) do
            if b^[pos-1] <> ord(s[pos]) then
                cmpStr := 1;
    end;
var
    error : integer;
    modh : modHeader;
    modi : ^modInstHdr;
    inst : PmpInstrument;
    pattData : PmpPattern;

    trackLen : word;

    i, c, r : integer;
    chans : word;
    numPatts : word;
    foffset : longint;

    slength, loopStart, loopLength : word;
    maxSmpLength : longint;
    instx : byte;

    p : pointer;

    numRead : word;

    sdev : PSoundDevice;


begin
    sdev := SD;

    { point file ptr and buffers to NIL so that modLoadError() can be
       called at any point: }
    fileOpened := False;
    mmod := NIL;
    pattBuf := NIL;
    trackBuf := NIL;
    smpBuf := NIL;
    tempmem := NIL;

    { Open module file: }
    Assign(f, fileName);
    Reset(f, 1);
    if IOResult <> 0 then
    begin
        mERROR(errFileOpen, ID_modLoadModule);
        modLoadError(SD);
        modLoadModule := errFileOpen;
        exit;
    end;

    { Allocate memory for the module structure: }
    error := memAlloc(SizeOf(mpModule), @mmod);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    mmod^.orders := NIL;                { clear module structure so that }
    mmod^.insts := NIL;                 { it can be deallocated with }
    mmod^.patterns := NIL;              { modFree() at any point }
    mmod^.pattEMS := NIL;
    mmod^.instsUsed := NIL;

    { read Protracker module header: }
    BlockRead(f, modh, SizeOf(modHeader), numRead);
    if numRead <> SizeOf(modHeader) then
    begin
        mERROR(errFileRead, ID_modLoadModule);
        modLoadError(SD);
        modLoadModule := errFileRead;
        exit;
    end;

    chans := 0;

    { Check the module signature to determine number of channels: }
    if cmpstr(modh.sign, 'M.K.') = 0 then chans := 4;
    if cmpstr(modh.sign, 'M!K!') = 0 then chans := 4;
    if cmpstr(modh.sign, 'FLT4') = 0 then chans := 4;
    if cmpstr(modh.sign, 'OCTA') = 0 then chans := 8;

    if cmpstr(modh.sign[1], 'CHN') = 0 then
        { xCHN, where x is the number of channels }
        chans := ord(modh.sign[0]) - ord('0');

    if cmpstr(modh.sign[2], 'CH') = 0 then
        { xxCHN, where xx is the number of channels }
        chans := (ord(modh.sign[0] - ord('0'))) * 10 +
                  ord(modh.sign[1]) - ord('0');

    if cmpstr(modh.sign[0], 'TDZ') = 0 then
        { TDZx, where x is the number of channels }
        chans := ord(modh.sign[3]) - ord('0');



    { If number of channels is undetermined, the signature is invalid. }
    if chans = 0 then
    begin
        mERROR(errInvalidModule, ID_modLoadModule);
        modLoadError(SD);
        modLoadModule := errInvalidModule;
        exit;
    end;

    mmod^.numChans := chans;            { store number of channels }

    memcpy(mmod^.songName, modh.songName, 20);      { copy song name }
    mmod^.songName[20] := chr(0);           { force terminating '\0' }
    mmod^.songLength := modh.songLength;    { copy song length }
    mmod^.numInsts := 31;                   { set number of instruments }

    memcpy(mmod^.ID, modh.sign, 4);         { copy module signature }
    mmod^.IDnum := idMOD;                   { Protracker module }


    { find number of patterns in file: }
    numPatts := 0;
    for i := 0 to 127 do                { search all song data }
        if modh.orders[i] >= numPatts then
            numPatts := modh.orders[i] + 1;

    mmod^.numPatts := numPatts * chans; { store number of tracks }

    { allocate memory for pattern orders: }
    error := memAlloc(mmod^.songLength, @mmod^.orders);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { copy pattern orders }
    memcpy(mmod^.orders^, modh.orders, mmod^.songLength);


    { allocate memory for pattern (actually track) pointers }
    error := memAlloc(4 * mmod^.numPatts, @mmod^.patterns);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { allocate memory for pattern EMS flags }
    error := memAlloc(mmod^.numPatts, @mmod^.pattEMS);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    for i:= 0 to (mmod^.numPatts-1) do  { point all unallocated patterns }
        mmod^.patterns^[i] := NIL;      { to NIL for safety }

    foffset := SizeOf(modHeader);       { point foffset to first pattern }

    { allocate memory for instrument used flags: }
    error := memAlloc(mmod^.numInsts, @mmod^.instsUsed);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { Mark all instruments unused }
    for i := 0 to (mmod^.numInsts-1) do
        mmod^.instsUsed^[i] := 0;

    { allocate memory for instrument structures: }
    error := memAlloc(mmod^.numInsts * SizeOf(mpInstrument), @mmod^.insts);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { clear all instruments and find maximum instrument length: }
    maxSmpLength := 0;
    for i := 0 to (mmod^.numInsts-1) do
    begin
        mmod^.insts^[i].sample := NIL;
        mmod^.insts^[i].sdInstHandle := 0;
        if maxSmpLength < (2 * SWAP16(modh.instruments[i].slength)) then
            maxSmpLength := 2 * SWAP16(modh.instruments[i].slength);
    end;

    { check that none of the instruments is too long: }
    if maxSmpLength > SMPMAX then
    begin
        mERROR(errInvalidInst, ID_modLoadModule);
        modLoadError(SD);
        modLoadModule := errInvalidInst;
        exit;
    end;

    { allocate memory for pattern loading buffer }
    error := memAlloc(256 * chans, @pattBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { allocate memory for track conversion buffer }
    error := memAlloc(256, @trackBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    { convert all patterns: }

    for i := 0 to (numPatts-1) do
    begin
        { seek to pattern beginning }
        Seek(f, foffset);
        if IOResult <> 0 then
        begin
            mERROR(errFileRead, ID_modLoadModule);
            modLoadError(SD);
            modLoadModule := errFileRead;
            exit;
        end;

        { read pattern data }
        BlockRead(f, pattBuf^, 256 * chans, numRead);
        if numRead <> (256 * chans) then
        begin
            mERROR(errFileRead, ID_modLoadModule);
            modLoadError(SD);
            modLoadModule := errFileRead;
            exit;
        end;

        { convert all tracks of the pattern }
        for c := 0 to (chans-1) do
        begin
            { copy track data to track buffer: }
            for r := 0 to 63 do
                trackBuf^[r] := pattBuf^[r * chans + c];

            { check used instruments }
            for r := 0 to 63 do
            begin
                instx := ((trackBuf^[r] and $10) or ((trackBuf^[r] shr 20)
                    and $0F));
                if (instx > 0) and (instx < 32) then
                    mmod^.instsUsed^[instx-1] := 1;
            end;

            { convert track to internal format: }
            error := modConvertTrack(trackBuf, 0, @trackLen);
            if error <> OK then
            begin
                modLoadError(SD);
                mERROR(error, ID_modLoadModule);
                modLoadModule := error;
                exit;
            end;

            if useEMS = 1 then          { is EMS memory used? }
            begin
                { try to allocate EMS memory for track }
                error := emsAlloc(trackLen, @p);
                if error <> OK then
                begin
                    { failed - if only EMS memory should be used, or the
                      error is other than out of EMS memory, pass the error
                      on }
                    if (forceEMS = 1) or (error <> errOutOfEMS) then
                    begin
                        modLoadError(SD);
                        mERROR(error, ID_modLoadModule);
                        modLoadModule := error;
                        exit;
                    end
                    else
                    begin
                        { track not in EMS }
                        mmod^.pattEMS^[i * chans + c] := 0;

                        { try to allocate conventional memory instead }
                        error := memAlloc(trackLen, @p);
                        if error <> OK then
                        begin
                            modLoadError(SD);
                            mERROR(error, ID_modLoadModule);
                            modLoadModule := error;
                            exit;
                        end;
                        pattData := p;
                    end;
                end
                else
                begin
                    { EMS memory allocated succesfully - track in EMS }
                    mmod^.pattEMS^[i * chans + c] := 1;

                    { map EMS block to conventional memory and point pattData
                      to the memory area: }
                    error := emsMap(p, @pattData);
                    if error <> OK then
                    begin
                        modLoadError(SD);
                        mERROR(error, ID_modLoadModule);
                        modLoadModule := error;
                        exit;
                    end;
                end;
            end
            else
            begin
                { EMS memory not in use - allocate conventional memory }
                mmod^.pattEMS^[i * chans + c] := 0;

                error := memAlloc(trackLen, @p);
                if error <> OK then
                begin
                    modLoadError(SD);
                    mERROR(error, ID_modLoadModule);
                    modLoadModule := error;
                    exit;
                end;

                pattData := p;
            end;

            mmod^.patterns^[i * chans + c] := p;

            { copy track data from buffer to the correct memory area }
            memcpy(pattData^, trackBuf^, trackLen);
        end;

        foffset := foffset + 256 * chans;   { point foffset to next pattern }
    end;

    { deallocate pattern loading buffers: }
    error := memFree(trackBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;
    trackBuf := NIL;

    error := memFree(pattBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;
    pattBuf := NIL;


    { If EMS is used, allocate TEMPSIZE bytes of memory before the sample
       buffer and deallocate it after allocating the sample buffer to
       minimize memory fragmentation }
    if useEMS = 1 then
    begin
        error := memAlloc(TEMPSIZE, @tempmem);
        if error <> OK then
        begin
            modLoadError(SD);
            mERROR(error, ID_modLoadModule);
            modLoadModule := error;
            exit;
        end;
    end;

    { allocate memory for sample loading buffer: }
    error := memAlloc(maxSmpLength, @smpBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;

    if useEMS = 1 then
    begin
        error := memFree(tempmem);
        if error <> OK then
        begin
            modLoadError(SD);
            mERROR(error, ID_modLoadModule);
            modLoadModule := error;
            exit;
        end;
        tempmem := NIL;
    end;


    { point file offset to start of samples }
    foffset := longint(256 * chans) * longint(numPatts) + SizeOf(modHeader);

    for i := 0 to (mmod^.numInsts-1) do
    begin
        inst := @mmod^.insts^[i];       { point inst to current instrument
                                          structure }
        modi := @modh.instruments[i];   { point modi to current Protracker
                                          module instrument }

        { Convert sample length, loop start and loop end. They are stored
          as big-endian words, and refer to number of words instead of
          bytes }
        slength := 2 * SWAP16(modi^.slength);
        loopStart := 2 * SWAP16(modi^.loopStart);
        loopLength := 2 * SWAP16(modi^.loopLength);

        memcpy(inst^.iname, modi^.iname, 22);           { copy inst name }
        inst^.iname[22] := chr(0);                  { force terminating '\0' }
        inst^.loopStart := loopStart;               { copy sample loop start }
        inst^.loopEnd := loopStart + loopLength;        { sample loop end }

        { If sample loop end is past byte 2, the sample is looping
          (Protracker uses loop start = 0, length = 2 for no loop,
          Fasttracker start = 0, end = 0 }
        if inst^.loopEnd > 2 then
        begin
            inst^.looping := 1;
            inst^.length := inst^.loopEnd;  { use loop end as sample length }
        end                                 { if looping to avoid loading }
        else                                { unnecessary sample data }
        begin
            inst^.looping := 0;
            inst^.loopEnd := 0;         { set loop end to 0 if no loop }
            inst^.length := slength;    { use sample length }
        end;

        inst^.volume := modi^.volume;       { copy default volume }
        inst^.finetune := modi^.finetune;   { copy finetune }

        if mmod^.instsUsed^[i] = 1 then     { if not used, don't load }
        begin
            if inst^.length <> 0 then       { is there a sample for this inst? }
            begin
                { seek to sample start position: }
                Seek(f, foffset);
                if IOResult <> 0 then
                begin
                    mERROR(errFileRead, ID_modLoadModule);
                    modLoadError(SD);
                    modLoadModule := errFileRead;
                    exit;
                end;

                { read sample to buffer: }
                BlockRead(f, smpBuf^, inst^.length, numRead);
                if numRead <> inst^.length then
                begin
                    mERROR(errFileRead, ID_modLoadModule);
                    modLoadError(SD);
                    modLoadModule := errFileRead;
                    exit;
                end;
            end;

            { Point inst^.sample to NIL, as the instrument is not available
              - only the Sound Device has it }
            inst^.sample := NIL;

            { convert sample from signed to unsigned: }
            error := modConvertSample(smpBuf, inst^.length);
            if error <> OK then
            begin
                modLoadError(SD);
                mERROR(error, ID_modLoadModule);
                modLoadModule := error;
                exit;
            end;

            { add the instrument to Sound Device: }
            error := sdev^.AddInstrument(smpBuf, smp8bit, inst^.length,
                inst^.loopStart, inst^.loopEnd, inst^.volume, inst^.looping,
                @inst^.sdInstHandle);
            if error <> OK then
            begin
                modLoadError(SD);
                mERROR(error, ID_modLoadModule);
                modLoadModule := error;
                exit;
            end;

{$IFDEF REALVUMETERS}
            { if real VU meters are used, prepare VU meter information
              for this instrument }
            if realVU = 1 then
            begin
                error := vuPrepare(inst^.sdInstHandle, smpBuf, inst^.length,
                    inst^.loopStart, inst^.loopEnd);
                if error <> OK then
                begin
                    modLoadError(SD);
                    mERROR(error, ID_modLoadModule);
                    modLoadModule := error;
                    exit;
                end;
            end;
{$ENDIF}
        end;
        foffset := foffset + slength;   { point foffset to next sample }
    end;

    { deallocate sample loading buffer: }
    error := memFree(smpBuf);
    if error <> OK then
    begin
        modLoadError(SD);
        mERROR(error, ID_modLoadModule);
        modLoadModule := error;
        exit;
    end;
    smpBuf := NIL;

    Close(f);
    fileOpened := False;

    module^ := mmod;                    { return module ptr in module^ }
    modLoadModule := OK;
    exit;
end;



{$I+}

END.
