(** Debug writer for programs. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

include Program

(** Returns the calltype as a string. *)
let string_of_calltype ct =
	match ct with
	| Calltype_Cdecl -> "cdecl"
	| Calltype_Cdecl_FP -> "cdecl-floating-point"
	| Calltype_Stdcall -> "stdcall"
	| Calltype_Stdcall_FP -> "stdcall-floating-point"

(** Returns a descriptive (very verbose) string from the program. *)
let string_of_prg prg =
	(* Generate a string writer. *)
	let final_str = ref "" in
	let cur_indent = ref "" in

	let write_no_indent fmt = Printf.ksprintf (fun str -> final_str := !final_str ^ str ^ "\n") fmt in
	let write fmt = Printf.ksprintf (fun str -> final_str := !final_str ^ !cur_indent ^ str ^ "\n") fmt in
	(* Writes a line with a code range. *)
	let write_range range fmt =
		let write_it str =
			let range_str = Pos_obj.range_desc range in
			let base_str = !cur_indent ^ str in
			let pad_str = String.make (max 1 (100 - (String.length base_str))) ' ' in
			write_no_indent "%s%s%s" base_str pad_str range_str
		in
		Printf.ksprintf write_it fmt
	in

	let indent () = cur_indent := "    " ^ !cur_indent in
	let undent () = cur_indent := (String.sub !cur_indent 0 ((String.length !cur_indent)-4)) in

	(* Write the settings. *)
	write "Heap size: %d" prg.prg_heap_size;
	write "";
	write "";

	(* Write the constants. *)
	write "Constants (count: %d):" (List.length prg.prg_constants);
	let write_constant idx (name,var) =
		write "Constant #%d, name: '%s'" (idx+1) name;
		indent ();
		let _ =
			match var with
			| ConstInt (v,r) -> write_range r "Integer: %ld (0x%lx)" v v
			| ConstFloat (v,r) -> write_range r "Float: %f" v
			| ConstRawArray (v,r) -> write_range r "Raw data: \"%s\"" (String.escaped v)
			| _ -> ()
		in
		undent ()
	in
	Util.list_iteri write_constant prg.prg_constants;
	write "";
	write "";


	(* Write the globals. *)
	write "Globals (count: %d):" (List.length prg.prg_globals);
	let write_global idx (name,var) =
		write "Global #%d, name: '%s'" (idx+1) name;
		indent ();
		let _ =
			match var with
			| Global_Empty r -> write_range r "Uninitialized"
			| Global_Int (v,r) -> write_range r "Integer: %ld (0x%lx)" v v
			| Global_Float (v,r) -> write_range r "Float: %f" v
			| Global_Raw_Array (v,r) -> write_range r "Raw data: \"%s\"" (String.escaped v)
			| Global_Uninitialized_Array (size,r) -> write_range r "Uninitialized array, size: %d" size
		in
		undent ()
	in
	Util.list_iteri write_global prg.prg_globals;
	write "";
	write "";

	(* Write the assembly functions. *)
	write "Assembly functions (count: %d):" (List.length prg.prg_asmfun_list);
	let write_asmfun idx (name,(params,code,attrs,range)) =
		let attr_str =
			String.concat ", " (List.map string_of_fun_attr attrs)
		in

		write_range range "Assembly function #%d, name: '%s', parameter count: %d, line count: %d, attributes: %s" (idx+1) name params (List.length code) attr_str;
		indent ();
		List.iter (write "%s") code;
		undent ();
		write ""
	in
	Util.list_iteri write_asmfun prg.prg_asmfun_list;
	write "";

	(* Write the normal functions. *)
	write "Functions (count: %d):" (List.length prg.prg_fun_list);
	let write_fun idx (name,func) =
		write_range func.func_range "Function #%d, name: '%s', parameter count: %d, local variable count: %d" (idx+1) name (List.length func.func_params) (List.length func.func_locals);
		indent ();
		Util.list_iteri (fun i n -> write "Parameter #%d, name '%s'" (i+1) n) func.func_params;
		Util.list_iteri (fun i n -> write "Local variable #%d, name '%s'" (i+1) n) func.func_locals;

		(* The statement/code printer. *)
		let rec write_stmt stmt =
			(* Writes a list of parameters. *)
			let write_params stmts =
				let write_param idx s =
					write "PARAM #%d:" (idx+1);
					indent ();
					write_stmt s;
					undent ()
				in
				Util.list_iteri write_param stmts;
			in

			(* Write the statement. *)
			match stmt with
			| Nop r -> write_range r "Nop"
			| Block (code,r) ->
				write_range r "BLOCK {";
				indent ();
				List.iter write_stmt code;
				undent ();
				write_range r "}"
			| ConstInt (v,r) -> write_range r "INT %ld (0x%lx)" v v
			| ConstFloat (v,r) -> write_range r "FLOAT %f" v
			| ConstRawArray (v,r) -> write_range r "STRING \"%s\"" (String.escaped v)
			| GetGlobal (v,r) -> write_range r "GET_GLOBAL '%s'" v
			| GetLocal (v,r) -> write_range r "GET_LOCAL '%s'" v
			| GetParam (v,r) -> write_range r "GET_PARAM '%s'" v
			| GetExtSymbol (v,r) -> write_range r "GET_EXTERNAL_SYMBOL '%s'" v
			| SetGlobal (v,s,r) ->
				write_range r "SET_GLOBAL '%s':" v;
				indent (); write_stmt s; undent ()
			| SetLocal (v,s,r) ->
				write_range r "SET_LOCAL '%s':" v;
				indent (); write_stmt s; undent ()
			| SetParam (v,s,r) ->
				write_range r "SET_PARAM '%s':" v;
				indent (); write_stmt s; undent ()
			| Call (func,stmts,r) ->
				write_range r "CALL '%s' {" func;
				write_params stmts;
				write_range r "}"
			| CallAsm (func,stmts,r) ->
				write_range r "ASMCALL '%s' {" func;
				write_params stmts;
				write_range r "}"
			| CallC (calltype,stmts,r) ->
				write_range r "CCALL '%s' {" (string_of_calltype calltype);
				write_params stmts;
				write_range r "}"
			| If (c,t,f,r) ->
				write_range r "IF {";
				write "COND:";
				indent (); write_stmt c; undent ();
				write "TRUE:";
				indent (); write_stmt t; undent ();
				write "FALSE:";
				indent (); write_stmt f; undent ();
				write_range r "}"
			| While (cond,code,r) ->
				write_range r "WHILE {";
				write "COND:";
				indent (); write_stmt cond; undent ();
				write "CODE:";
				indent (); write_stmt code; undent ();
				write_range r "}"
			| Break r -> write_range r "BREAK"
			| Continue r -> write_range r "CONTINUE"
			| Return (c,r) ->
				write_range r "RETURN {";
				indent (); write_stmt c; undent ();
				write_range r "}"
		in
		write "Code:";
		indent ();
		write_stmt func.func_code;
		undent ();

		undent ();
		write ""
	in
	Util.list_iteri write_fun prg.prg_fun_list;
	write "";

	(* Done! *)
	!final_str
