(** set of optimizers for the IMBC. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

include Imbc

(** Counts opcodes in the given program. Returns (opcode,count) assoc list. *)
let count_opcodes imbc count_all_variants =
	let test_op op1 op2 =
		if count_all_variants then
			op1 = op2
		else
			opcode_eq op1 op2
	in

	let count_fun count (_,func) =
		let count_opcode count opcode =
			try 	let (opcode,cnt) = List.find (fun (op,_) -> test_op op opcode) count in
				Util.assoc_replace (opcode,cnt + 1) count
			with Not_found -> (opcode,1) :: count
		in
		List.fold_left count_opcode count func.ifun_code
	in
	List.fold_left count_fun [] imbc.imbc_funs


(** Optimizes some common opcode parameters to their fixed equivelant (if called often enough). *)
let optimize_common_opcode_params imbc =
	(* Count the number each opcode (full with parameters) is used. *)
	let opcode_count = count_opcodes imbc true in

	(* Sort the counts to descending order. *)
	let opcode_count = Sort.list (fun (_,c1) (_,c2) -> c1 >= c2) opcode_count in

	(* Process each opcode and test if it could be optimized by making it fixed. *)
	let add_fixed_opcode imbc old_opcode new_opcode =
		let process_opcode op =
			if op = old_opcode then new_opcode else op
		in

		let process_fun (n,f) =
			(n,{f with ifun_code = List.map process_opcode f.ifun_code })
		in

		let new_funs = List.map process_fun imbc.imbc_funs in
		{ imbc with
			imbc_opcodes = new_opcode :: imbc.imbc_opcodes;
			imbc_funs = new_funs }
	in

	let process_opcode imbc (opcode,count) =
		match opcode with
		| _ when ((List.length imbc.imbc_opcodes) >= 0xff) ->
			(* Opcode list full. *)
			imbc
		| _ when count < 10 ->
			imbc
		| PushLocal v -> add_fixed_opcode imbc opcode (FixedPushLocal v)
		| PushPresetGlobal v -> add_fixed_opcode imbc opcode (FixedPushPresetGlobal v)
		| PushUninitGlobal v -> add_fixed_opcode imbc opcode (FixedPushUninitGlobal v)
		| PushConstant v -> add_fixed_opcode imbc opcode (FixedPushConstant v)
		| PushConstantByte v -> add_fixed_opcode imbc opcode (FixedPushConstantByte v)
		| StoreLocal v -> add_fixed_opcode imbc opcode (FixedStoreLocal v)
		| StorePresetGlobal v -> add_fixed_opcode imbc opcode (FixedStorePresetGlobal v)
		| StoreUninitGlobal v -> add_fixed_opcode imbc opcode (FixedStoreUninitGlobal v)
		| Call v -> add_fixed_opcode imbc opcode (FixedCall v)
		| _ -> imbc
	in
	List.fold_left process_opcode imbc opcode_count

(** Sorts the constants to rising order. *)
let sort_constants imbc =
	(* Get the constant list that has the original index with it. *)
	let indexed_constants =
		Util.list_mapi (fun idx c -> (c,idx)) imbc.imbc_constants
	in

	(* Sort the constants. *)
	let sort_pred (c1,_) (c2,_) =
		let get_int32 v =
			match v with
			| Val_Int x -> x
			| Val_Float x -> Int32.of_float x
			| _ -> failwith "Cannot convert type to integer."
		in

		match (c1,c2) with
		| (Val_External s1,Val_External s2) -> String.compare s1 s2
		| (Val_External _,_) -> 1
		| (_,Val_External _) -> -1
		| (Val_String_Const v1,Val_String_Const v2) -> compare v1 v2
		| (Val_String_Const _,_) -> 1
		| (_,Val_String_Const _) -> -1
		| (v1,v2) -> Int32.compare (get_int32 v1) (get_int32 v2)
	in
	let indexed_constants =
		List.sort sort_pred indexed_constants
	in

	let new_constants =
		List.map (fun (c,_) -> c) indexed_constants
	in

	(* Remap the constants in all opcodes. *)
	let map_opcode c =
		match c with
		| PushConstant v ->
			let new_idx = Util.list_find_idx (fun (_,old_idx) -> old_idx = v ) indexed_constants in
			PushConstant new_idx
		| FixedPushConstant v ->
			let new_idx = Util.list_find_idx (fun (_,old_idx) -> old_idx = v ) indexed_constants in
			FixedPushConstant new_idx
		| x -> x
	in

	let remap_fun (name,f) =
		let new_code = List.map map_opcode f.ifun_code in

		(name,{ f with ifun_code = new_code })
	in

	{ imbc with
	  imbc_opcodes = List.map map_opcode imbc.imbc_opcodes;
	  imbc_constants = new_constants;
	  imbc_funs = List.map remap_fun imbc.imbc_funs }

(** Sorts the opcodes to lowering order (by use). *)
let sort_opcodes imbc =
	(* Count the number each opcode (full with parameters) is used. *)
	let opcode_count = count_opcodes imbc false in

	(* Finds the sort value for an opcode. *)
	let find_sort_value opcode =
		let (_,cnt) = List.find (fun (op,_) -> opcode_eq op opcode) opcode_count in
		cnt
	in

	(* Sort the opcodes. *)
	let sort_pred op1 op2 =
		(find_sort_value op1) >= (find_sort_value op2)
	in
	let new_opcodes = Sort.list sort_pred imbc.imbc_opcodes in

(* List.iter (fun op -> Printf.printf "%d %s\n" (find_sort_value op) (string_of_opcode op)) new_opcodes; *)

	{ imbc with
	  imbc_opcodes = new_opcodes }
