(** Parses an il4 file. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type item =
	| Atom of string * Pos_obj.range
	| Int of int32 * Pos_obj.range
	| Float of float * Pos_obj.range
	| Str of string * Pos_obj.range
	| Items of item_list * Pos_obj.range
and item_list = item list

(** Skips any white space (including comments). *)
let rec skip_ws stm =
	let rec skip_comment () =
		match Pos_obj.obj (Char_reader.peek stm) with
		| '\r' | '\n' -> ()
		| _ -> Char_reader.skip stm; skip_comment ()
	in

	match Pos_obj.obj (Char_reader.peek stm) with
	| ' ' | '\t' | '\r' | '\n' -> Char_reader.skip stm; skip_ws stm
	| '#' -> skip_comment (); skip_ws stm
	| _ -> ()

(** Deducts the atom type from the atom string. *)
let deduct_atom_from_string str range =
	try Int (Int32.of_string str, range)
	with _ ->
		try Float (float_of_string str, range)
		with _ -> Atom ((String.lowercase str), range)
	
(** Parses an atom. *)
let parse_atom stm =
	let start_pos = Pos_obj.pos (Char_reader.peek stm) in

	try
		let rec read acc end_pos =
			let cur_ch = Char_reader.peek stm in

			match Pos_obj.obj cur_ch with
			| '(' | ')' | '"' | '\'' | '\n' | '\r' | '\t' | ' ' -> (acc, end_pos)
			| c ->
				Char_reader.skip stm;
				read (c :: acc) (Pos_obj.pos cur_ch)
		in

		let rev_str, end_pos = read [] start_pos in
		let str = Util.string_of_chars (List.rev rev_str) in

		deduct_atom_from_string str (Pos_obj.make_range start_pos end_pos)
	with End_of_file -> failwith ("Premature end of file while parsing an atom constant starting at " ^ (Pos_obj.pos_desc start_pos))

type string_read_mode =
	| Mode_Normal
	| Mode_Escaped
	| Mode_HexModeStart
	| Mode_HexModeHalf of char

(** Parses a string constant. *)
let parse_string stm =
	let hex_chars = "0123456789abcdefABCDEF" in

	(* Read the start char *)
	let start_ch_obj = Char_reader.read stm in
	let start_ch = Pos_obj.obj start_ch_obj in
	let start_pos = Pos_obj.pos start_ch_obj in

	try
		let rec read acc mode =
			let cur_ch = Char_reader.read stm in

			match (Pos_obj.obj cur_ch,mode) with
			| ('\n',_) | ('\r',_) -> failwith ("String hit end of line; string started at " ^ (Pos_obj.pos_desc start_pos))

			| ('x',Mode_Escaped) -> read acc Mode_HexModeStart
			| (c,Mode_Escaped) -> read (c :: acc) Mode_Normal

			| (c,Mode_HexModeStart) -> read acc (Mode_HexModeHalf c)
			| (c2,Mode_HexModeHalf c1) ->
				let hex_str = String.create 2 in
				hex_str.[0] <- c1;
				hex_str.[1] <- c2;
				let c =
					try
						ignore (String.index hex_chars c1);
						ignore (String.index hex_chars c2);
						Scanf.sscanf hex_str "%x" char_of_int
					with _ -> failwith ("Invalid hex constant inside the string (0x" ^ hex_str ^ "); string started at " ^ (Pos_obj.pos_desc start_pos))
				in
				read (c :: acc) Mode_Normal

			| ('\\',Mode_Normal) -> read acc Mode_Escaped
			| (c,Mode_Normal) when c = start_ch -> acc, (Pos_obj.pos cur_ch)
			| (c,Mode_Normal) -> read (c :: acc) Mode_Normal
		in

		let rev_str, end_pos = read [] Mode_Normal in
		let str = Util.string_of_chars (List.rev rev_str) in
		Str (str, (Pos_obj.make_range start_pos end_pos))
	with End_of_file -> failwith ("Premature end of file while parsing a string constant starting at " ^ (Pos_obj.pos_desc start_pos))

(** Parses an item list, form: (item item item (item item ())). *)
let rec parse_list stm =
	skip_ws stm;
	let start_ch = Char_reader.read stm in
	let start_pos = Pos_obj.pos start_ch in

	(* Reads up until the list ends. *)
	let rec read_list acc =
		skip_ws stm;
		let cur_ch = Char_reader.peek stm in

		match Pos_obj.obj cur_ch with
		| '(' -> read_list ((parse_list stm) :: acc)
		| ')' -> Char_reader.skip stm; (List.rev acc, Pos_obj.pos cur_ch)
		| '"' -> read_list ((parse_string stm) :: acc)
		| '\'' -> read_list ((parse_string stm) :: acc)
		| _ -> read_list ((parse_atom stm) :: acc)
	in

	(* Start reading the list. *)
	try
		match Pos_obj.obj start_ch with
		| '(' ->
			let items, end_pos = read_list [] in
			Items (items, (Pos_obj.make_range start_pos end_pos))
		| _ -> failwith ("Expecting a list but instead got '" ^ (Util.string_of_char (Pos_obj.obj start_ch)) ^ " at " ^ (Pos_obj.pos_desc start_pos))
	with End_of_file -> failwith ("Premature end of file while parsing the list starting at " ^ (Pos_obj.pos_desc start_pos))

(** Parses the opened Char_reader stream. *)
let parse_char_stream stm =
	let prg = ref [] in

	let rec do_parse () =
		prg := (parse_list stm) :: !prg;
		do_parse ()
	in

	(try do_parse ()
	with End_of_file -> ());

	List.rev !prg

(** Parses an IL4 file. *)
let parse fn =
	let fin = Char_reader.make fn in

	let do_run () = parse_char_stream fin in
	let finalize () = Char_reader.close fin in

	Util.finally do_run finalize


(** Returns the item range. *)
let range_of_item item =
	match item with
	| Atom (_,r) | Int (_,r) | Float (_,r) | Str (_,r) | Items (_,r) -> r

(** Returns the item as an atom. Throws an exception on error. *)
let atom_of_item item =
	match item with
	| Atom (x,_) -> x
	| Int (_,r) | Float (_,r) | Str (_,r) | Items (_,r) -> failwith ("Expected an atom at " ^ (Pos_obj.range_desc r))
	
(** Returns the item as an int. Throws an exception on error. *)
let int_of_item item =
	match item with
	| Int (x,_) -> x
	| Atom (_,r) | Float (_,r) | Str (_,r) | Items (_,r) -> failwith ("Expected an integer value at " ^ (Pos_obj.range_desc r))

(** Returns the item as a float. Throws an exception on error. *)
let float_of_item item =
	match item with
	| Float (x,_) -> x
	| Atom (_,r) | Int (_,r) | Str (_,r) | Items (_,r) -> failwith ("Expected a floating point value at " ^ (Pos_obj.range_desc r))

(** Returns the item as a string. Throws an exception on error. *)
let string_of_item item =
	match item with
	| Str (x,_) -> x
	| Atom (_,r) | Int (_,r) | Float (_,r) | Items (_,r) -> failwith ("Expected a string value at " ^ (Pos_obj.range_desc r))

(** Returns the item as a list of items. Throws an exception on error. *)
let items_of_item item =
	match item with
	| Items (x,_) -> x
	| Atom (_,r) | Int (_,r) | Float (_,r) | Str (_,r) -> failwith ("Expected a list at " ^ (Pos_obj.range_desc r))
