(** Single character reader. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(** The data buffer type. *)
type buf =
{
	buf_data : string;
	mutable buf_pos : int;
	mutable buf_size : int;
}

(** Creates a new read buffer. *)
let new_buf size =
	{
		buf_data = String.create size;
		buf_pos = 0;
		buf_size = 0;
	}

(** Reads from the read buffer. Fills from the channel if empty. *)
let read_buf buf chn =
	if buf.buf_pos >= buf.buf_size then
		begin
		buf.buf_pos <- 0;
		buf.buf_size <- input chn buf.buf_data 0 (String.length buf.buf_data);
		if buf.buf_size <= 0 then raise End_of_file
		end;

	(* Return a character. *)
	let ret = buf.buf_data.[buf.buf_pos] in
	buf.buf_pos <- buf.buf_pos + 1;
	ret
	

(** The reader type. *)
type t =
{
	file_name: string;
	mutable file_line: int;
	mutable file_col: int;
	mutable file_chn: in_channel option;
	file_buf : buf;
	mutable file_peeked: char Pos_obj.t option;
}

(** Creates a new character reader. *)
let make fn =
	{
		file_name = fn;
		file_line = 1;
		file_col = 1;
		file_chn = Some (open_in fn);
		file_buf = new_buf 4096;
		file_peeked = None;
	}

(** Reads a new character. Throws End_of_file when done. *)
let read f =
	let chn =
		match f.file_chn with
		| Some c -> c
		| _ -> raise End_of_file
	in

	let read_new () =
		let ch = read_buf f.file_buf chn in
		let ret = Pos_obj.make ch f.file_name f.file_line f.file_col in
	
		(* Advance the file position. *)
		(match ch with
		| '\n' | '\r' -> f.file_line <- f.file_line + 1; f.file_col <- 1
		| '\t' -> f.file_col <- f.file_col + 8
		| _ -> f.file_col <- f.file_col + 1);
	
		ret
	in

	(* Do we have one in the peek buffer? *)
	match f.file_peeked with
	| Some ret -> f.file_peeked <- None; ret
	| None -> read_new ()

(** Peeks the next character. Throws End_of_file if peek fails. *)
let peek f =
	let ret = read f in
	f.file_peeked <- Some ret;
	ret

(** Skips the next character. Throws End_of_file when done. *)
let skip f =
	ignore (read f)

(** Closes a character reader. *)
let close f =
	match f.file_chn with
	| Some chn -> f.file_chn <- None; close_in chn
	| _ -> ()
