//
//
//

#ifndef __IMPORTABLEI_H__
#define __IMPORTABLEI_H__

namespace Import {
	class ImportableI;
};

#include "PajaTypes.h"
#include "DataBlockI.h"
#include "EditableI.h"
#include "UndoC.h"
#include "ClassIdC.h"
#include "ImportInterfaceC.h"
#include "FileHandleC.h"
#include "TimeContextC.h"



namespace PluginClass {
	//! The importable super class ID.
	/*! Every importer should return SUPERCLASS_IMPORT as super class if it only
		implements the ImportableI interface. */
	const PluginClass::SuperClassIdC		SUPERCLASS_IMPORT = PluginClass::SuperClassIdC( 0x2000000 );
};

namespace Import {

	//! Importable file interface.
	/*!	Importable files are key elements in the Demopaja system. Importers
		imports and holds the external data in the system, and effects are used to display
		it. This allows data sharing between the effects.
		All the importers are derived from this base class.
	*/
	class ImportableI : public Edit::EditableI
	{
	public:
		//! Returns the name of the file this importable refers to.
		virtual const char*					get_filename() = 0;
		//! Loads the file specified in by the argument.
		virtual bool						load_file( const char* szName, ImportInterfaceC* pInterface ) = 0;

		//! Returns the number of referenced files.
		/*!	If the importable has references to other files (such as textures
			in 3D meshes) this method returns number of file references.

			File loading is implemented in the ImportInterfaceC which is passed to the
			load_file() method.

			The default implementation returns zero.
		*/
		virtual PajaTypes::uint32			get_reference_file_count();

		//! Returns file reference at speficied index.
		/*!	The default implementation returns zero. */
		virtual FileHandleC*				get_reference_file( PajaTypes::uint32 ui32Index );

		//! Returns super class ID of the importer.
		virtual PluginClass::SuperClassIdC	get_super_class_id() = 0;
		//! Returns unique class ID of the importer.
		virtual PluginClass::ClassIdC		get_class_id() = 0;
		//! Returns importer's class name as NULL terminated string.
		virtual const char*					get_class_name() = 0;

		//! Return short info about the file.
		/*! The information can be for example width and height and bit depth, etc.
			This information is displayed in the File Inspector window.
		*/
		virtual const char*					get_info() = 0;

		//! Returns duration of the file in given time context.
		/*! If the duration cannot be determined (for example the data is a still image)
			negative value should be returned.

			Example:
			\code
			int32
			MASImportC::get_duration( TimeContextC* pTimeContext )
			{
				int32	i32Length = m_i32LastFrame - m_i32FirstFrame;
				// Convert the FPS timecode to Demopaja timecode.
				return pTimeContext->convert_fps_to_time( i32Length, m_i32FPS );
			}
			\endcode
		*/
		virtual PajaTypes::int32			get_duration( PajaSystem::TimeContextC* pTimeContext ) = 0;

		//! Returns start label (a numerical value).
		/*! The value does not have to be related to the returned duration.
			The start label is shown in the duration bar in the timegraph.
			The Demopaja system will find nice values for the labels between the
			start and end label returned from the importable.

			Example:
			\code
			float32
			MASImportC::get_start_label( TimeContextC* pTimeContext )
			{
				return m_i32FirstFrame;
			}
			\endcode
		*/
		virtual PajaTypes::float32			get_start_label( PajaSystem::TimeContextC* pTimeContext ) = 0;
		
		//! Returns end label (a numerical value).
		/*! The value does not have to be related to the returned duration.
			The start label is shown in the duration bar in the timegraph.
			The Demopaja system will find nice values for the labels between the
			start and end label returned from the importable.

			Example:
			\code
			float32
			MASImportC::get_end_label( TimeContextC* pTimeContext )
			{
				return m_i32LastFrame;
			}
			\endcode
		*/
		virtual PajaTypes::float32			get_end_label( PajaSystem::TimeContextC* pTimeContext ) = 0;

		//! Returns the default effect class ID this importable creates if dropped to a File List (or to the Layout View).
		virtual PluginClass::ClassIdC		get_default_effect() = 0;

	protected:
		//! Default constructor.
		ImportableI();
		//! Constructor with reference to the original.
		ImportableI( Edit::EditableI* pOriginal );
		//! Default destructor.
		virtual ~ImportableI();

	};

};

#endif