/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes

// extern includes
#include <caosGL/gfx/cState.h>
#include <caosGL/gfx/cTextureSpace.h>
#include <caosGL/core/cParser.h>
#include <caosGL/core/DynGL.h>
#include <caosGL/core/math.h>
#include <caosGL/gfx/caos3dtypes.h>

#include <caosGL/gfx/cMesh.h>

using namespace caosGL::gfx;

namespace caosGL {
	namespace gfx {
		/**
		 *<br> class:		cMesh
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	caosGL::gfx::cBaseNode
		 *<br> implements:	caosGL::gfx::iDrawable
		 *<br> purpose:		Implements static mesh funcionality, as in vertecies, faces, edges, etc...
		 *
		 */

		static vector <tAttr> attributeNames;
		static log4cpp::Category& cat = log4cpp::Category::getInstance ("caosGL::gfx::cMesh");

		/********************************************************************************************/
		cMesh::cMesh (const string & n, cGroup * f) : super (n,f) {
			init ();
		}

		/********************************************************************************************/
		cMesh::~cMesh () {
		}

		/********************************************************************************************/
		tVoid cMesh::addVertexList (tVector * vList, tDWord num) {
			for (int i = 0 ; i < num ; i++) {
				vertex.push_back (vList[i]);
			}
		}

		/********************************************************************************************/
		tVoid cMesh::addFaceList (tFace * fList, tDWord num) {
			for (int i = 0 ; i < num ; i++) {
				face.push_back (fList[i]);
			}
		}
		
		/********************************************************************************************/
		tVoid cMesh::addUVList (tUV * uvList, tDWord num) {
			for (int i = 0 ; i < num ; i++) {
				uv.push_back (uvList[i]);
				if (uvList[i].vertex == 0) {
					cFace & f = face [uvList[i].face];
					f.uv_a = i;
				} else if (uvList[i].vertex == 1) {
					cFace & f = face [uvList[i].face];
					f.uv_b = i;
				} else if (uvList[i].vertex == 2) {
					cFace & f = face [uvList[i].face];
					f.uv_c = i;
				}
			}
		}
		
		/********************************************************************************************/
		tVoid cMesh::addNormalList (tNormal * nList, tDWord num) {
			for (int i = 0 ; i < num ; i++) {
				normal.push_back (nList[i]);
				if (nList[i].vertex == 0) {
					cFace & f = face [nList[i].face];
					f.n_a = i;
				} else if (nList[i].vertex == 1) {
					cFace & f = face [nList[i].face];
					f.n_b = i;
				} else if (nList[i].vertex == 2) {
					cFace & f = face [nList[i].face];
					f.n_c = i;
				}
			}
		}
		
		/********************************************************************************************/
		tVoid cMesh::addEdgeList (tEdge * eList, tDWord num) {
			for (int i = 0 ; i < num ; i++) {
				edge.push_back (eList[i]);
			}
		}
		
		/********************************************************************************************/
		tVoid cMesh::dump () {
			{cat.info ("Vertices:"); tInt i = 0;
			for (vector <cVector>::const_iterator it = vertex.begin(); it != vertex.end (); it++) {
				cat.info ("%5d: %f,%f,%f,%f", i++, it->x, it->y, it->z, it->w);
			}}
			{cat.info ("Faces:"); tInt i = 0;
			for (vector <cFace>::const_iterator it = face.begin(); it != face.end (); it++) {
				cat.info ("%5d: %d,%d,%d,%d,%d,%d,%d,%d,%d", i++, it->a, it->b, it->c, it->uv_a, it->uv_b, it->uv_c, 
					it->n_a, it->n_b, it->n_c);
			}}
			{cat.info ("UV:"); tInt i = 0;
			for (vector <cUV>::const_iterator it = uv.begin(); it != uv.end (); it++) {
				cat.info ("%5d: %f,%f,%d,%d", i++, it->u, it->v, it->face, it->vertex);
			}}
			{cat.info ("Normals:"); tInt i = 0;
			for (vector <cNormal>::const_iterator it = normal.begin(); it != normal.end (); it++) {
				cat.info ("%5d: %f,%f,%f,%d,%d", i++, it->v.x, it->v.y, it->v.z, it->face, it->vertex);
			}}
			{cat.info ("Edges:"); tInt i = 0;
			for (vector <cEdge>::const_iterator it = edge.begin(); it != edge.end (); it++) {
				cat.info ("%5d: %d,%d", i++, it->a, it->b);
			}}
		}

		/********************************************************************************************/
		tVoid cMesh::doNormalObj () {
			if (glIsList (_list))
				glDeleteLists (_list, 1);
			_list = glGenLists (1);
			glNewList (_list, GL_COMPILE);
				glBegin (GL_TRIANGLES); tInt i = 0;
				for (vector<cFace>::const_iterator it = face.begin(); it != face.end(); it++) {
					const cFace & f = *it;
					glTexCoord2f (uv [f.uv_a].u, uv [f.uv_a].v);
					glNormal3f (normal [f.n_a].v.x, normal [f.n_a].v.y, normal [f.n_a].v.z);
					glVertex3f (vertex [f.a].x,vertex [f.a].y,vertex [f.a].z);
					glTexCoord2f (uv [f.uv_b].u, uv [f.uv_b].v);
					glNormal3f (normal [f.n_b].v.x, normal [f.n_b].v.y, normal [f.n_b].v.z);
					glVertex3f (vertex [f.b].x,vertex [f.b].y,vertex [f.b].z);
					glTexCoord2f (uv [f.uv_c].u, uv [f.uv_c].v);
					glNormal3f (normal [f.n_c].v.x, normal [f.n_c].v.y, normal [f.n_c].v.z);
					glVertex3f (vertex [f.c].x,vertex [f.c].y,vertex [f.c].z);
					i++;
				}
				glEnd ();
			glEndList ();
				
		}

		/********************************************************************************************/
		tVoid cMesh::doEvnObj () {
			if (glIsList (_list))
				glDeleteLists (_list, 1);
			_list = glGenLists (1);
			glNewList (_list, GL_COMPILE);
				glBegin (GL_TRIANGLES); tInt i = 0;
				for (vector<cFace>::const_iterator it = face.begin(); it != face.end(); it++) {
					const cFace & f = *it;
					tFloat u,v;
					normal [f.n_a].v.angles(u,v);u=rad2deg(u)/360.0;v=rad2deg(v)/360.0;
					glTexCoord2f (u, v);
					glNormal3f (normal [f.n_a].v.x, normal [f.n_a].v.y, normal [f.n_a].v.z);
					glVertex3f (vertex [f.a].x,vertex [f.a].y,vertex [f.a].z);
					normal [f.n_b].v.angles(u,v);u=rad2deg(u)/360.0;v=rad2deg(v)/360.0;
					glTexCoord2f (u, v);
					glNormal3f (normal [f.n_b].v.x, normal [f.n_b].v.y, normal [f.n_b].v.z);
					glVertex3f (vertex [f.b].x,vertex [f.b].y,vertex [f.b].z);
					normal [f.n_c].v.angles(u,v);u=rad2deg(u)/360.0;v=rad2deg(v)/360.0;
					glTexCoord2f (u, v);
					glNormal3f (normal [f.n_c].v.x, normal [f.n_c].v.y, normal [f.n_c].v.z);
					glVertex3f (vertex [f.c].x,vertex [f.c].y,vertex [f.c].z);
					i++;
				}
				glEnd ();
			glEndList ();
		}
		
		// from iDrawable
		/********************************************************************************************/
		tBool cMesh::draw (const tFloat time) {
			if (_trsp) {
				cState::enable(cState::oGL_BLEND);
				glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			} else {
				cState::disable(cState::oGL_BLEND);
			}
			if (_texture) {
				cState::enable(cState::oGL_TEXTURE_2D);
				_texture->prepare();
			} else {
				cState::disable(cState::oGL_TEXTURE_2D);
			}
			glColor4f (_colR, _colG, _colB, _colA);
			glCallList (_list);
			return true;
		}

		// from iDrawable
		/********************************************************************************************/
		tBool cMesh::transparent () const {
			return _trsp;
		}

		// from iDrawable
		/********************************************************************************************/
		tVoid cMesh::transparent (const tBool trasnp) {
			_trsp = trasnp;
		}
		
		// from cBaseNode
		/********************************************************************************************/
		tBool cMesh::compile () {
			if (_tex!="")
				_texture = cTextureSpace::instance ()->get (_tex);
//				dump ();
//				if (_env) {
//					doEvnObj ();
//				} else {
				doNormalObj ();
//				}
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cMesh::visit (tFloat t) {
			const tBool ret = evaluateAll (t);
//				cat.info ("%s::%s alpha: %f", getTypeName().c_str(), _name.c_str(), _colA);
			if (!dSMALL(1.0-_colA)) {
				_trsp = true;
			}
			return ret;
		}

		// from cBaseNode
		/********************************************************************************************/
		tVoid cMesh::leave () {
			return;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cMesh::init () {
			#define ATTRIB(n,t,v,d) ATTRIB_INIT_VAR(n,v)
			#include cMesh_attribsFile
			#undef ATTRIB
			_list = -1;
			_texture = cNULL;
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cMesh::set (const tDWord key, const string & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_S(n)
				#include cMesh_attribsFile
				#undef ATTRIB
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cMesh::set (const tDWord key, const tFloat & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_N(n)
				#include cMesh_attribsFile
				#undef ATTRIB
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cMesh::get (const tDWord key, string & value) const {
			if (super::get (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_GET(n)
				#include cMesh_attribsFile
				#undef ATTRIB
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const vector <tAttr> * cMesh::getAttributeNames () const {
			if (attributeNames.size () == 0) {
				const vector <tAttr> * v = super::getAttributeNames ();
				attributeNames.insert (attributeNames.begin (), v->begin (), v->end ());
				#define ATTRIB(n,t,v,d) ATTRIB_ATTRIBNAMES(n,d)
				#include cMesh_attribsFile
				#undef ATTRIB
			}
			return &attributeNames;
		}

		// from cBaseNode
		/********************************************************************************************/
		const string cMesh::getTypeName () const { return "caosGL::gfx::cMesh"; }
	}
}

// for node creation
#include <caosGL/core/cRegistry.h>
#include <caosGL/gfx/cNodeCreator.h>

class cMeshNodeCreator : public cNodeCreator {
public:
	cMeshNodeCreator () {
		name ("caosGL::gfx::cMesh");
	}
	cBaseNode * createNode (const string n, cGroup * f) {
		return new caosGL::gfx::cMesh (n,f);
	}
};
caosGL::core::cRegisterNodeCreator <cMeshNodeCreator> cMeshNodeCreatorInstance;

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof