/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/math.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/gfx/cKey.h>

#include <caosGL/core/cUtil.h>

using namespace caosGL::core;

// extern includes
//#include <iostream>
#include <string.h>
#include <math.h>

#include <caosGL/gfx/cAnimCurve.h>

namespace caosGL {
	namespace gfx {
		/**
		 *<br> class:		cAnimCurve
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	caosGL::gfx::cEvaluatable
		 *<br> implements:	<none>
		 *<br> purpose:		An animation curve. You feed it time, it gives you a value. It works best
		 *                  for forward animations.
		 *
		 */

		static vector <tAttr> attributeNames;
		static tFloat sMachineTolerance;
		typedef struct ag_polynomial {
			tFloat *p;
			tInt deg;
		} AG_POLYNOMIAL;
		
		tInt polyZeroes (tFloat Poly[], tInt deg, tFloat a, tInt a_closed, tFloat b, tInt b_closed, tFloat Roots[]);
		tFloat ag_zeroin (tFloat a, tFloat b, tFloat tol, AG_POLYNOMIAL *pars);
		tFloat ag_zeroin2 (tFloat a, tFloat b, tFloat fa, tFloat fb, tFloat tol, AG_POLYNOMIAL *pars);
		tFloat ag_horner1 (tFloat P[], tInt deg, tFloat s);
		tVoid bezierToPower (tFloat a1, tFloat b1, tFloat c1, tFloat d1, tFloat *a2, tFloat *b2, tFloat *c2, tFloat *d2);
		tVoid checkMonotonic (tFloat *x1, tFloat *x2);
		tVoid constrainInsideBounds (tFloat *x1, tFloat *x2);
		tVoid init_tolerance ();
		tVoid dbl_add (tFloat *a, tFloat *b, tFloat *aplusb);
		tVoid dbl_mult (tFloat *a, tFloat *b, tFloat *atimesb);
		tInt dbl_gt (tFloat *a, tFloat *b);
		
		/********************************************************************************************/
		cAnimCurve::cAnimCurve (const string n, cGroup * f) : super (n,f) {
			init ();
		}

		/********************************************************************************************/
		tVoid cAnimCurve::init (tInt __numKeys,
					tBool __isWeighted,
					tBool __isStatic,
					eInfinityType __preInfinity,
					eInfinityType __postInfinity,
					cKey* __lastKey,
					tInt __lastIndex,
					tInt __lastInterval,
					tBool __isStep,
					cKey* __keyList) {
			_numKeys = __numKeys;
			_isWeighted = __isWeighted;
			_isStatic = __isStatic;
			_preInfinity = __preInfinity;
			_postInfinity = __postInfinity;
			_lastKey = __lastKey;
			_lastIndex = __lastIndex;
			_lastInterval = __lastInterval;
			_isStep = __isStep;
			_keyList = __keyList;

		}

		/********************************************************************************************/
		cAnimCurve::~cAnimCurve () {
			delete [] _keyList;
 		}

		/********************************************************************************************/
		tVoid cAnimCurve::preInfinity (const eInfinityType __preInfinity) {_preInfinity=__preInfinity;}

		/********************************************************************************************/
		const cAnimCurve::eInfinityType cAnimCurve::preInfinity () {return _preInfinity;}
		
		/********************************************************************************************/
		tVoid cAnimCurve::postInfinity (const eInfinityType __postInfinity) {_postInfinity=__postInfinity;}

		/********************************************************************************************/
		const cAnimCurve::eInfinityType cAnimCurve::postInfinity () {return _postInfinity;}

		/********************************************************************************************/
		const tInt cAnimCurve::numKeys () {return _numKeys;}

		/********************************************************************************************/
		const cKey* cAnimCurve::keyList () {return _keyList;}
		

		// from cBaseNode
		/********************************************************************************************/
		tBool cAnimCurve::compile () {
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cAnimCurve::visit (tFloat t) {
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tVoid cAnimCurve::leave () {
			return;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cAnimCurve::init () {
			#define ATTRIB(n,t,v,d) ATTRIB_INIT_VAR(n,v)
			#include cAnimCurve_attribsFile
			#undef ATTRIB
			_keyList = NULL;
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cAnimCurve::set (const tDWord key, const string & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_S(n)
				#include cAnimCurve_attribsFile
				#undef ATTRIB
			case 'obj': _obj = value; return true;
			case 'attr': _attr = cUtil::doFli (value); return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cAnimCurve::set (const tDWord key, const tFloat & value) {
			if (super::set (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_SET_N(n)
				#include cAnimCurve_attribsFile
				#undef ATTRIB
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cAnimCurve::get (const tDWord key, string & value) const {
			if (super::get (key, value)) return true;
			switch (key) {
				#define ATTRIB(n,t,v,d) ATTRIB_GET(n)
				#include cAnimCurve_attribsFile
				#undef ATTRIB
			case 'obj': value = _obj; return true;
			case 'attr': value = cUtil::doFli (_attr); return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const vector <tAttr> * cAnimCurve::getAttributeNames () const {
			if (attributeNames.size () == 0) {
				const vector <tAttr> * v = super::getAttributeNames ();
				attributeNames.insert (attributeNames.begin (), v->begin (), v->end ());
				#define ATTRIB(n,t,v,d) ATTRIB_ATTRIBNAMES(n,d)
				#include cAnimCurve_attribsFile
				#undef ATTRIB
				attributeNames.push_back(make_pair('obj',string("The Object that this AC points to")));
				attributeNames.push_back(make_pair('attr',string("The Attribute that this AC points to")));
			}
			return &attributeNames;
		}

		// from cBaseNode
		/********************************************************************************************/
		const string cAnimCurve::getTypeName () const { return "caosGL::gfx::cAnimCurve"; }

		/********************************************************************************************/
		/**
		 *	Function Name:
		 *		engineAnimEvaluate
		 *
		 *	Description:
		 *		A function to evaluate an animation curve at a specified time
		 *
		 *  Input Arguments:
		 *		EtCurve *animCurve			The animation curve to evaluate
		 *		tFloat time					The time (in seconds) to evaluate
		 *
		 *  Return Value:
		 *		tFloat value				The evaluated value of the curve at time
		 */
		const tFloat cAnimCurve::evaluate (const tFloat time) {
			tBool withinInterval = false;
			cKey *nextKey;
			tInt index;
			tFloat value = 0.0;
			tFloat x[4];
			tFloat y[4];
			//tFloat time = 9.35;

			/* make sure we have something to evaluate */
			if (_numKeys == 0) {
				return (value);
			}

			/* check if the time falls into the pre-infinity */
			if (time < _keyList[0].time) {
				if (_preInfinity == eInfinityConstant) {
					return (_keyList[0].value);
				}
				return (evaluateInfinities (time, true));
			}

			/* check if the time falls into the post-infinity */
			if (time > _keyList[_numKeys - 1].time) {
				if (_postInfinity == eInfinityConstant) {
					return (_keyList[_numKeys - 1].value);
				}
				return (evaluateInfinities (time, false));
			}

			/* check if the animation curve is static */
			if (_isStatic) {
				return (_keyList[0].value);
			}

			/* check to see if the time falls within the last segment we evaluated */
			if (_lastKey != cNULL) {
				if ((_lastIndex < (_numKeys - 1))
				&&	(time > _lastKey->time)) {
					nextKey = &(_keyList[_lastIndex + 1]);
					if (time == nextKey->time) {
						_lastKey = nextKey;
						++_lastIndex;
						return (_lastKey->value);
					}
					if (time < nextKey->time ) {
						index = _lastIndex + 1;
						withinInterval = true;
					}
				}
				else if ((_lastIndex > 0)
					&&	(time < _lastKey->time)) {
					nextKey = &(_keyList[_lastIndex + 1]);
					if (time > nextKey->time) {
						index = _lastIndex;
						withinInterval = true;
					}
					if (time == nextKey->time) {
						_lastKey = nextKey;
						--_lastIndex;
						return (_lastKey->value);
					}
				}
			}

			/* it does not, so find the new segment */
			if (!withinInterval) {
				if (find (time, &index) || (index == 0)) {
					/**
					 *	Exact match or before range of this action,
					 *	return exact keyframe value.
					 */
					_lastKey = &(_keyList[index]);
					_lastIndex = index;
					return (_lastKey->value);
				}
				else if (index == _numKeys) {
					/* Beyond range of this action return end keyframe value */
					_lastKey = &(_keyList[0]);
					_lastIndex = 0;
					return (_keyList[_numKeys - 1].value);
				}
			}

			/* if we are in a new segment, pre-compute and cache the bezier parameters */
			if (_lastInterval != (index - 1)) {
				_lastInterval = index - 1;
				_lastIndex = _lastInterval;
				_lastKey = &(_keyList[_lastInterval]);
				if ((_lastKey->outTanX == 0.0)
				&&	(_lastKey->outTanY == 0.0)) {
					_isStep = true;
				}
				else {
					_isStep = false;
					x[0] = _lastKey->time;
					y[0] = _lastKey->value;
					x[1] = x[0] + (_lastKey->outTanX * dOneThird);
					y[1] = y[0] + (_lastKey->outTanY * dOneThird);

					nextKey = &(_keyList[index]);
					x[3] = nextKey->time;
					y[3] = nextKey->value;
					x[2] = x[3] - (nextKey->inTanX * dOneThird);
					y[2] = y[3] - (nextKey->inTanY * dOneThird);

					if (_isWeighted) {
						BezierCreate (x, y);
					}
					else {
						HermiteCreate (x, y);
					}
				}
			}

			/* finally we can evaluate the segment */
			if (_isStep) {
				value = _lastKey->value;
			}
			else if (_isWeighted) {
				value = BezierEvaluate (time);
			}
			else {
				value = HermiteEvaluate (time);
			}
			return (value);
		}

		/********************************************************************************************/
		tInt dbl_gt (tFloat *a, tFloat *b)
		{
			return (*a > *b ? 1 : 0);
		}

		/********************************************************************************************/
		tVoid dbl_mult (tFloat *a, tFloat *b, tFloat *atimesb)
		{
			tFloat product = (*a) * (*b);
			*atimesb = product;
		}

		/********************************************************************************************/
		tVoid dbl_add (tFloat *a, tFloat *b, tFloat *aplusb)
		{
			tFloat sum = (*a) + (*b);
			*aplusb = sum;
		}

		/********************************************************************************************/
		/**
		 * Description:
		 *		Init the machine tolerance, sMachineTolerance, is defined to be the
		 *		double that satisfies:
		 *		(1)  sMachineTolerance = 2^(k), for some integer k;
		 *		(2*) (1.0 + sMachineTolerance > 1.0) is TRUE;
		 *		(3*) (1.0 + sMachineTolerance/2.0 == 1.0) is TRUE.
		 *		(*)  When NO floating point optimization is used.
		 *
		 *		To foil floating point optimizers, sMachineTolerance must be
		 *		computed using dbl_mult(), dbl_add() and dbl_gt().
		 */
		tVoid init_tolerance ()
		{
			tFloat one, half, sum;
			one  = 1.0;
			half = 0.5;
			sMachineTolerance = 1.0;
			do {
				dbl_mult (&sMachineTolerance, &half, &sMachineTolerance);
				dbl_add (&sMachineTolerance, &one, &sum);
			} while (dbl_gt (&sum, &one));
			sMachineTolerance = 2.0 * sMachineTolerance;
		}

		/********************************************************************************************/
		/**
		 *	Description:
		 *		We want to ensure that (x1, x2) is inside the ellipse
		 *		(x1^2 + x2^2 - 2(x1 +x2) + x1*x2 + 1) given that we know
		 *		x1 is within the x bounds of the ellipse.
		 */
		tVoid constrainInsideBounds (tFloat *x1, tFloat *x2)
		{
			tFloat b, c,  discr,  root;

			if ((*x1 + sMachineTolerance) < dFourThirds) {
				b = *x1 - 2.0;
				c = *x1 - 1.0;
				discr = sqrt (b * b - 4 * c * c);
				root = (-b + discr) * 0.5;
				if ((*x2 + sMachineTolerance) > root) {
					*x2 = root - sMachineTolerance;
				}
				else {
					root = (-b - discr) * 0.5;
					if (*x2 < (root + sMachineTolerance)) {
						*x2 = root + sMachineTolerance;
					}
				}
			}
			else {
				*x1 = dFourThirds - sMachineTolerance;
				*x2 = dOneThird - sMachineTolerance;
			}
		}

		/********************************************************************************************/
		/**
		 *	Description:
		 *
		 *		Given the bezier curve
		 *			 B(t) = [t^3 t^2 t 1] * | -1  3 -3  1 | * | 0  |
		 *									|  3 -6  3  0 |   | x1 |
		 *									| -3  3  0  0 |   | x2 |
		 *									|  1  0  0  0 |   | 1  |
		 *
		 *		We want to ensure that the B(t) is a monotonically increasing function.
		 *		We can do this by computing
		 *			 B'(t) = [3t^2 2t 1 0] * | -1  3 -3  1 | * | 0  |
		 *									 |  3 -6  3  0 |   | x1 |
		 *									 | -3  3  0  0 |   | x2 |
		 *									 |  1  0  0  0 |   | 1  |
		 *
		 *		and finding the roots where B'(t) = 0.  If there is at most one root
		 *		in the interval [0, 1], then the curve B(t) is monotonically increasing.
		 *
		 *		It is easier if we use the control vector [ 0 x1 (1-x2) 1 ] since
		 *		this provides more symmetry, yields better equations and constrains
		 *		x1 and x2 to be positive.
		 *
		 *		Therefore:
		 *			 B'(t) = [3t^2 2t 1 0] * | -1  3 -3  1 | * | 0    |
		 *									 |  3 -6  3  0 |   | x1   |
		 *									 | -3  3  0  0 |   | 1-x2 |
		 *									 |  1  0  0  0 |   | 1    |
		 *
		 *				   = [t^2 t 1 0] * | 3*(3*x1 + 3*x2 - 2)  |
		 *								   | 2*(-6*x1 - 3*x2 + 3) |
		 *								   | 3*x1                 |
		 *								   | 0                    |
		 *
		 *		gives t = (2*x1 + x2 -1) +/- sqrt(x1^2 + x2^2 + x1*x2 - 2*(x1 + x2) + 1)
		 *				  --------------------------------------------------------------
		 *								3*x1 + 3* x2 - 2
		 *
		 *		If the ellipse [x1^2 + x2^2 + x1*x2 - 2*(x1 + x2) + 1] <= 0, (Note
		 *		the symmetry) x1 and x2 are valid control values and the curve is
		 *		monotonic.  Otherwise, x1 and x2 are invalid and have to be projected
		 *		onto the ellipse.
		 *
		 *		It happens that the maximum value that x1 or x2 can be is 4/3.
		 *		If one of the values is less than 4/3, we can determine the
		 *		boundary constraints for the other value.
		 */
		tVoid checkMonotonic (tFloat *x1, tFloat *x2)
		{
			tFloat d;

			/**
			 * We want a control vector of [ 0 x1 (1-x2) 1 ] since this provides
			 * more symmetry. (This yields better equations and constrains x1 and x2
			 * to be positive.)
			 */
			*x2 = 1.0 - *x2;

			/* x1 and x2 must always be positive */
			if (*x1 < 0.0) *x1 = 0.0;
			if (*x2 < 0.0) *x2 = 0.0;

			/**
			 * If x1 or x2 are greater than 1.0, then they must be inside the
			 * ellipse (x1^2 + x2^2 - 2(x1 +x2) + x1*x2 + 1).
			 * x1 and x2 are invalid if x1^2 + x2^2 - 2(x1 +x2) + x1*x2 + 1 > 0.0
			 */
			if ((*x1 > 1.0) || (*x2 > 1.0)) {
				d = *x1 * (*x1 - 2.0 + *x2) + *x2 * (*x2 - 2.0) + 1.0;
				if ((d + sMachineTolerance) > 0.0) {
					constrainInsideBounds (x1, x2);
				}
			}

			*x2 = 1.0 - *x2;
		}

		/********************************************************************************************/
		/**
		 *	Description:
		 *		Convert the control values for a polynomial defined in the Bezier
		 *		basis to a polynomial defined in the power basis (t^3 t^2 t 1).
		 */
		tVoid bezierToPower (
			tFloat a1, tFloat b1, tFloat c1, tFloat d1,
			tFloat *a2, tFloat *b2, tFloat *c2, tFloat *d2)
		{
			tFloat a = b1 - a1;
			tFloat b = c1 - b1;
			tFloat c = d1 - c1;
			tFloat d = b - a;
			*a2 = c - b - d;
			*b2 = d + d + d;
			*c2 = a + a + a;
			*d2 = a1;
		}

		/********************************************************************************************/
		/**
		 *   Evaluate a polynomial in array form ( value only )
		 *   input:
		 *      P               array 
		 *      deg             degree
		 *      s               parameter
		 *   output:
		 *      ag_horner1      evaluated polynomial
		 *   process: 
		 *      ans = sum (i from 0 to deg) of P[i]*s^i
		 *   restrictions: 
		 *      deg >= 0           
		 */
		tFloat ag_horner1 (tFloat P[], tInt deg, tFloat s)
		{
			tFloat h = P[deg];
			while (--deg >= 0) h = (s * h) + P[deg];
			return (h);
		}


		/********************************************************************************************/
		/**
		 *	Description
		 *   Compute parameter value at zero of a function between limits
		 *       with function values at limits
		 *   input:
		 *       a, b      real interval
		 *       fa, fb    double values of f at a, b
		 *       f         real valued function of t and pars
		 *       tol       tolerance
		 *       pars      pointer to a structure
		 *   output:
		 *       ag_zeroin2   a <= zero of function <= b
		 *   process:
		 *       We find the zeroes of the function f(t, pars).  t is
		 *       restricted to the interval [a, b].  pars is passed in as
		 *       a pointer to a structure which contains parameters
		 *       for the function f.
		 *   restrictions:
		 *       fa and fb are of opposite sign.
		 *       Note that since pars comes at the end of both the
		 *       call to ag_zeroin and to f, it is an optional parameter.
		 */
		tFloat ag_zeroin2 (tFloat a, tFloat b, tFloat fa, tFloat fb, tFloat tol, AG_POLYNOMIAL *pars)
		{
			tInt test;
			tFloat c, d, e, fc, del, m, machtol, p, q, r, s;

			/* initialization */
			machtol = sMachineTolerance;

			/* start iteration */
		label1:
			c = a;  fc = fa;  d = b-a;  e = d;
		label2:
			if (fabs(fc) < fabs(fb)) {
				a = b;   b = c;   c = a;   fa = fb;   fb = fc;   fc = fa;
			}

			/* convergence test */
			del = 2.0 * machtol * fabs(b) + 0.5*tol;
			m = 0.5 * (c - b);
			test = ((fabs(m) > del) && (fb != 0.0));
			if (test) {
				if ((fabs(e) < del) || (fabs(fa) <= fabs(fb))) {
					/* bisection */
					d = m;  e= d;
				}
				else {
					s = fb / fa;
					if (a == c) {
						/* linear interpolation */
						p = 2.0*m*s;    q = 1.0 - s;
					}
					else {
						/* inverse quadratic interpolation */
						q = fa/fc;
						r = fb/fc;
						p = s*(2.0*m*q*(q-r)-(b-a)*(r-1.0));
						q = (q-1.0)*(r-1.0)*(s-1.0);
					}
					/* adjust the sign */
					if (p > 0.0) q = -q;  else p = -p;
					/* check if interpolation is acceptable */
					s = e;   e = d;
					if ((2.0*p < 3.0*m*q-fabs(del*q))&&(p < fabs(0.5*s*q))) {
						d = p/q;
					}
					else {
						d = m;	e = d;
					}
				}
				/* complete step */
				a = b;	fa = fb;
				if ( fabs(d) > del )   b += d;
				else if (m > 0.0) b += del;  else b -= del;
				fb = ag_horner1 (pars->p, pars->deg, b);
				if (fb*(fc/fabs(fc)) > 0.0 ) {
					goto label1;
				}
				else {
					goto label2;
				}
			}
			return (b);
		}

		/********************************************************************************************/
		/**
		 *	Description:
		 *   Compute parameter value at zero of a function between limits
		 *   input:
		 *       a, b            real interval
		 *       f               real valued function of t and pars
		 *       tol             tolerance
		 *       pars            pointer to a structure
		 *   output:
		 *       ag_zeroin       zero of function
		 *   process:
		 *       Call ag_zeroin2 to find the zeroes of the function f(t, pars).
		 *       t is restricted to the interval [a, b].
		 *       pars is passed in as a pointer to a structure which contains
		 *       parameters for the function f.
		 *   restrictions:
		 *       f(a) and f(b) are of opposite sign.
		 *       Note that since pars comes at the end of both the
		 *         call to ag_zeroin and to f, it is an optional parameter.
		 *       If you already have values for fa,fb use ag_zeroin2 directly
		 */
		tFloat ag_zeroin (tFloat a, tFloat b, tFloat tol, AG_POLYNOMIAL *pars)
		{
			tFloat fa, fb;

			fa = ag_horner1 (pars->p, pars->deg, a);
			if (fabs(fa) < sMachineTolerance) return(a);

			fb = ag_horner1 (pars->p, pars->deg, b);
			if (fabs(fb) < sMachineTolerance) return(b);

			return (ag_zeroin2 (a, b, fa, fb, tol, pars));
		} 

		/********************************************************************************************/
		/**
		 * Description:
		 *   Find the zeros of a polynomial function on an interval
		 *   input:
		 *       Poly                 array of coefficients of polynomial
		 *       deg                  degree of polynomial
		 *       a, b                 interval of definition a < b
		 *       a_closed             include a in interval (TRUE or FALSE)
		 *       b_closed             include b in interval (TRUE or FALSE)
		 *   output: 
		 *       polyzero             number of roots 
		 *                            -1 indicates Poly == 0.0
		 *       Roots                zeroes of the polynomial on the interval
		 *   process:
		 *       Find all zeroes of the function on the open interval by 
		 *       recursively finding all of the zeroes of the derivative
		 *       to isolate the zeroes of the function.  Return all of the 
		 *       zeroes found adding the end points if the corresponding side
		 *       of the interval is closed and the value of the function 
		 *       is indeed 0 there.
		 *   restrictions:
		 *       The polynomial p is simply an array of deg+1 doubles.
		 *       p[0] is the constant term and p[deg] is the coef 
		 *       of t^deg.
		 *       The array roots should be dimensioned to deg+2. If the number
		 *       of roots returned is greater than deg, suspect numerical
		 *       instabilities caused by some nearly flat portion of Poly.
		 */
		tInt polyZeroes (tFloat Poly[], tInt deg, tFloat a, tInt a_closed, tFloat b, tInt b_closed, tFloat Roots[])
		{
			tInt i, left_ok, right_ok, nr, ndr, skip;
			tFloat e, f, s, pe, ps, tol, *p, p_x[22], *d, d_x[22], *dr, dr_x[22];
			AG_POLYNOMIAL ply;

			e = pe = 0.0;  
			f = 0.0;

			for (i = 0 ; i < deg + 1; ++i) {
				f += fabs(Poly[i]);
			}
			tol = (fabs(a) + fabs(b))*(deg+1)*sMachineTolerance;

			/* Zero polynomial to tolerance? */
			if (f <= tol)  return(-1);

			p = p_x;  d = d_x;  dr = dr_x;
			for (i = 0 ; i < deg + 1; ++i) {
				p[i] = 1.0/f * Poly[i];
			}

			/* determine true degree */
			while ( fabs(p[deg]) < tol) deg--;

			/* Identically zero poly already caught so constant fn != 0 */
			nr = 0;
			if (deg == 0) return (nr);

			/* check for linear case */
			if (deg == 1) {
				Roots[0] = -p[0] / p[1];
				left_ok  = (a_closed) ? (a<Roots[0]+tol) : (a<Roots[0]-tol);
				right_ok = (b_closed) ? (b>Roots[0]-tol) : (b>Roots[0]+tol);
				nr = (left_ok && right_ok) ? 1 : 0;
				if (nr) {
					if (a_closed && Roots[0]<a) Roots[0] = a;
					else if (b_closed && Roots[0]>b) Roots[0] = b;
				}
				return (nr);
			}
			/* handle non-linear case */
			else {
				ply.p = p;  ply.deg = deg;

				/* compute derivative */
				for (i=1; i<=deg; i++) d[i-1] = i*p[i];

				/* find roots of derivative */
				ndr = polyZeroes ( d, deg-1, a, 0, b, 0, dr );
				if (ndr == -1) return (0);

				/* find roots between roots of the derivative */
				for (i=skip=0; i<=ndr; i++) {
					if (nr>deg) return (nr);
					if (i==0) {
						s=a; ps = ag_horner1( p, deg, s);
						if ( fabs(ps)<=tol && a_closed) Roots[nr++]=a;
					}
					else { s=e; ps=pe; }
					if (i==ndr) { e = b; skip = 0;}
					else e=dr[i];
					pe = ag_horner1( p, deg, e );
					if (skip) skip = 0;
					else {
						if ( fabs(pe) < tol ) {
							if (i!=ndr || b_closed) {
								Roots[nr++] = e;
								skip = 1;
							}
						}
						else if ((ps<0 && pe>0)||(ps>0 && pe<0)) {
							Roots[nr++] = ag_zeroin(s, e, 0.0, &ply );
							if ((nr>1) && Roots[nr-2]>=Roots[nr-1]-tol) { 
								Roots[nr-2] = (Roots[nr-2]+Roots[nr-1]) * 0.5;
								nr--;
							}
						}
					}
				}
			}

			return (nr);
		} 

		/********************************************************************************************/
		/**
		 *	Description:
		 *		Create a constrained single span cubic 2d bezier curve using the
		 *		specified control points.  The curve interpolates the first and
		 *		last control point.  The internal two control points may be
		 *		adjusted to ensure that the curve is monotonic.
		 */
		tVoid cAnimCurve::BezierCreate (tFloat x[4], tFloat y[4])
		{
			static tBool sInited = false;
			tFloat rangeX, dx1, dx2, nX1, nX2, oldX1, oldX2;

			if (!sInited) {
				init_tolerance ();
				sInited = true;
			}

			rangeX = x[3] - x[0];
			if (rangeX == 0.0) {
				return;
			}
			dx1 = x[1] - x[0];
			dx2 = x[2] - x[0];

			/* normalize X control values */
			nX1 = dx1 / rangeX;
			nX2 = dx2 / rangeX;

			/* if all 4 CVs equally spaced, polynomial will be linear */
			if ((nX1 == dOneThird) && (nX2 == dTwoThirds)) {
				_isLinear = true;
			} else {
				_isLinear = false;
			}

			/* save the orig normalized control values */
			oldX1 = nX1;
			oldX2 = nX2;

			/**
			 * check the inside control values yield a monotonic function.
			 * if they don't correct them with preference given to one of them.
			 *
			 * Most of the time we are monotonic, so do some simple checks first
			 */
			if (nX1 < 0.0) nX1 = 0.0;
			if (nX2 > 1.0) nX2 = 1.0;
			if ((nX1 > 1.0) || (nX2 < -1.0)) {
				checkMonotonic (&nX1, &nX2);
			}

			/* compute the new control points */
			if (nX1 != oldX1) {
				x[1] = x[0] + nX1 * rangeX;
				if (oldX1 != 0.0) {
					y[1] = y[0] + (y[1] - y[0]) * nX1 / oldX1;
				}
			}
			if (nX2 != oldX2) {
				x[2] = x[0] + nX2 * rangeX;
				if (oldX2 != 1.0) {
					y[2] = y[3] - (y[3] - y[2]) * (1.0 - nX2) / (1.0 - oldX2);
				}
			}

			/* save the control points */
			_fX1 = x[0];
			_fX4 = x[3];

			/* convert Bezier basis to power basis */
			bezierToPower (
				0.0, nX1, nX2, 1.0,
				&(_fCoeff[3]), &(_fCoeff[2]), &(_fCoeff[1]), &(_fCoeff[0])
			);
			bezierToPower (
				y[0], y[1], y[2], y[3],
				&(_fPolyY[3]), &(_fPolyY[2]), &(_fPolyY[1]), &(_fPolyY[0])
			);
		}

		/********************************************************************************************/
		/**
		 * Description:
		 *		Given the time between fX1 and fX4, return the
		 *		value of the curve at that time.
		 */
		tFloat cAnimCurve::BezierEvaluate (tFloat time)
		{
			tFloat t, s, poly[4], roots[5];
			tInt numRoots;

			if (_fX1 == time) {
				s = 0.0;
			}
			else if (_fX4 == time) {
				s = 1.0;
			}
			else {
				s = (time - _fX1) / (_fX4 - _fX1);
			}

			if (_isLinear) {
				t = s;
			}
			else {
				poly[3] = _fCoeff[3];
				poly[2] = _fCoeff[2];
				poly[1] = _fCoeff[1];
				poly[0] = _fCoeff[0] - s;

				numRoots = polyZeroes (poly, 3, 0.0, 1, 1.0, 1, roots);
				if (numRoots == 1) {
					t = roots[0];
				}
				else {
					t = 0.0;
				}
			}
			return (t * (t * (t * _fPolyY[3] + _fPolyY[2]) + _fPolyY[1]) + _fPolyY[0]);
		}

		/********************************************************************************************/
		tVoid cAnimCurve::HermiteCreate (tFloat x[4], tFloat y[4])
		{
			tFloat dx, dy, tan_x, m1, m2, length, d1, d2;

			/* save the control points */
			_fX1 = x[0];

			/**
			 *	Compute the difference between the 2 keyframes.					
			 */
			dx = x[3] - x[0];
			dy = y[3] - y[0];

			/**
			 * 	Compute the tangent at the start of the curve segment.			
			 */
			tan_x = x[1] - x[0];
			m1 = m2 = dMaxTan;
			if (tan_x != 0.0) {
				m1 = (y[1] - y[0]) / tan_x;
			}

			tan_x = x[3] - x[2];
			if (tan_x != 0.0) {
				m2 = (y[3] - y[2]) / tan_x;
			}

			length = 1.0 / (dx * dx);
			d1 = dx * m1;
			d2 = dx * m2;
			_fCoeff[0] = (d1 + d2 - dy - dy) * length / dx;
			_fCoeff[1] = (dy + dy + dy - d1 - d1 - d2) * length;
			_fCoeff[2] = m1;
			_fCoeff[3] = y[0];
		}

		/********************************************************************************************/
		/*
		 * Description:
		 *		Given the time between fX1 and fX2, return the function
		 *		value of the curve
		*/
		tFloat cAnimCurve::HermiteEvaluate (tFloat time)
		{
			tFloat t;
			t = time - _fX1;
			return (t * (t * (t * _fCoeff[0] + _fCoeff[1]) + _fCoeff[2]) + _fCoeff[3]);
		}

		/********************************************************************************************/
		/**
		 *	Function Name:
		 *		evaluateInfinities
		 *
		 *	Description:
		 *		A static helper function to evaluate the infinity portion of an
		 *	animation curve.  The infinity portion is the parts of the animation
		 *	curve outside the range of keys.
		 *
		 *  Input Arguments:
		 *		EtCurve *animCurve			The animation curve to evaluate
		 *		tFloat time					The time (in seconds) to evaluate
		 *		tBool evalPre
		 *			true				evaluate the pre-infinity portion
		 *			false			evaluate the post-infinity portion
		 *
		 *  Return Value:
		 *		tFloat value				The evaluated value of the curve at time
		 */
		tFloat cAnimCurve::evaluateInfinities (tFloat time, tBool evalPre)
		{
			tFloat value = 0.0;
			tFloat	valueRange;
			tFloat	factoredTime, firstFloat, lastFloat, timeRange;
			tFloat	remainder, tanX, tanY;
			tFloat numCycles;
			double notUsed; // this has to be a double, so declare it as such

			/* make sure we have something to evaluate */
			if (_numKeys == 0) {
				return (value);
			}

			/* find the number of cycles of the base animation curve */
			firstFloat = _keyList[0].time;
			lastFloat = _keyList[_numKeys - 1].time;
			timeRange = lastFloat - firstFloat;
			if (timeRange == 0.0) {
				/**
				 * Means that there is only one key in the curve.. Return the value
				 * of that key..
				 */
				return (_keyList[0].value);
			}
			if (time > lastFloat) {
				remainder = fabs (modf ((time - lastFloat) / timeRange, &notUsed));
				numCycles = notUsed;
			}
			else {
				remainder = fabs (modf ((time - firstFloat) / timeRange, &notUsed));
				numCycles = notUsed;
			}
			factoredTime = timeRange * remainder;
			numCycles = fabs (numCycles) + 1;

			if (evalPre) {
				/* evaluate the pre-infinity */
				if (_preInfinity == eInfinityOscillate) {
					if ((remainder = modf (numCycles / 2.0, &notUsed)) != 0.0) {
						factoredTime = firstFloat + factoredTime;
					}
					else {
						factoredTime = lastFloat - factoredTime;
					}
				}
				else if ((_preInfinity == eInfinityCycle)
				||	(_preInfinity == eInfinityCycleRelative)) {
					factoredTime = lastFloat - factoredTime;
				}
				else if (_preInfinity == eInfinityLinear) {
					factoredTime = firstFloat - time;
					tanX = _keyList[0].inTanX;
					tanY = _keyList[0].inTanY;
					value = _keyList[0].value;
					if (tanX != 0.0) {
						value -= ((factoredTime * tanY) / tanX);
					}
					return (value);
				}
			}
			else {
				/* evaluate the post-infinity */
				if (_postInfinity == eInfinityOscillate) {
					if ((remainder = modf (numCycles / 2.0, &notUsed)) != 0.0) {
						factoredTime = lastFloat - factoredTime;
					}
					else {
						factoredTime = firstFloat + factoredTime;
					}
				}
				else if ((_postInfinity == eInfinityCycle)
				||	(_postInfinity == eInfinityCycleRelative)) {
					factoredTime = firstFloat + factoredTime;
				}
				else if (_postInfinity == eInfinityLinear) {
					factoredTime = time - lastFloat;
					tanX = _keyList[_numKeys - 1].outTanX;
					tanY = _keyList[_numKeys - 1].outTanY;
					value = _keyList[_numKeys - 1].value;
					if (tanX != 0.0) {
						value += ((factoredTime * tanY) / tanX);
					}
					return (value);
				}
			}

			value = evaluate (factoredTime);

			/* Modify the value if infinityType is cycleRelative */
			if (evalPre && (_preInfinity == eInfinityCycleRelative)) {
				valueRange = _keyList[_numKeys - 1].value -
								_keyList[0].value;
				value -= (numCycles * valueRange);
			}
			else if (!evalPre && (_postInfinity == eInfinityCycleRelative)) {
				valueRange = _keyList[_numKeys - 1].value -
								_keyList[0].value;
				value += (numCycles * valueRange);
			}
			return (value);
		}

		/********************************************************************************************/
		/**
		 *	Function Name:
		 *		find
		 *
		 *	Description:
		 *		A static helper method to find a key prior to a specified time
		 *
		 *  Input Arguments:
		 *		EtCurve *animCurve			The animation curve to search
		 *		tFloat time					The time (in seconds) to find
		 *		tInt *index				The index of the key prior to time
		 *
		 *  Return Value:
		 *      tBool result
		 *			true				time is represented by an actual key
		 *										(with the index in index)
		 *			false			the index key is the key less than time
		 *
		 *	Note:
		 *		keys are sorted by ascending time, which means we can use a binary
		 *	search to find the key
		 */
		tBool cAnimCurve::find (tFloat time, tInt *index)
		{
			tInt len, mid, low, high;

			/* make sure we have something to search */
			if (index == cNULL) {
				return (false);
			}

			/* use a binary search to find the key */
			*index = 0;
			len = _numKeys;
			if (len > 0) {
				low = 0;
				high = len - 1;
				do {
					mid = (low + high) >> 1;
					if (time < _keyList[mid].time) {
						high = mid - 1;			/* Search lower half */
					} else if (time > _keyList[mid].time) {
						low  = mid + 1;			/* Search upper half */
					}
					else {
						*index = mid;	/* Found item! */
						return (true);
					}
				} while (low <= high);
				*index = low;
			}
			return (false);
		}
	} // namespace gfx
} // namespace caosGL

// for node creation
#include <caosGL/core/cRegistry.h>
#include <caosGL/gfx/cNodeCreator.h>

class cAnimCurveNodeCreator : public cNodeCreator {
public:
	cAnimCurveNodeCreator () {
		name ("caosGL::gfx::cAnimCurve");
	}
	cBaseNode * createNode (const string n, cGroup * f) {
		return new caosGL::gfx::cAnimCurve (n,f);
	}
};
caosGL::core::cRegisterNodeCreator <cAnimCurveNodeCreator> cAnimCurveNodeCreatorInstance;

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof