/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// --------------------------------------------------------------------------
// File:        cZipFile.cpp
//
// Purpose:     The implementation of a quick'n dirty ZIP file reader class.
//              (C) Copyright 2000 Javier Arevalo. Use and modify as you like
//              Get zlib from http://www.cdrom.com/pub/infozip/zlib/
// --------------------------------------------------------------------------

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/core/cZipFile.h>

// extern includes
#include <zlib.h>
#include <string.h>
#include <io.h>


// DEBUG
#include <iostream>
using namespace std;

namespace caosGL {
	namespace core {
		/**
		 *<br> class:		cZipFile
		 *<br> namespace:	caosGL::util::zip
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		Zip File reader
		 *
		 */

		// --------------------------------------------------------------------------
		// Basic types.
		// --------------------------------------------------------------------------
		typedef unsigned long dword;
		typedef unsigned short word;
		typedef unsigned char byte;

		cZipFile::~cZipFile () {End();}

		// --------------------------------------------------------------------------
		// ZIP file structures. Note these have to be packed.
		// --------------------------------------------------------------------------

		#pragma pack(2)
		// --------------------------------------------------------------------------
		// --------------------------------------------------------------------------
		struct cZipFile::sZipLocalHeader {
			enum {
				SIGNATURE = 0x04034b50,
				COMP_STORE  = 0,
				COMP_DEFLAT = 8,
			};
			dword   sig;
			word    version;
			word    flag;
			word    compression;      // COMP_xxxx
			word    modTime;
			word    modDate;
			dword   crc32;
			dword   cSize;
			dword   ucSize;
			word    fnameLen;         // Filename string follows header.
			word    xtraLen;          // Extra field follows filename.
		};
		
		// --------------------------------------------------------------------------
		// --------------------------------------------------------------------------
		struct cZipFile::sZipDirHeader {
			enum {
				SIGNATURE = 0x06054b50,
				CAOSGL = 0x534F4143,
			};
			dword   sig;
			word    nDisk;
			word    nStartDisk;
			word    nDirEntries;
			word    totalDirEntries;
			dword   dirSize;
			dword   dirOffset;
			word    cmntLen;
		};
		
		// --------------------------------------------------------------------------
		// --------------------------------------------------------------------------
		struct cZipFile::sZipDirFileHeader {
			enum {
				SIGNATURE   = 0x02014b50,
				COMP_STORE  = 0,
				COMP_DEFLAT = 8,
			};
			dword   sig;
			word    verMade;
			word    verNeeded;
			word    flag;
			word    compression;      // COMP_xxxx
			word    modTime;
			word    modDate;
			dword   crc32;
			dword   cSize;            // Compressed size
			dword   ucSize;           // Uncompressed size
			word    fnameLen;         // Filename string follows header.
			word    xtraLen;          // Extra field follows filename.
			word    cmntLen;          // Comment field follows extra field.
			word    diskStart;
			word    intAttr;
			dword   extAttr;
			dword   hdrOffset;
			
			char *GetName   () const { return (char *)(this + 1);   }
			char *GetExtra  () const { return GetName() + fnameLen; }
			char *GetComment() const { return GetExtra() + xtraLen; }
		};

		#pragma pack()

		/********************************************************************************************/
		// --------------------------------------------------------------------------
		// Function:      Init
		// Purpose:       Initialize the object and read the zip file directory.
		// Parameters:    A stdio FILE* used for reading.
		// --------------------------------------------------------------------------
		eStatus cZipFile::Init(istream * f) {
			End();
			if (f == cNULL)
				return RET_FAIL;
			m_f = f;
			
			long whereHeader = findHeader ();
			// Assuming no extra comment at the end, read the whole end record.
			sZipDirHeader dh;
			
			m_f->seekg (whereHeader);
			long tell = f->tellg ();
			f->seekg (0,ios_base::end);
			long length = f->tellg ();
			f->seekg (tell);
			long dhOffset = tell;
			memset(&dh, 0, sizeof(dh));
			m_f->read ((tChar*)&dh, sizeof(dh));
			
			// Check
			if (!(dh.sig == sZipDirHeader::SIGNATURE || dh.sig == sZipDirHeader::CAOSGL))
				return RET_FAIL;
			
			// Go to the beginning of the directory.
			m_f->seekg(dhOffset - dh.dirSize);
			
			// Allocate the data buffer, and read the whole thing.
			m_pDirData = new char[dh.dirSize + dh.nDirEntries*sizeof(*m_papDir)];
			if (!m_pDirData)
				return RET_FAIL;
			memset(m_pDirData, 0, dh.dirSize + dh.nDirEntries*sizeof(*m_papDir));
			m_f->read ((tChar*)m_pDirData, dh.dirSize);
			
			// Now process each entry.
			char *pfh = m_pDirData;
			m_papDir = (const sZipDirFileHeader **)(m_pDirData + dh.dirSize);
			
			eStatus ret = RET_OK;
			
			for (int i = 0; i < dh.nDirEntries && ret == RET_OK; i++) {
				sZipDirFileHeader &fh = *(sZipDirFileHeader*)pfh;
				
				// Store the address of nth file for quicker access.
				m_papDir[i] = &fh;
				
				// Check the directory entry integrity.
				if (fh.sig != sZipDirFileHeader::SIGNATURE/* && 
					fh.sig != sZipDirHeader::CAOSGL*/) {
					ret = RET_FAIL;
				} else {
					pfh += sizeof(fh);
					
					// Convert UNIX slashes to DOS backlashes.
					for (int j = 0; j < fh.fnameLen; j++)
						if (pfh[j] == '\\')
							pfh[j] = '/';
						
						// Skip name, extra and comment fields.
						pfh += fh.fnameLen + fh.xtraLen + fh.cmntLen;
				}
			}
			if (ret != RET_OK) {
				delete[] m_pDirData;
			} else {
				m_nEntries = dh.nDirEntries;
			}
			
			return ret;
		}

		/********************************************************************************************/
		// --------------------------------------------------------------------------
		// Function:      End
		// Purpose:       Finish the object
		// Parameters:    
		// --------------------------------------------------------------------------
		void cZipFile::End() {
			if (IsOk())  {
				delete[] m_pDirData;
				m_nEntries = 0;
			}
		}

		/********************************************************************************************/
		// --------------------------------------------------------------------------
		// Function:      GetFilename
		// Purpose:       Return the name of a file
		// Parameters:    The file index and the buffer where to store the filename
		// --------------------------------------------------------------------------
		void cZipFile::GetFilename(int i, char *pszDest) const {
			if (pszDest != cNULL) {
				if (i < 0 || i >= m_nEntries) {
					*pszDest = '\0';
				} else {
					memcpy(pszDest, m_papDir[i]->GetName(), m_papDir[i]->fnameLen);
					pszDest[m_papDir[i]->fnameLen] = '\0';
				}
			}
		}

		const string cZipFile::GetFilename(int i) const {
			tChar chr [1024];
			GetFilename (i, chr);
			return chr;
		}

		/********************************************************************************************/
		// --------------------------------------------------------------------------
		// Function:      GetFileLen
		// Purpose:       Return the length of a file so a buffer can be allocated
		// Parameters:    The file index.
		// --------------------------------------------------------------------------
		int cZipFile::GetFileLen(int i) const {
			if (i < 0 || i >= m_nEntries) {
				return -1;
			} else {
				return m_papDir[i]->ucSize;
			}
		}

		/********************************************************************************************/
		// --------------------------------------------------------------------------
		// Function:      ReadFile
		// Purpose:       Uncompress a complete file
		// Parameters:    The file index and the pre-allocated buffer
		// --------------------------------------------------------------------------
		eStatus cZipFile::ReadFile(int i, void *pBuf) {
			if (pBuf == cNULL || i < 0 || i >= m_nEntries) 
				return RET_FAIL;
			
			// Quick'n dirty read, the whole file at once.
			// Ungood if the ZIP has huge files inside
			
			// Go to the actual file and read the local header.
			m_f->seekg (m_papDir[i]->hdrOffset);
			sZipLocalHeader h;
			
			memset(&h, 0, sizeof(h));
			m_f->read ((tChar*)&h, sizeof(h));
			if (h.sig != sZipLocalHeader::SIGNATURE)
				return RET_FAIL;
			
			// Skip extra fields
			m_f->seekg (h.fnameLen + h.xtraLen, ios_base::cur);
			
			if (h.compression == sZipLocalHeader::COMP_STORE) {
				// Simply read in raw stored data.
				m_f->read ((tChar*)pBuf, h.cSize);
				return RET_OK;
			} else if (h.compression != sZipLocalHeader::COMP_DEFLAT) 
				return RET_FAIL;
			
			// Alloc compressed data buffer and read the whole stream
			char *pcData = new char[h.cSize];
			if (!pcData) 
				return RET_FAIL;
			
			memset(pcData, 0, h.cSize);
			m_f->read ((tChar*)pcData, h.cSize);
			
			eStatus ret = RET_OK;
			
			// Setup the inflate stream.
			z_stream stream;
			int err;
			
			stream.next_in = (Bytef*)pcData;
			stream.avail_in = (uInt)h.cSize;
			stream.next_out = (Bytef*)pBuf;
			stream.avail_out = h.ucSize;
			stream.zalloc = (alloc_func)0;
			stream.zfree = (free_func)0;
			
			// Perform inflation. wbits < 0 indicates no zlib header inside the data.
			err = inflateInit2(&stream, -MAX_WBITS);
			if (err == Z_OK) {
				err = inflate(&stream, Z_FINISH);
				inflateEnd(&stream);
				if (err == Z_STREAM_END)
					err = Z_OK;
				inflateEnd(&stream);
			}
			if (err != Z_OK)
				ret = RET_FAIL;
			
			delete[] pcData;
			return ret;
		}

		/********************************************************************************************/
		long cZipFile::findHeader () {
			unsigned int head;
			long tell = m_f->tellg ();
			m_f->seekg (0,ios_base::end);
			long length = m_f->tellg ();
			m_f->seekg (tell);
			long pos = length - sizeof(sZipDirHeader);
			
			while (pos>0) {
				m_f->seekg (pos);
				m_f->read ((tChar*)&head, sizeof(head));
				
				if (head == sZipDirHeader::SIGNATURE || head == sZipDirHeader::CAOSGL)
					break;
				--pos;
			}
			
			m_f->seekg (pos + sizeof(sZipDirHeader));
			long num = length - (pos + sizeof(sZipDirHeader));
			comment = new char [num+1];
			m_f->read ((tChar*)comment, num);
			comment[num] = 0;
			
			return pos;
		}
	} // namespace util
} // namespace caosGL

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof