/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

// package includes
#include <caosGL/core/DynBass.h>

// extern includes
#include <caosGL/core/cVFS.h>
#include <caosGL/core/cParser.h>

#include <caosGL/core/cBassSndNode.h>

static log4cpp::Category& cat = log4cpp::Category::getInstance ("caosGL::core::cBassSndNode");

namespace caosGL {
	namespace core {
		/**
		 *<br> class:		cBassSndNode
		 *<br> namespace:	caosGL::core
		 *<br> inherits:	<none>
		 *<br> implements:	<none>
		 *<br> purpose:		
		 *
		 */

		static vector <tAttr> attributeNames;
		/********************************************************************************************/
		cBassSndNode::cBassSndNode (const string n, caosGL::gfx::cGroup * f) : super (n,f), buf (cNULL), hMusic (0), playing (false), paused (false) {
			init ();
		}

		/********************************************************************************************/
		cBassSndNode::~cBassSndNode () {
			delete [] buf;
		}
		
		/********************************************************************************************/
		const tBool cBassSndNode::prepare (const string what) {
			stop ();
			cat.debug ("Preparing a Bass sound node for %s", what.c_str ());
			istream * f = cVFS::instance ()->getStream (what);
			if (f->good ()) {
				f->seekg (0,ios_base::end);
				tLong length = f->tellg ();
				f->seekg (0);
				if (buf) {
					delete [] buf;
				}
				buf = new tByte [length];
				f->read ((tChar*) buf, length);
				cat.debug ("Trying to load an mp3");
				hMusic = BASS_StreamCreateFile (true, buf, 0, length, 0);
				if (hMusic != 0) {
					stream = true;
				} else {
					cat.debug ("It's not a mp3, trying a module");
					hMusic = BASS_MusicLoad (true, buf, 0, length, 0);
					if (hMusic != 0) {
						cat.error ("At the moment, mods are only partially supported...");
						stream = false;
					} else {
						cat.error ("Couldn't load the sound node... is it an audio file?");
					}
				}
				playing = false;
				paused = false;
			}
			return hMusic != 0;
		}

		/********************************************************************************************/
		const tBool cBassSndNode::play () {
			if (!playing) {
				paused = false;
				playing = true;
				if (stream) {
					return BASS_StreamPlay (hMusic, false, 0) != 0;
				} else {
					return BASS_MusicPlay (hMusic) != 0;
				}
			} else if (paused) {
				paused = false;
				playing = true;
				return BASS_ChannelResume (hMusic) != 0;
			}
			return false;
		}

		/********************************************************************************************/
		const tBool cBassSndNode::pause () {
			if (playing) {
				paused = true;
				playing = false;
				return BASS_ChannelPause (hMusic) != 0;
			}
			return false;
		}

		/********************************************************************************************/
		const tBool cBassSndNode::stop () {
			if (playing) {
				paused = false; 
				playing = false;
				return BASS_ChannelStop (hMusic) != 0;
			} else if (paused) {
				playing = false;
				paused = false;
				return BASS_ChannelStop (hMusic) != 0;

			}
			return false;
		}
		
		/********************************************************************************************/
		tBool cBassSndNode::pos (const tFloat time) {
			DWORD bytes,freq,flags,samples,p;
			bytes = BASS_StreamGetLength (hMusic);
			BASS_ChannelGetAttributes (hMusic, &freq, 0, 0);
			flags = BASS_ChannelGetFlags (hMusic);
			
			samples = freq * (flags & BASS_SAMPLE_MONO?1:2) * (flags & BASS_SAMPLE_8BITS?1:2);

/*			cat.debug ("Stereo: %s, 16Bits: %s, frequency: %u, samples: %u, time: %u", (flags & BASS_SAMPLE_MONO?"no":"yes"),
				(flags & BASS_SAMPLE_8BITS?"no":"yes"), (unsigned int)freq, (unsigned int)samples, (unsigned int)time);
*/
			p =  (double)samples * (double)time;
/*			cat.debug ("Going to go for pos %u", (unsigned int)p);*/
			return BASS_ChannelSetPosition (hMusic, p)!=0;
		}

		/********************************************************************************************/
		const tFloat cBassSndNode::pos () const {
			DWORD bytes,freq,flags,p;
			bytes = BASS_StreamGetLength (hMusic);
			BASS_ChannelGetAttributes (hMusic, &freq, 0, 0);
			flags = BASS_ChannelGetFlags (hMusic);
			p = BASS_ChannelGetPosition (hMusic);
			tDWord _pos = 1000 / ((freq * (flags & BASS_SAMPLE_MONO?1:2) * (flags & BASS_SAMPLE_8BITS?1:2)) * p);
			return (tFloat)_pos/1000.0;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cBassSndNode::compile () {
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cBassSndNode::visit (tFloat t) {
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		tVoid cBassSndNode::leave () {
			return;
		}

		// from cBaseNode
		/********************************************************************************************/
		tBool cBassSndNode::init () {
			return true;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cBassSndNode::set (const tDWord key, const string & value) {
			if (super::set (key, value)) return true;
			switch (key) {
			case 'play': {
				if (cParser::parseBool (value)) {
					if (play ()) {
						return true;
					} else {
						return false;
					}
				}
			} return true;
			case 'stop': {
				if (cParser::parseBool (value)) {
					if (stop ()) {
						return true;
					} else {
						return false;
					}
				}
			} return true;
			case 'paus': {
				if (cParser::parseBool (value)) {
					if (pause ()) {
						return true;
					} else {
						return false;
					}
				}
			} return true;
			case 'prep': {
				if (prepare (value)) {
					return true;
				} else {
					return false;
				}
			} return true;
			case 'pos':  {
				if (pos (cParser::parseDouble (value))) {
					return true;
				} else {
					return false;
				}
			} return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cBassSndNode::set (const tDWord key, const tFloat & value) {
			if (super::set (key, value)) return true;
			switch (key) {
			case 'play': {} return false;
			case 'stop': {} return false;
			case 'paus': {} return false;
			case 'prep': {} return false;
			case 'pos':  {
				if (pos (value)) {
					return true;
				} else {
					return false;
				}
			} return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}

		// from cBaseNode
		/********************************************************************************************/
		const tBool cBassSndNode::get (const tDWord key, string & value) const {
			if (super::get (key, value)) return true;
			switch (key) {
			case 'play': { value = cParser::toString (playing); } return true;
			case 'stop': { value = cParser::toString (!playing); } return true;
			case 'paus': { value = cParser::toString (paused); } return true;
			case 'prep': { value = cParser::toString (hMusic != 0); } return true;
			case 'pos':  { value = cParser::toString (pos ()); } return true;
			case '    ': return false;
			default: return false;
			}
			return false;
		}
		
		// from cBaseNode
		/********************************************************************************************/
		const vector <tAttr> * cBassSndNode::getAttributeNames () const {
			if (attributeNames.size () == 0) {
				const vector <tAttr> * v = super::getAttributeNames ();
				attributeNames.insert (attributeNames.begin (), v->begin (), v->end ());
				attributeNames.push_back(make_pair('play',string("Plays the sound")));
				attributeNames.push_back(make_pair('stop',string("Stops the sound")));
				attributeNames.push_back(make_pair('paus',string("Pauses the sound")));
				attributeNames.push_back(make_pair('prep',string("Prepared the sound")));
				attributeNames.push_back(make_pair('pos',string("Get / Set the position")));
			}
			return &attributeNames;
		}

		// from cBaseNode
		/********************************************************************************************/
		const string cBassSndNode::getTypeName () const { return "caosGL::core::cBassSndNode"; }
	}
}

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof