/* For license details see bottom.
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 */


#ifndef CAOSGL_GFX_CANIMCURVE
#define CAOSGL_GFX_CANIMCURVE

// system includes
#include <caosGL/core/globals.h>
#include <caosGL/core/types.h>

#include <caosGL/gfx/cEvaluatable.h>

// package includes
#include <caosGL/gfx/cKey.h>
#include <caosGL/gfx/cReadKey.h>

// extern includes
#include <vector>
#include <string>
using namespace std;

#define cAnimCurve_attribsFile <caosGL/gfx/cAnimCurve.attribs>

namespace caosGL { 
	namespace gfx {
		/**
		 *<br> class:		cAnimCurve
		 *<br> namespace:	caosGL::gfx
		 *<br> inherits:	caosGL::gfx::cEvaluatable
		 *<br> implements:	<none>
		 *<br> purpose:		An animation curve. You feed it time, it gives you a value. It works best
		 *                  for forward animations.
		 *
		 */

		class CAOSGL_API cAnimCurve : public cEvaluatable {
			typedef cEvaluatable super;
		public:
			enum eInfinityType {
				eInfinityConstant      = 0,
				eInfinityLinear        = 1,
				eInfinityCycle         = 2,
				eInfinityCycleRelative = 3,
				eInfinityOscillate     = 4
			};

			cAnimCurve (const string n, cGroup * f);
			virtual ~cAnimCurve ();

			tVoid init (tInt __numKeys,
						tBool __isWeighted,
						tBool __isStatic,
						eInfinityType __preInfinity,
						eInfinityType __postInfinity,
						cKey* __lastKey,
						tInt __lastIndex,
						tInt __lastInterval,
						tBool __isStep,
						cKey* __keyList);

			virtual const tFloat evaluate (const tFloat time);

			tVoid		  preInfinity (const eInfinityType preInfinity);
			const eInfinityType preInfinity ();

			tVoid		  postInfinity (const eInfinityType postInfinity);
			const eInfinityType postInfinity ();
			
			const tInt numKeys ();
			tVoid numKeys (const tInt numKeys);

			const cKey* keyList ();
	
			// from cBaseNode
			virtual tBool compile  ();
			virtual tBool visit (tFloat t);
			virtual tVoid leave ();
			virtual tBool init ();
			virtual const bool set (const tDWord key, const string & value);
			virtual const bool set (const tDWord key, const tFloat & value);
			virtual const bool get (const tDWord key, string & value) const;
			virtual const vector <tAttr> * getAttributeNames () const;
			virtual const string getTypeName () const;

		protected:
			tInt			_numKeys;		/* The number of keys in the anim curve				*/
			tBool			_isWeighted;		/* whether or not this curve has weighted tangents	*/
			tBool			_isStatic;		/* whether or not all the keys have the same value	*/
			eInfinityType	_preInfinity;	/* how to evaluate pre-infinity						*/
			eInfinityType	_postInfinity;	/* how to evaluate post-infinity					*/

			/* evaluate cache */
			cKey*			_lastKey;		/* lastKey evaluated								*/
			tInt			_lastIndex;		/* last index evaluated								*/
			tInt			_lastInterval;	/* last interval evaluated							*/
			tBool			_isStep;		/* whether or not this interval is a step interval	*/
			tBool			_isLinear;		/* whether or not this interval is linear			*/
			tFloat			_fX1;			/* start x of the segment							*/
			tFloat			_fX4;			/* end x of the segment								*/
			tFloat			_fCoeff[4];		/* bezier x parameters, only used for weighted curves*/
			tFloat			_fPolyY[4];		/* bezier y parameters								*/

			cKey*			_keyList;		/* This must be the last element and contains		*/
											/* an array of keys sorted in ascending order		*/
											/* by time											*/

			tBool find (tFloat time, tInt *index);
			tFloat evaluateInfinities (tFloat time, tBool evalPre);
			tFloat HermiteEvaluate (tFloat time);
			tVoid HermiteCreate (tFloat x[4], tFloat y[4]);
			tFloat BezierEvaluate (tFloat time);
			tVoid BezierCreate (tFloat x[4], tFloat y[4]);

			#define ATTRIB(n,t,v,d) ATTRIB_DEFINE_VAR(n,t)
			#include cAnimCurve_attribsFile
			#undef ATTRIB
		};
	}
}

#endif // CAOSGL_GFX_CANIMCURVE

/**
 * The Catalyst of Design Software License, Version 1.0
 *
 * Copyright (c) 2002 Catalyst of Design (David Morris-Oliveros).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by 
 *        Catalyst of Design (http://talsit.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "caosGL" and "Catalyst of Design" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact caosGL@talsit.org.
 *
 * 5. Products derived from this software may not be called "caosGL",
 *    nor may "caosGL" appear in their name, without prior written
 *    permission of Catalyst of Design.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CATALYST OF DESIGN OR ITS 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */
// eof