#include <math.h>
#include "tri.h"
#include "vector.h"
#include "mesh.h"

#define	FOV	15.0f
#define	PI	3.1415f

int mesh_alloc( mesh *m, int n_verts, int n_faces )
{
	mesh_free( m );
	m->n_verts = n_verts;
	m->n_faces = n_faces;
	m->verts = new vector [n_verts]; if ( !m->verts ) return 0;
	m->norms = new vector [n_verts]; if ( !m->norms ) return 0;
	m->texuv = new tex_coord [n_verts]; if ( !m->texuv ) return 0;
	m->faces = new int [n_faces * 3]; if ( !m->faces ) return 0;
	return 1;
}

void mesh_free( mesh *m )
{
	if ( m->verts ) delete [] m->verts;
	if ( m->norms ) delete [] m->norms;
	if ( m->texuv ) delete [] m->texuv;
	if ( m->faces ) delete [] m->faces;
	m->n_verts = 0;
	m->n_faces = 0;
}

void mesh_generate_normals( mesh *m, int flip )
{
	int	i;
	int	j0, j1, j2;
	float	ax, ay, az;
	float	bx, by, bz;
	float	nx, ny, nz, nl;
	for ( i = 0; i < m->n_verts; i++ ) {
		m->norms[i].x = 0.0f;
		m->norms[i].y = 0.0f;
		m->norms[i].z = 0.0f;
	}
	for ( i = 0; i < m->n_faces; i++ ) {
		j0 = i * 3;
		j1 = j0 + 1;
		j2 = j0 + 2;
		ax = m->verts[m->faces[j2]].x - m->verts[m->faces[j0]].x;
		ay = m->verts[m->faces[j2]].y - m->verts[m->faces[j0]].y;
		az = m->verts[m->faces[j2]].z - m->verts[m->faces[j0]].z;
		bx = m->verts[m->faces[j1]].x - m->verts[m->faces[j0]].x;
		by = m->verts[m->faces[j1]].y - m->verts[m->faces[j0]].y;
		bz = m->verts[m->faces[j1]].z - m->verts[m->faces[j0]].z;
		nx = ay * bz - az * by;
		ny = az * bx - ax * bz;
		nz = ax * by - ay * bx;
		nl = sqrt( nx * nx + ny * ny + nz * nz );
		nx /= nl;
		ny /= nl;
		nz /= nl;
		m->norms[m->faces[j0]].x += nx;
		m->norms[m->faces[j0]].y += ny;
		m->norms[m->faces[j0]].z += nz;
		m->norms[m->faces[j1]].x += nx;
		m->norms[m->faces[j1]].y += ny;
		m->norms[m->faces[j1]].z += nz;
		m->norms[m->faces[j2]].x += nx;
		m->norms[m->faces[j2]].y += ny;
		m->norms[m->faces[j2]].z += nz;
	}
	for ( i = 0; i < m->n_verts; i++ ) {
		nx = m->norms[i].x;
		ny = m->norms[i].y;
		nz = m->norms[i].z;
		nl = sqrt( nx * nx + ny * ny + nz * nz );
		m->norms[i].x /= nl;
		m->norms[i].y /= nl;
		m->norms[i].z /= nl;
		if ( flip ) {
			m->norms[i].x = -m->norms[i].x;
			m->norms[i].y = -m->norms[i].y;
			m->norms[i].z = -m->norms[i].z;
		}
	}
}

int mesh_generate_cube( mesh *m, float size )
{
	int	i;
	vector	cube_vertices[]	= { {  1,  1,  1 },
				    { -1,  1,  1 },
				    {  1, -1,  1 },
				    { -1, -1,  1 },
				    {  1,  1, -1 },
				    { -1,  1, -1 },
				    {  1, -1, -1 },
				    { -1, -1, -1 } };
	int	cube_faces[]	= { 0, 2, 3,  3, 1, 0,
				    4, 5, 7,  7, 6, 4,
				    0, 1, 5,  5, 4, 0,
				    1, 3, 7,  7, 5, 1,
				    3, 2, 6,  6, 7, 3,
				    2, 0, 4,  4, 6, 2 };
	if ( !mesh_alloc( m, 8, 12 ) ) return 0;
	size /= 2.0f;
	for ( i = 0; i < 8; i++ ) {
		m->verts[i].x = cube_vertices[i].x * size;
		m->verts[i].y = cube_vertices[i].y * size;
		m->verts[i].z = cube_vertices[i].z * size;
	}
	for ( i = 0; i < 12 * 3; i++ ) m->faces[i] = cube_faces[i];
	return 1;
}

int mesh_generate_torus( mesh *m, int segments,
	int sides, float radius1, float radius2 )
{
	float	a1, a2;
	int	i, j, k, n;
	if ( segments < 3 ) segments = 3;
	if ( sides < 3 ) sides = 3;
	if ( !mesh_alloc( m, segments * sides, (segments * sides) * 2 ) ) return 0;
	for ( k = 0, i = 0; i < segments; i++ ) {
		for ( j = 0; j < sides; j++ ) {
			a1 = i * (PI * 2.0f / segments);
			a2 = j * (PI * 2.0f / sides);
			m->verts[k].x = (cos( a2 ) * radius2 + radius1) * cos( a1 );
			m->verts[k].y = (cos( a2 ) * radius2 + radius1) * sin( a1 );
			m->verts[k].z = radius2 * sin( a2 );
			k++;
		}
	}
	for ( k = 0, n = 0, i = 0; i < segments; i++ ) {
		for ( j = 0; j < sides; j++ ) {
			m->faces[k * 3] = n + j;
			m->faces[k * 3 + 1] = (n + ((j + 1) % sides)) % m->n_verts;
			m->faces[k * 3 + 2] = (n + ((j + 1) % sides) + sides) % m->n_verts;
			k++;
			m->faces[k * 3] = (n + ((j + 1) % sides) + sides) % m->n_verts;
			m->faces[k * 3 + 1] = (n + j + sides) % m->n_verts;
			m->faces[k * 3 + 2] = n + j;
			k++;
		}
		n += sides;
	}
	return 1;
}

void mesh_rotate( mesh *m, float xa, float ya, float za )
{
	int	i;
	for ( i = 0; i < m->n_verts; i++ ) {
		vector_rotate_x( &m->verts[i], xa );
		vector_rotate_y( &m->verts[i], ya );
		vector_rotate_z( &m->verts[i], za );
		vector_rotate_x( &m->norms[i], xa );
		vector_rotate_y( &m->norms[i], ya );
		vector_rotate_z( &m->norms[i], za );
	}
}

void mesh_render( mesh *m )
{
	int		i, i3;
	tri_vertex	verts_2d[3];
	int		v0, v1, v2;
	vector		n;

	// generate phong uv
	for ( i = 0; i < m->n_verts; i++ ) {
		n = m->norms[i];
		vector_rotate_x( &n, PI / 8.0f );
		vector_rotate_y( &n, -PI / 8.0f );
		m->texuv[i].pu = (n.x + 1.0f) * 128.0f;
		m->texuv[i].pv = (n.y + 1.0f) * 128.0f;
	}

	for ( i = 0; i < m->n_faces; i++ ) {
		i3 = i * 3;
		v0 = m->faces[i3];
		v1 = m->faces[i3 + 1];
		v2 = m->faces[i3 + 2];

		vector_project_2d( &verts_2d[0], &m->verts[v0], FOV, FOV );
		vector_project_2d( &verts_2d[1], &m->verts[v1], FOV, FOV );
		vector_project_2d( &verts_2d[2], &m->verts[v2], FOV, FOV );

		// backface-cull
		{
		float	v1, w1, v2, w2;
		v1 = verts_2d[2].x - verts_2d[1].x;
		w1 = verts_2d[0].x - verts_2d[1].x;
		v2 = verts_2d[2].y - verts_2d[1].y;
		w2 = verts_2d[0].y - verts_2d[1].y;
		if ( (v1 * w2 - v2 * w1) > 0.0f ) continue;
		}

		verts_2d[0].pu = m->texuv[v0].pu;
		verts_2d[0].pv = m->texuv[v0].pv;
		verts_2d[1].pu = m->texuv[v1].pu;
		verts_2d[1].pv = m->texuv[v1].pv;
		verts_2d[2].pu = m->texuv[v2].pu;
		verts_2d[2].pv = m->texuv[v2].pv;

		verts_2d[0].tu = m->texuv[v0].tu;
		verts_2d[0].tv = m->texuv[v0].tv;
		verts_2d[1].tu = m->texuv[v1].tu;
		verts_2d[1].tv = m->texuv[v1].tv;
		verts_2d[2].tu = m->texuv[v2].tu;
		verts_2d[2].tv = m->texuv[v2].tv;

		tri( &verts_2d[0], &verts_2d[1], &verts_2d[2] );
	}
}

void mesh_scale( mesh *m, float sx, float sy, float sz )
{
	int	i;
	for ( i = 0; i < m->n_verts; i++ ) {
		m->verts[i].x *= sx;
		m->verts[i].y *= sy;
		m->verts[i].z *= sz;
	}
}

void mesh_generate_chrome_uv( mesh *m, float max_u, float max_v )
{
	int	i;
	for ( i = 0; i < m->n_verts; i++ ) {
		m->texuv[i].tu = (m->norms[i].x * 0.5f + 0.5f) * max_u;
		m->texuv[i].tv = (m->norms[i].y * 0.5f + 0.5f) * max_v;
	}
}
