// Magic Software, Inc.
// http://www.magic-software.com
// Copyright (c) 2000, All Rights Reserved
//
// Source code from Magic Software is supplied under the terms of a license
// agreement and may not be copied or disclosed except in accordance with the
// terms of that agreement.  The various license agreements may be found at
// the Magic Software web site.  This file is subject to the license
//
// FREE SOURCE CODE
// http://www.magic-software.com/License/free.pdf

#ifndef MGCCURVE_H
#define MGCCURVE_H



template <class Vector>
class MgcCurve
{
public:
    // abstract base class
    MgcCurve (float fTMin, float fTMax);
    virtual ~MgcCurve ();

    // interval on which curve parameter is defined
    float GetMinTime () const;
    float GetMaxTime () const;

    // position and derivatives
    virtual Vector GetPosition (float fTime) const = 0;
    virtual Vector GetFirstDerivative (float fTime) const = 0;
    virtual Vector GetSecondDerivative (float fTime) const = 0;
    virtual Vector GetThirdDerivative (float fTime) const = 0;

    // differential geometric quantities
    Vector GetTangent (float fTime) const;
    float GetSpeed (float fTime) const;
    virtual float GetLength (float fT0, float fT1) const = 0;
    float GetTotalLength () const;

    // inverse mapping of s = Length(t) given by t = Length^{-1}(s)
    virtual float GetTime (float fLength, int iIterations = 32,
        float fTolerance = 1e-06) const = 0;

    // subdivision
    void SubdivideByTime (int iNumPoints, Vector*& rakPoint) const;
    void SubdivideByLength (int iNumPoints, Vector*& rakPoint) const;

    // Subdivision by variation. The pointers pkP0 and pkP1 correspond to the
    // curve points at fT0 and fT1.  If the pointer values are not null, the
    // assumption is that the caller has passed in the curve points.
    // Otherwise, the function computes the curve points.
    virtual float GetVariation (float fT0, float fT1,
        const Vector* pkP0 = 0, const Vector* pkP1 = 0) const = 0;
    void SubdivideByVariation (float fMinVariation,
        unsigned int uiMaxLevel, int& iNumPoints, Vector*& rakPoint) const;

protected:
    // curve parameter is t where tmin <= t <= tmax
    float m_fTMin, m_fTMax;

    // subdivision
    class PointList
    {
    public:
        PointList (const Vector& rkPoint)
        {
            m_kPoint = rkPoint;
            m_kNext = 0;
        }

        Vector m_kPoint;
        PointList* m_kNext;
    };

    void SubdivideByVariation (float fT0, const Vector& rkP0,
        float fT1, const Vector& rkP1, float kMinVariation,
        unsigned int uiLevel, int& iNumPoints, PointList*& rpkList) const;
};

#include "MgcCurve.inl"

#endif
