<?php

/**
 * mystic_library.php : A set of functions to access mystic bbs data files
 *
 * @package mystic_library
 * @author $Author: stimpy $
 * @originalauthor $Originalauthor: frank $
 * @copyright $Copyright 2011 Frank Linhares/netsurge%demonic%bbs-scene.org$
 * @version $Revision: 11 $
 * @lastrevision $Date: 2019-03-12 11:37:17 +0100 (Tue, 12 Mar 2019) $
 * @modifiedby $LastChangedBy: stimpy $
 * @filesource $URL: http://miserybbs.com/svn/miserybbs/scripts/mystic_library.php $
 * @modifiedsource $URL: https://gitlab.ambhost.net/stimpy/phplib_mystic $
*/
 
/**
 * mystic_library.php
 *
 * @package mystic_library
 *
 * This library is a collection of functions that are used to access and parse 
 * various data file from Mystic BBS 1.11+
 * A description of each function and what they return is listed in the doc 
 * block for each function.
 * 
 * This library was originaly written for MysticBBS v1.09 by Frank Linhares 
 * in 2011 and updated to MysticBBS 1.11 and object oriented by 
 * Philipp Giebel in 2017, finally updated to MysticBBS v1.12 in 2019 by 
 * Philipp Giebel again.
 * This is still work in progres - all the functions are at various states of
 * decay - some are pretty foolproof and up-to-date, some are barely running
 * at all.. Just try it out yourself and don't blame me, if your cat blows up..
 * 
 * Include this library on all pages that will be showing data from Mystic BBS 
 * like this:
 * 		include 'inc/mystic_library.php';
 * 
 * Initialize class like this:
 *      $mystic = new mystic('/path/to/your/datafiles/', <mystic version>);
 *  eg:
 *      $mystic = new mystic('/mystic/data/', 1.12);
 *
 *  supported mystic versions: 1.11 and 1.12
 *
 * Official support for mystic_php_library could once be found at the official 
 * support bbs, miserybbs.com
 * As of December 2017, this site is offline.
 * Inofficial support for my updated version is, as of December 2017, available
 * at: https://gitlab.ambhost.net/stimpy/phplib_mystic
 * 
 * Basic help and info is written on top of every function in this file. *
 *
**/
  
  class mystic {
    private $data_path = './';
    private $version = 1.12;
    function __construct( $data_path, $version=1.12 ) {
      $this->setDatapath( $data_path );
      $this->setVersion( $version );
    }
    function __destruct() {
    }
    function setDatapath( $data_path ) {
      $this->data_path = $data_path;
    }
    function setVersion( $version ) {
      $this->version = $version;
    }
    function getVersion() {
      return $this->version;
    }

/**
 * read and parse last 10 callers from Mystic
 *
 * @uses: mystic::lastcallers(  number of last callers $int, 
 *                              true to parse pipe colours or false to strip pipe colours $bool )
 *
 * @return: an array containing the following keys identifying the last 
 *          X callers; where X is set by $number. If no number is specified it 
 *          will display the the last ten callers:
 *
 *          date = unix timestamp of call
 *          new = 1 if new user, 0 if not
 *          ip = ip address of caller
 *          host = hostname of caller
 *          node = node number
 *          caller = caller number (total)
 *          user = username/handle
 *          city = from userprofile
 *          address = from userprofile
 *          gender = F=Female/M=Male
 *          email = from userprofile
 *          info = from userprofile
 *          opt1 = from userprofile
 *          opt2 = from userprofile
 *          opt3 = from userprofile
 *
 * @author: Philipp Giebel
 * @originalauthor: Frank Linhares
**/
    function lastcallers( $number = 10, $pipe = false ) {

      $lcallers = NULL;
      if ( $number > 10 ) $number = 10;
      $record_length = 896;

      if ( file_exists( $this->data_path .'callers.dat' ) ) {
        $fp = fopen( $this->data_path .'callers.dat', 'rb' );

        for ( $i = 0; $i < $number; $i++ ) {
          $data = fread( $fp, $record_length );

          if ( $data != '' ) {
            // Create a data structure
            $data_format =  'ldate/'.         # Get the date
                            'Cnew/'.          # Get new user flag
                            'Ciplen/'.        # Get the length of the ip address field
                            'A15ip/'.         # Get the ip address
                            'Chostlen/'.      # Get the length of the host name field
                            'A50host/'.       # Get the host name
                            'Cnode/'.         # Get the node number
                            'lcaller/'.       # Get the caller number
                            'Cuserlen/'.      # Get the length of the user field
                            'A30user/'.       # Get the username (30) padded with null
                            'Ccitylen/'.      # Get the length of the city field
                            'A25city/'.       # Get the city (25) padded with null
                            'Caddresslen/'.   # Get the length of the address field
                            'A30address/'.    # Get the address (30) padded with null      
                            'Cgender/'.       # Get the gender
                            'Cemaillen/'.     # Get the length of the email field
                            'A35email/'.      # Get the email (35) padded with null
                            'Cinfolen/'.      # Get the length of the usernote field
                            'A30info/'.       # Get the usernote (30) padded with null
                            'Copt1len/'.      # Get the length of the opt1 field
                            'A35opt1/'.       # Get the opt1 field (35) padded with null
                            'Copt2len/'.      # Get the length of the opt2 field
                            'A35opt2/'.       # Get the opt2 field (35) padded with null
                            'Copt3len/'.      # Get the length of the opt3 field
                            'A35opt3/';       # Get the opt3 field (35) padded with null

            // Unpack the data structure
            $lcallers[] = unpack( $data_format, $data );
          }
        }

        if ( isset( $lcallers ) ) {
          $lcallers = $this->lengthfix( $lcallers );

          foreach ( $lcallers as &$dfix ) {
            // change date from dos to unix and make human readable
            $dfix['date'] = $this->dos2unixtime($dfix['date']);
          }

          $lcallers = $this->pipe2span( $lcallers, $pipe );

          rsort($lcallers);
        }
      }
      return $lcallers;

    }

    /**
     * read and parse oneliners from Mystic
     *
     * @uses: mystic::oneliner( number of shouts $int, 
     *                          true to parse pipe colours or false to strip pipe colours $bool )
     *
     * @return: an array containing the following keys identifying the last 
     *          X callers; where X is set by $number. If no number is specified 
     *          it will display the the last ten callers:
     *
     *       text = line of text
     *       from = author of that line
     *
     * @author: Philipp Giebel
    **/

    function oneliner( $number = 10, $pipe = false ) {

      $oneliner = NULL;
      if ( $number > 10 ) $number = 10;
      $record_length = 111;

      if ( file_exists( $this->data_path .'oneliner.dat' ) ) {
        $fp = fopen( $this->data_path .'oneliner.dat', 'rb' );

        for ( $i = 0; $i < $number; $i++ ) {
          $data = fread ($fp, $record_length);
          if ( $data != '' ) {
            // Create a data structure
            $data_format =  'Ctextlen/'.    # Get the length of the text field
                            'A79text/'.     # Get the text (79) padded with null
                            'Cfromlen/'.    # Get the length of the from field
                            'A30from/';     # Get the from field (30) padded with null

            // Unpack the data structure
            $oneliner[] = unpack( $data_format, $data );
          }
        }
        if ( isset( $oneliner ) ) {
          $oneliner = $this->lengthfix( $oneliner );
          $oneliner = $this->pipe2span( $oneliner, $pipe );
        }
      }

      return $oneliner;
    }

    /**
     * read and parse oneliners from Mystic
     *
     * @uses: mystic::files(  name of the file area $var, 
     *                        number of records to show $int, 
     *                        offset records $int,
     *                        true to parse pipe colours or false to strip pipe colours $bool )
     *
     * @return: an array of arrays containing the following keys
     *
     *        filename  = name of file
     *        size      = size of the file
     *        datetime  = unix timestamp of upload date and time
     *        uploader  = uploader of the file
     *        flags     = flags of the file
     *        downloads = downloads of the file
     *        rating    = rating of the file
     *        desc      = description of the file
     *
     * @author: Philipp Giebel
    **/

    function files( $area, $number = 10, $start = 0, $pipe = false ) {

      $res = NULL;
      if ( ( file_exists( $this->data_path.$area.'.dir' ) ) AND ( file_exists( $this->data_path.$area.'.des' ) ) ) {

        if ($number > 10) $number = 10;

        $record_length = 121;

        $fp = fopen( $this->data_path.$area.'.dir', 'r' );
        $fp2 = fopen( $this->data_path.$area.'.des', 'r' );

        if ( $start > 0 ) {
          if ( $start*$record_length > filesize( $this->data_path . $area .'.dir' ) ) return false;
          fseek( $fp, $start*$record_length );
        }

        for ( $i = 0; $i < $number; $i++ ) {
          $data = fread( $fp, $record_length );

          if ( $data != '' ) {
            $data_format =  'Cfilenamelen/'.
                            'A70filename/'.
                            'lsize/'.
                            'ldatetime/'.
                            'Cuploaderlen/'.
                            'A30uploader/'.
                            'Cflags/'.
                            'ldownloads/'.
                            'Crating/'.
                            'ldescptr/'.
                            'Cdesclines/';

            $files[] = unpack ($data_format, $data);
          }
        }

        if ( isset( $files ) ) {
          $files = $this->lengthfix( $files );

          foreach ( $files as &$fix ) {
            // change date from dos to unix and make human readable
            $fix['datetime'] = $this->dos2unixtime( $fix['datetime'] );
            fseek( $fp2, $fix['descptr'] );
            $d = array();
            for ( $i = 1; $i <= $fix['desclines']; $i++ ) {
              $l = fgets($fp2, 2);
              $j = fgets( $fp2, ord($l)+1 );
              $j = preg_replace( '/\[[0-9;]*m/', '', $j );
              $j = preg_replace( '/SAUCE00.*/', '', $j );
              $j = preg_replace_callback( '/\[([0-9]*)C/', function( $treffer ) { return str_repeat( ' ', intval( $treffer[1] ) ); }, $j );
              $d[] = iconv( 'cp437', 'utf-8', $j );
            }
            $fix['desc'] = $d;
            if ( $fix['flags'] == 0 ) $res[] = $fix;
          }

          if ( count( $res ) > 0 ) {
            $res = $this->pipe2span( $res, $pipe );
          }
        }

        fclose($fp);
        fclose($fp2);
      }

      return $res;
    }

    /**
     * read and parse oneliners from Mystic
     *
     * @uses: mystic::fbases( number of records to show $int, 
     *                        offset records $int,
     *                        true to parse pipe colours or false to strip pipe colours $bool )
     *
     * @return: an array of arrays containing the following keys
     *
     *       index      = unique index of the filebase
     *       name       = name of the filebase
     *       ftpname    = ftpname of the filebase
     *       filename   = filename of the data file
     *       dispfile   = displayfile of the filebase
     *       template   = template of the filebase
     *       listacs    = ACS needed to list contents
     *       ftpacs     = ACS needed for ftp access
     *       dlacs      = ACS needed to download
     *       ulacs      = ACS needed to upload
     *       commentacs = ACS needed for commenting
     *       sysopacs   = ACS needed for sysop access
     *       path       = path to files
     *       defscan    = scan this filebase?
     *       flags      = flags of the filebase
     *       created    = creation date of the filebase as unix timestamp
     *       netaddr    = network address (id?)
     *       echotag    = echotag for the filebase
     *
     * @author: Philipp Giebel
    **/

    function fbases( $number = 10, $start = 0, $pipe = false ) {

      $bases = NULL;
      if ($number > 10) $number = 10;

      if ( $this->version >= 1.12 ) {
        $record_length = 640;
				$data_format =  'Sindex/'.
												'Cnamelen/'.
												'A60name/'.
												'Cftpnamelen/'.
												'A60ftpname/'.
												'Cfilenamelen/'.
												'A40filename/'.
												'Cdispfilelen/'.
												'A20dispfile/'.
												'Ctemplatelen/'.
												'A20template/'.
												'Clistacslen/'.
												'A30listacs/'.
												'Cftpacslen/'.
												'A30ftpacs/'.
												'Cdlacslen/'.
												'A30dlacs/'.
												'Culacslen/'.
												'A30ulacs/'.
												'Chatchacslen/'.
												'A30hatchacs/'.
												'Csysopacslen/'.
												'A30sysopacs/'.
												'Cpasseacslen/'.
												'A30passeacs/'.
												'Cpathlen/'.
												'A80path/'.
												'Cdefscan/'.
												'lflags/'.
												'lcreated/'.
												'C1netaddr/'.
												'C1echotaglen/'.
												'A30echotag/'.
												'Clisteacslen/'.
												'A30listeacs/';
      } else {
        $record_length = 495;
				$data_format =  'Sindex/'.
												'Cnamelen/'.
												'A40name/'.
												'Cftpnamelen/'.
												'A60ftpname/'.
												'Cfilenamelen/'.
												'A40filename/'.
												'Cdispfilelen/'.
												'A20dispfile/'.
												'Ctemplatelen/'.
												'A20template/'.
												'Clistacslen/'.
												'A30listacs/'.
												'Cftpacslen/'.
												'A30ftpacs/'.
												'Cdlacslen/'.
												'A30dlacs/'.
												'Culacslen/'.
												'A30ulacs/'.
												'Ccommentacslen/'.
												'A30commentacs/'.
												'Csysopacslen/'.
												'A30sysopacs/'.
												'Cpathlen/'.
												'A80path/'.
												'Cdefscan/'.
												'lflags/'.
												'lcreated/'.
												'C1netaddr/'.
												'C1echotaglen/'.
												'A30echotag/';
      }

      $fp = fopen( $this->data_path .'fbases.dat', 'r' );
      if ( $start > 0 ) {
        if ( $start*$record_length > filesize($this->data_path.'fbases.dat') ) return false;
        fseek( $fp, $start*$record_length );
      }

      for ($i = 0; $i < $number; $i++) {
        $data = fread ($fp, $record_length);
        if ( $data != false ) {
          $bases[] = unpack( $data_format, $data );
        }
      }
      if ( isset( $bases ) ) {
        $bases = $this->lengthfix( $bases );
        foreach ( $bases as &$fix ) {
          $fix['created'] = $this->dos2unixtime ( $fix['created'] );
        }
      }
      fclose( $fp );
      return $bases;
    }

    /**
     * read and parse Mystic's chat(#).dat file
     *
     * @uses: mystic::chat( number of nodes you are running $int,
     *                      true to parse pipe colours or false to strip pipe colours $bool )
     *
     * @return: an array containing the following keys identifying 
     *          user information from each node. 
     *
     * 		   active = is anyone on the node. 1 = Yes, 0 = No
     *		   name = username/handle
     *		   action = user's current action
     *		   location = user's city and state
     *		   gender = user's gender. M = Male, F = Female
     *		   age = user's age
     *		   baud = user's connecting baud rate
     *		   invisible = user's invisibility status
     *		   available = user's availability status
     *		   inchat = is the user in multi-node chat. 1 = Yes, 0 = No
     *		   room = which multi-node chat room is the user in
     *
     * @author: Philipp Giebel
     * @originalauthor: Frank Linhares
     **/

    function chat( $nodes = 1, $pipe = false ) {	

      $mystic_chat = NULL;
      // loop through the number of nodes with node1.dat, node2.dat, etc..
      for ( $i = 1; $i <= $nodes; $i++ ) {
        if ( file_exists( $this->data_path .'chat'. $i .'.dat' ) ) {
          $data = file_get_contents( $this->data_path .'chat'. $i .'.dat' );
          // Create a data structure
          $data_format =  'Cactive/'.         # Get the date
                          'Cnamelen/'.        # Get the length of the name field 
                          'A30name/'.         # Get the username (30) padded with null
                          'Cactionlen/'.      # Get the length of the action field 
                          'A40action/'.       # Get the user's current action (40) padded with null		
                          'Clocationlen/'.    # Get the length of the location field 
                          'A30location/'.     # Get the user's city and state (40) padded with null			
                          'Agender/'.         # Get the gender (m for male f for female)
                          'Cage/'.            # Get the users age
                          'Cbaudlen/'.        # Get the length of the baud field 
                          'A6baud/'.          # Get the baud rate (6) padded with null
                          'cinvisible/'.      # Check if the user is invisible
                          'cavailable/'.      # Check if the user is available		
                          'cinchat/'.         # Check if the user is in multi-node chat		
                          'Croom/';           # If in multi-node chat which room

          // Unpack the data structure
          $mystic_chat[] = unpack( $data_format, $data );
        }
      }

      if ( isset( $mystic_chat ) ) {
        // inject node number into array	
        for ( $i = 0; $i <= $nodes; $i++ ) {
          $mystic_chat[$i]['node']=$i+1;
        }

        array_pop( $mystic_chat );
        $mystic_chat = $this->lengthfix( $mystic_chat );

        // check if node is active, if not clear name and set action to "waiting for caller" 	
        foreach ( $mystic_chat as &$wfix ) {
          if ( $wfix['active'] == 0 ) {
            $wfix['name'] = "";
            $wfix['action'] = 'waiting for caller';
          }

          // don't allow users who want to be invisible to be displayed.		
          if ( ( $wfix['active'] == 1 ) AND ( $wfix['invisible'] == 1 ) ) {
            $wfix['name'] = "";
            $wfix['action'] = 'waiting for caller';
          }
        }

        $mystic_chat = $this->pipe2span( $mystic_chat );
      }

      return $mystic_chat;
    }

    /**
     * read and parse Mystic's history.dat file
     *
     * @uses: mystic::history(  number of records to show $int )
     * 
     * @return: an array containing the following keys identifying system stats. 
     *
     * 		   date = Date
     *		   emails = number of emails sent today
     *		   posts = number of posts today
     *		   downloads = number of downloads today
     *		   uploads = number of uploads today
     *		   dlkb = number of kilobytes downloaded today
     *		   ulkb = number of kilobytes uploaded today
     *		   calls = total number of calls today
     *		   newusers = number of new users today
     *		   telnet = number of telnet connections today
     *		   ftp = number of ftp connections today
     *		   pop = number of pop3 connections today
     *		   smtp = number of smtp connections today
     *		   nntp = number of nntp connections today
     *		   http = number of http connections today
     *
     * @author: Philipp Giebel
     * @originalauthor: Frank Linhares
     **/

    function history( $number = 10 ) {	

      if ( $number > 10 ) $number = 10;
      
      // get file size in order to determine how many records are stored
      $filesize = filesize( $this->data_path .'history.dat' );
      $record_length = 64;

      // divide file size by record length to determine number of records stored
      $record_number = $filesize / $record_length;
      if ( $number > $record_number ) $number = $record_number;
      
      // Open the mystic BBS data file in binary mode
      $fp = fopen( $this->data_path.'history.dat', 'rb' );

      for ( $i = $record_number; $i > $record_number-$number; $i-- ) {
        $data = fread( $fp, $record_length );
        if ( isset( $data ) ) {
          /* Create a data structure */
          $data_format =  'ldate/'.         # Get the date
                          'Semails/'.       # Get the number of emails sent
                          'Sposts/'.        # Get the number of posts
                          'Sdownloads/'.    # Get the number of downloads
                          'Suploads/'.      # Get the number of uploads
                          'ldlkb/'.         # Get the number of downloaded kb
                          'lupkb/'.         # Get the number of uploaded kb
                          'lcalls/'.        # Get the number of calls
                          'Snewusers/'.     # Get the number of new users
                          'Stelnet/'.       # Get the number of telnet connections
                          'Sftp/'.          # Get the number of ftp connections
                          'Spop/'.          # Get the number of pop3 connections
                          'Ssmtp/'.         # Get the number of smtp connections
                          'Snntp/'.         # Get the number of nntp connections
                          'Shttp/';         # Get the number of http connections

          /* Unpack the data structure */
          $myshistory[] = unpack( $data_format, $data );
        }
      }  	   

      if ( isset( $myshistory ) ) {
        foreach ( $myshistory as &$dfix ) {
          // change date from dos to unix and make human readable
          $dfix['date'] = $this->dos2unixtime( $dfix['date'] );
        }
      }

      return $myshistory;
    }

    /**
     * read and parse Mystic users.dat file.
     * This function is missing some user data, which still has to be reverse
     * engineered, since the provided info from "docs/records.110" is outdated.
     * 
     * @uses: mystic::userlist( number of records to show $int, 
     *                          offset records $int,
     *                          true to parse pipe colours or false to strip pipe colours $bool )
     * 
     * @return: an array containing the following keys identifying users with accounts on the bbs. 
     *
     * 		  permidx       = Permission index
     *        validated     = unix timestamp of validation date
     *        handle        = username / handle
     *        realname      = real name as provided by user
     *        password      = password (unencrypted!)
     *        address       = address as provided by user
     *        city          = city as provided by user
     *        zipcode       = zipcode as provided by user
     *        homephone     = home phone number as provided by user
     *        dataphone     = data phone number as provided by user
     *        date          = birthdate as provided by user
     *        gender        = gender (M/F) as provided by user
     *        email         = email address as provided by user
     *        info          = additional information provided by user
     *        opt1-opt10    = optional data fields as provided by user
     *        theme         = theme setting (name)
     *        expires       = unix timestamp of account expiration date
     *        expiresto     = userlevel to fall back on expiration
     *        lastpwchange  = date of last pw change (no timestamp! yet..)
     *        startmenu     = user's start menu (first menu after login)
     *        archive       = user's default archive format setting
     *        security      = userlevel
     *        screensize    = user's screen length setting
     *        peerip        = ip at last login
     *        peerhost      = hostname at last login
     *        firston       = unix timestamp of first login
     *        laston        = unix timestamp of last login
     *        calls         = total calls
     *        callstoday    = calls today
     *        dls           = total number of downloads
     *        dlstoday      = number of downloads today
     *        dlk           = total kb downloaded
     *        dlktoday      = kb downloaded today
     *        uls           = total number of uploads
     *        ulk           = kb uploaded
     *        posts         = total number of posts made
     *        emails        = total number of "emails" sent
     *        timeleft      = minutes left today
     *        timebank      = minutes at the timebank
     *        fileratings   = total number of file ratings
     *        filecomment   = total number fo file comments
     *        lastfbase     = last joined filebase
     *        lastfgroup    = last joined file group (maybe wrong)
     *        lastmbase     = last joined messagebase
     *        lastmgroup    = last joined message group (maybe wrong)
     * 
     * The rest is missing for now and needs further reverse engineering...
     * These are supposedly the missing variables with hopefuly correct type
     * but in no particular order:
     *                   'Cedittype/'.
     *                   'Cfilelist/'.
     *                   'Csiguse/'.
     *                   'lsigoffset/'.
     *                   'Csiglength/'.
     *                   'Chotkeys/'.
     *                   'Cmreadtype/'.
     *                   'Cuselbindex/'.
     *                   'Cuselbquote/'.
     *                   'Cuselbmidc/'.
     *                   'Cusefullchat/'.
     *                   'lcredits/'.
     *                   'Cprotocol/'.
     *                   'Ccodepage/'.
     *                   'Cqwknetwork/'.
     *                   'Cemailvalcodelen/'.
     *                   'A8emailvalcode/'.
     *                   'lemailvalsend/'.
     *                   'lemailvaldate/'.
     *                   'A26af1/'.
     *                   'A26af2/'.
     *                   'A20vote/'.
     *                   'Cqwkfiles/'.
     *                   'Cdatatype/'.
     *                   'Cscreencols/';
     *
     * @author: Philipp Giebel
     * @originalauthor: Frank Linhares
     **/

    function userlist( $number = 10, $start = 0, $pipe = false ) {	

      if ( $number > 10 ) $number = 10;
      
      // get file size in order to determine how many records are stored
      $filesize = filesize( $this->data_path .'users.dat' ); 

      if ( $this->version >= 1.12 ) {
        $record_length = 1536;
        $data_format =  'lpermidx/'.        # Get the permission index
                        'lflags/'.          # Get the flags (longint)
                        'Cpasswordlen/'.    # Get the length of the password field 
                        'A100password/'.    # Get the password (100)
                        'Chandlelen/'.      # Get the length of the handle field 
                        'A30handle/'.       # Get the handle (30) padded with null
                        'Crealnamelen/'.    # Get the length of the realname field 
                        'A30realname/'.     # Get the real name (30) padded with null      
                        'Caddresslen/'.     # Get the length of the address field 
                        'A30address/'.      # Get the address (30) padded with null
                        'Ccitylen/'.        # Get the length of the city field 
                        'A25city/'.         # Get the city (25) padded with null
                        'Czipcodelen/'.     # Get the length of the zipcode field 
                        'A9zipcode/'.       # Get the zipcode (9) padded with null
                        'Chomephonelen/'.   # Get the length of the homephone field 
                        'A15homephone/'.    # Get the home phone (15) padded with null
                        'Cdataphonelen/'.   # Get the length of the dataphone field 
                        'A15dataphone/'.    # Get the dataphone (15) padded with null
                        'lbdate/'.          # Get the birth date
                        'Agender/'.         # Get the gender (m for male f for female)
                        'Cemaillen/'.       # Get the length of the email field 
                        'A60email/'.        # Get the email address (60) padded with null
                        'Copt1len/'.        # Get the length of the opt1 field 
                        'A60opt1/'.         # Get the opt1 field (60) padded with null
                        'Copt2len/'.        # Get the length of the opt2 field
                        'A60opt2/'.         # Get the opt2 field (60) padded with null
                        'Copt3len/'.        # Get the length of the opt3 field
                        'A60opt3/'.         # Get the opt3 field (60) padded with null
                        'Copt4len/'.        # Get the length of the opt3 field
                        'A60opt4/'.         # Get the opt4 field (60) padded with null
                        'Copt5len/'.        # Get the length of the opt3 field
                        'A60opt5/'.         # Get the opt5 field (60) padded with null
                        'Copt6len/'.        # Get the length of the opt3 field
                        'A60opt6/'.         # Get the opt6 field (60) padded with null
                        'Copt7len/'.        # Get the length of the opt3 field
                        'A60opt7/'.         # Get the opt7 field (60) padded with null
                        'Copt8len/'.        # Get the length of the opt3 field
                        'A60opt8/'.         # Get the opt8 field (60) padded with null
                        'Copt9len/'.        # Get the length of the opt3 field
                        'A60opt9/'.         # Get the opt9 field (60) padded with null
                        'Copt10len/'.       # Get the length of the opt3 field
                        'A60opt10/'.        # Get the opt10 field (60) padded with null
                        'Cinfolen/'.        # Get the length of the info field
                        'A30info/'.         # Get the usernote (30) padded with null
                        'Cthemelen/'.
                        'A20theme/'.
                        'A8foo/'.
                        'Csecurity/'.
                        'lexpires/'.
                        'Cexpiresto/'.
                        'Clastpwchangelen/'.
                        'llastpwchange/'.
                        'Cstartmenulen/'.
                        'A20startmenu/'.
                        'Carchivelen/'.
                        'A4archive/'.
                        'Cqwkfiles/'.
                        'Cdatetype/'.
                        'Cscreensize/'.
                        'Cscreencols/'.
                        'Cpeeriplen/'.
                        'A45peerip/'.
                        'Cpeerhostlen/'.
                        'A80peerhost/'.
                        'Cpeercountrylen/'.
                        'A60peercountry/'.
                        'lfirston/'.        # Get the first on date
                        'llaston/'.         # Get the last on date
                        'lcalls/'.          # Get the number of calls
                        'scallstoday/'.     # Get the # of calls today
                        'sdls/'.            # Get the # of downloads
                        'sdlstoday/'.       # Get the # of downloads today  
                        'ldlk/'.            # Get the total downloads in k
                        'ldlktoday/'.       # Get the total downloads in k today 
                        'luls/'.            # Get the total # of uploads
                        'lulk/'.            # Get the total uploads in k
                        'lposts/'.          # Get the total posts
                        'lemails/'.         # Get the total sent emails
                        'ltimeleft/'.       # Get the amount of time left today
                        'stimebank/'.
                        'Clastfbase/'.
                        'Clastmbase/'.
                        'Clastmgroup/'.
                        'Clastfgroup/';
      } else {
        $record_length = 1536;
        $data_format =  'lpermidx/'.        # Get the permission index
                        'lvalidated/'.      # Get the flags (longint)
                        'Chandlelen/'.      # Get the length of the handle field 
                        'A30handle/'.       # Get the handle (30) padded with null
                        'Crealnamelen/'.    # Get the length of the realname field 
                        'A30realname/'.     # Get the real name (30) padded with null      
                        'Cpasswordlen/'.    # Get the length of the password field 
                        'A15password/'.     # Get the password (15) padded with null
                        'Caddresslen/'.     # Get the length of the address field 
                        'A30address/'.      # Get the address (30) padded with null
                        'Ccitylen/'.        # Get the length of the city field 
                        'A25city/'.         # Get the city (25) padded with null
                        'Czipcodelen/'.     # Get the length of the zipcode field 
                        'A9zipcode/'.       # Get the zipcode (9) padded with null
                        'Chomephonelen/'.   # Get the length of the homephone field 
                        'A15homephone/'.    # Get the home phone (15) padded with null
                        'Cdataphonelen/'.   # Get the length of the dataphone field 
                        'A15dataphone/'.    # Get the dataphone (15) padded with null
                        'lbdate/'.          # Get the birth date
                        'Agender/'.         # Get the gender (m for male f for female)
                        'Cemaillen/'.       # Get the length of the email field 
                        'A60email/'.        # Get the email address (60) padded with null
                        'Cinfolen/'.        # Get the length of the info field
                        'A30info/'.         # Get the usernote (30) padded with null
                        'Copt1len/'.        # Get the length of the opt1 field 
                        'A60opt1/'.         # Get the opt1 field (60) padded with null
                        'Copt2len/'.        # Get the length of the opt2 field
                        'A60opt2/'.         # Get the opt2 field (60) padded with null
                        'Copt3len/'.        # Get the length of the opt3 field
                        'A60opt3/'.         # Get the opt3 field (60) padded with null
                        'Copt4len/'.        # Get the length of the opt3 field
                        'A60opt4/'.         # Get the opt4 field (60) padded with null
                        'Copt5len/'.        # Get the length of the opt3 field
                        'A60opt5/'.         # Get the opt5 field (60) padded with null
                        'Copt6len/'.        # Get the length of the opt3 field
                        'A60opt6/'.         # Get the opt6 field (60) padded with null
                        'Copt7len/'.        # Get the length of the opt3 field
                        'A60opt7/'.         # Get the opt7 field (60) padded with null
                        'Copt8len/'.        # Get the length of the opt3 field
                        'A60opt8/'.         # Get the opt8 field (60) padded with null
                        'Copt9len/'.        # Get the length of the opt3 field
                        'A60opt9/'.         # Get the opt9 field (60) padded with null
                        'Copt10len/'.       # Get the length of the opt3 field
                        'A60opt10/'.        # Get the opt10 field (60) padded with null
                        'Cthemelen/'.
                        'A30theme/'.
                        'Cexpireslen/'.
                        'A8expires/'.
                        'Cexpiresto/'.
                        'Clastpwchangelen/'.
                        'A8lastpwchange/'.
                        'Cstartmenulen/'.
                        'A20startmenu/'.
                        'Carchivelen/'.
                        'A4archive/'.
                        'ssecurity/'.
                        'sscreensize/'.
                        'Cpeeriplen/'.
                        'A20peerip/'.
                        'Cpeerhostlen/'.
                        'A50peerhost/'.
                        'lfirston/'.        # Get the first on date
                        'llaston/'.         # Get the last on date
                        'lcalls/'.          # Get the number of calls
                        'scallstoday/'.     # Get the # of calls today
                        'sdls/'.            # Get the # of downloads
                        'sdlstoday/'.       # Get the # of downloads today  
                        'ldlk/'.            # Get the total downloads in k
                        'ldlktoday/'.       # Get the total downloads in k today 
                        'luls/'.            # Get the total # of uploads
                        'lulk/'.            # Get the total uploads in k
                        'lposts/'.          # Get the total posts
                        'lemails/'.         # Get the total sent emails
                        'ltimeleft/'.       # Get the amount of time left today
                        'stimebank/'.
                        'lfileratings/'.
                        'lfilecomment/'.
                        'Clastfbase/'.
                        'Clastfgroup/'.
                        'Clastmbase/'.
                        'Clastmgroup/';
      }

      // divide file size by record length to determine number of records stored
      $record_number = $filesize / $record_length;
      
      if ( $start > $record_number ) return false;
      
      if ( $start+$number > $record_number ) $number = $record_number-$start;
      
      // Open the mystic BBS data file in binary mode 
      $fp = fopen( $this->data_path .'users.dat', 'rb' );

      if ( $start > 0 ) {
        if ( $start*$record_length > filesize($this->data_path.'users.dat') ) return false;
        fseek( $fp, $start*$record_length );
      }

      for ($i = 0; $i < $number; $i++) {
        $data = fread ($fp, $record_length);
        $users[] = unpack( $data_format, $data );
      }  

      $users = $this->lengthfix( $users );

      foreach ( $users as &$dfix ) {		
        // change date from dos to unix and make human readable
        $dfix['firston'] = $this->dos2unixtime( $dfix['firston'] );
        $dfix['laston'] = $this->dos2unixtime( $dfix['laston'] );
        $dfix['bdate'] = jdtogregorian( $dfix['bdate'] );
      }

      $users = $this->pipe2span( $users, $pipe );

      return $users;
    }

    /**
     * convert mystic's dos julien based time to unix time
     * 
     * @uses: mystic::dos2unixtime( dostime $var )
     * 
     * @return: unix time 
     *
     * @author: Frank Linhares
     **/

    function dos2unixtime( $dostime ) {
      $sec  = 2 * ( $dostime & 0x1f );
      $min  = ( $dostime >> 5 ) & 0x3f;
      $hrs  = ( $dostime >> 11 ) & 0x1f;
      $day  = ( $dostime >> 16 ) & 0x1f;
      $mon  = ( ( $dostime >> 21 ) & 0x0f );
      $year = ( ( $dostime >> 25 ) & 0x7f ) + 1980;
      return mktime( $hrs, $min, $sec, $mon, $day, $year );
    }

    /**
     * calculate a user's birthday
     * 
     * @uses: mystic::age(birthday $var)
     * 
     * @return: age
     *
     * @author: Frank Linhares
     **/

    function age($birthday) {
      return floor( ( time() - strtotime( $birthday ) ) / 31556926 );
    }

    /**
     * fix length of fields having a length definition
     * 
     * @uses: mystic::lengthfix( one or more raw mystic data records $array )
     * 
     * @return: fixedarray
     *
     * @author: Philipp Giebel
     **/
    function lengthfix( $input ) {
      // fix length of variables
      foreach ( $input as &$lengthfix ) {
        foreach ( array_keys( $lengthfix ) as $key ) {
          if ( array_key_exists( $key .'len', $lengthfix ) ) {
            $lengthfix[$key] = substr( $lengthfix[$key], 0, $lengthfix[$key.'len'] );
            unset( $lengthfix[$key.'len'] );
          }
        }
      }
      return $input;
    }

    /**
     * Strip or rewrite pipe codes to html span elements
     * 
     * @uses: mystic::pipe2span(  one or more raw mystic data records $array,
     *                            true to replace with html or false to strip pipe codes $bool )
     * 
     * @return: fixedarray
     *
     * @author: Philipp Giebel
     **/
    function pipe2span( $input, $pipe = false ) {
      // convert pipe codes to proper colours
      if ( $pipe === true ) {
        foreach ( $input as &$dfix ) {
          foreach ( $dfix as &$d ) {
            for ( $i = 0; $i <= 24; $i++ ) {
              $d = str_replace( "|". sprintf( "%02d", $i ), "<span class=\"pipe_". sprintf( "%'.02d\n", $i ) ."\">", $d );
            }
          }
        }
      } else {
        foreach ( $input as &$dfix ) {
          foreach ( $dfix as &$d ) {
            for ( $i = 0; $i <= 24; $i++ ) {
              $d = str_replace( "|". sprintf( "%02d", $i ), "", $d );
            }
          }
        }
      }
      return $input;
    }
  }
?>
