var TileSpawner = pc.createScript('tileSpawner');

/**
 * 1) Entité personnage (la fille).
 */
TileSpawner.attributes.add('character', {
    type: 'entity',
    title: 'Character Entity',
    description: 'Entité du personnage qui marche.'
});

/**
 * 2) Le bloc/pavé à cloner (un seul).
 */
TileSpawner.attributes.add('blockPrefab', {
    type: 'entity',
    title: 'Block Prefab',
    description: 'Entité (cube, par ex.) qui sera clonée.'
});

/**
 * 3) Distance en avant du personnage pour faire apparaitre le pavé.
 */
TileSpawner.attributes.add('spawnDistance', {
    type: 'number',
    default: 2,
    title: 'Distance de spawn',
    description: 'Distance en avant du personnage où le pavé apparaîtra.'
});

/**
 * 4) Hauteur finale du pavé (lorsqu’il a fini de sortir du sol).
 */
TileSpawner.attributes.add('spawnHeight', {
    type: 'number',
    default: 0,
    title: 'Hauteur finale'
});

/**
 * 5) Distance minimale parcourue avant de faire apparaitre un nouveau pavé.
 */
TileSpawner.attributes.add('distanceBetweenBlocks', {
    type: 'number',
    default: 1,
    title: 'Distance entre pavés'
});

/**
 * 6) Profondeur de départ (valeur négative) pour simuler que le pavé sort du sol.
 */
TileSpawner.attributes.add('startingDepth', {
    type: 'number',
    default: -1,
    title: 'Profondeur de départ'
});

/**
 * 7) Vitesse de remontée. (Ici, elle détermine à quelle vitesse "t" passe de 0 à 1).
 */
TileSpawner.attributes.add('riseSpeed', {
    type: 'number',
    default: 1,
    title: 'Vitesse de remontée'
});

/**
 * 8) Nombre maximum de pavés actifs simultanés (0 ou -1 = illimité).
 */
TileSpawner.attributes.add('maxBlocks', {
    type: 'number',
    default: 0,
    title: 'Nombre max de pavés'
});

/**
 * On stocke les pavés pour les animer (same logic qu’avant).
 * Ajout d’une variable "progress" pour le ratio 0..1 d’animation (ease).
 */
TileSpawner.prototype.initialize = function() {
    // Dernière position du personnage (pour mesurer la distance parcourue).
    this.lastCharacterPosition = this.character ? this.character.getPosition().clone() : new pc.Vec3(0,0,0);

    this.distanceAccumulated = 0;
    this.spawnedBlocks = [];
};

TileSpawner.prototype.update = function(dt) {
    if (!this.character) return;
    if (!this.blockPrefab) return;

    // 1) Calculer la distance parcourue cette frame
    var currentPos = this.character.getPosition();
    var distFrame = currentPos.clone().sub(this.lastCharacterPosition).length();
    this.distanceAccumulated += distFrame;

    // 2) Vérifier si on spawn un nouveau pavé
    if (this.distanceAccumulated >= this.distanceBetweenBlocks) {
        this.distanceAccumulated = 0; // on réinitialise

        // Clone unique
        var newBlock = this.blockPrefab.clone();
        newBlock.name = this.blockPrefab.name + "_clone";

        this.app.root.addChild(newBlock);
        newBlock.enabled = true;

        // Position devant le perso
        var forward = this.character.forward.clone().normalize();
        var spawnPos = currentPos.clone().add(forward.scale(this.spawnDistance));
        spawnPos.y = this.startingDepth;
        newBlock.setPosition(spawnPos);

        // Stocker le pavé dans le tableau (ajout d’un "progress")
        // progress = 0 => tout en bas, 1 => arrivé à spawnHeight
        this.spawnedBlocks.push({
            entity: newBlock,
            isRising: true,
            progress: 0
        });

        // Gérer le nombre max
        if (this.maxBlocks > 0 && this.spawnedBlocks.length > this.maxBlocks) {
            var oldest = this.spawnedBlocks.shift();
            if (oldest && oldest.entity) {
                oldest.entity.destroy();
            }
        }
    }

    // 3) Animer la remontée de chaque pavé via un easing "easeOutElastic"
    for (var i = 0; i < this.spawnedBlocks.length; i++) {
        var blockData = this.spawnedBlocks[i];
        if (!blockData.isRising) continue;

        // On avance la "progress" en fonction de la vitesse
        // (On normalise par la différence de hauteur pour que la durée dépende de l'écart)
        var totalHeight = this.spawnHeight - this.startingDepth;
        if (totalHeight <= 0) {
            // Si spawnHeight <= startingDepth, on évite la division par 0
            blockData.isRising = false;
            continue;
        }

        // Incrémente la progression
        var increment = (this.riseSpeed * dt) / totalHeight;  
        blockData.progress += increment;

        if (blockData.progress >= 1) {
            // Animation terminée
            blockData.progress = 1;
            blockData.isRising = false;
        }

        // Calcul du ratio "élastique"
        var eased = this.easeOutElastic(blockData.progress);

        // On interpole entre startingDepth et spawnHeight
        var newY = this.startingDepth + eased * totalHeight;

        // Mise à jour de la position
        var blockEntity = blockData.entity;
        var pos = blockEntity.getPosition();
        blockEntity.setPosition(pos.x, newY, pos.z);
    }

    // 4) Mémoriser la position pour la prochaine frame
    this.lastCharacterPosition.copy(currentPos);
};

/**
 * Fonction d’easing "easeOutElastic"
 * t : [0..1]
 * (Formule standard d’un easing élastique en sortie)
 */
TileSpawner.prototype.easeOutElastic = function(t) {
    var c4 = (2 * Math.PI) / 3;

    return t === 0 
        ? 0
        : t === 1 
            ? 1
            : Math.pow(2, -10 * t) * Math.sin((t * 10 - 0.75) * c4) + 1;
};
