var ShaderBifrost = pc.createScript('shaderBifrost');

// Vitesse d’animation
ShaderBifrost.attributes.add('speed', {
    type: 'number',
    default: 1.0,
    title: 'Speed'
});

// Résolution virtuelle du shader
ShaderBifrost.attributes.add('resolution', {
    type: 'vec2',
    default: [1024, 1024],
    title: 'Resolution'
});

// Texture diffuse à combiner
ShaderBifrost.attributes.add('diffuseMap', {
    type: 'asset',
    assetType: 'texture',
    title: 'Diffuse Map'
});

// Intensité de l’effet Bifrost
ShaderBifrost.attributes.add('starIntensity', {
    type: 'number',
    default: 0.5,
    title: 'Effect Intensity',
    min: 0,
    max: 2
});

ShaderBifrost.prototype.initialize = function () {
    this.time = 0;
    var gd = this.app.graphicsDevice;

    // --- Vertex Shader ---
    var vertexShader = `
        attribute vec3 aPosition;
        attribute vec2 aUv0;
        uniform mat4 matrix_model;
        uniform mat4 matrix_viewProjection;
        varying vec2 vUv0;

        void main(void) {
            vUv0 = aUv0;
            gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
        }
    `;

    // --- Fragment Shader ---
    // On adapte la logique du Shadertoy (mainImage) en utilisant :
    // - uTime à la place de iTime
    // - uResolution.x / y à la place de iResolution.x / y
    // - vUv0 * uResolution comme "fragCoord"
    // - Ajout d'un sampler2D pour la texture diffuse, qu'on combine à la fin
    var fragmentShader = `
        #define PI 3.14159
        #define TAU 6.28318

        precision highp float;

        varying vec2 vUv0;
        uniform float uTime;
        uniform vec2 uResolution;
        uniform sampler2D uDiffuseMap;
        uniform float uStarIntensity;

        void main(void) {
            // "fragCoord" virtuel : 0..uResolution
            vec2 fragCoord = vUv0 * uResolution;

            // Reprise de la formule Shadertoy
            // vec2 uv = (fragCoord - 0.5 * iResolution.xy) / iResolution.y;
            vec2 uv = (fragCoord - 0.5 * uResolution) / uResolution.y;

            uv.y = 1.0 - uv.y * 1.5;  // V-flip

            // Couleur de base
            vec3 col = vec3(0.0955, 0.05, 0.9525);

            // uv.x = uv.x * 100.0  => H-count
            uv.x *= 64.0;
            float dx = fract(uv.x);
            uv.x = floor(uv.x);

            float t = uTime * 0.14;

            // uv.y *= 0.15 => stretch vertical
            uv.y *= 0.15;

            float o = sin(uv.x * 215.4);    // offset
            float s = cos(uv.x * 33.1) * 0.53 + 0.7; // speed factor
            float trail = mix(95.0, 35.0, s);
            float yv = fract(uv.y + t * s + o) * trail;
            yv = 1.0 / yv;
            yv = smoothstep(0.0, 1.0, yv * yv);
            yv = sin(yv * PI) * (s * 5.0);

            float d2 = sin(dx * PI);
            yv *= d2 * d2;

            // Couleur finale de l'effet
            vec3 effectColor = col * yv;

            // Lecture de la texture diffuse
            vec3 diffuseColor = texture2D(uDiffuseMap, vUv0).rgb;

            // Exemple d'addition : le paramètre uStarIntensity dose l’effet
            vec3 finalColor = diffuseColor + effectColor * uStarIntensity;
            gl_FragColor = vec4(finalColor, 1.0);
        }
    `;

    // Définition et création du shader
    this.shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vertexShader,
        fshader: fragmentShader
    };
    this.shader = new pc.Shader(gd, this.shaderDefinition);

    // Création du material
    this.material = new pc.Material();
    this.material.shader = this.shader;

    // Paramètres initiaux
    this.material.setParameter('uResolution', [this.resolution.x, this.resolution.y]);
    this.material.setParameter('uTime', this.time);
    this.material.setParameter('uStarIntensity', this.starIntensity);

    if (this.diffuseMap) {
        this.material.setParameter('uDiffuseMap', this.diffuseMap.resource);
    }

    this.material.update();

    // Affectation du material à l’entité (model ou render)
    if (this.entity.model && this.entity.model.meshInstances) {
        for (var i = 0; i < this.entity.model.meshInstances.length; i++) {
            this.entity.model.meshInstances[i].material = this.material;
        }
    } else if (this.entity.render && this.entity.render.meshInstances) {
        for (var r = 0; r < this.entity.render.meshInstances.length; r++) {
            this.entity.render.meshInstances[r].material = this.material;
        }
    }
};

ShaderBifrost.prototype.update = function (dt) {
    // Mise à jour du temps (animé par "speed")
    this.time += dt * this.speed;

    // Passe les paramètres au shader
    this.material.setParameter('uTime', this.time);
    this.material.setParameter('uResolution', [this.resolution.x, this.resolution.y]);
    this.material.setParameter('uStarIntensity', this.starIntensity);

    if (this.diffuseMap) {
        this.material.setParameter('uDiffuseMap', this.diffuseMap.resource);
    }
};
