var StarShader = pc.createScript('starShader');

// Vitesse de défilement des étoiles
StarShader.attributes.add('speed', {
    type: 'number',
    default: 1.0,
    title: 'Speed'
});

// Résolution virtuelle du shader
StarShader.attributes.add('resolution', {
    type: 'vec2',
    default: [1024, 1024],
    title: 'Resolution'
});

// Texture diffuse que l’on va combiner au shader
StarShader.attributes.add('diffuseMap', {
    type: 'asset',
    assetType: 'texture',
    title: 'Diffuse Map'
});

// Intensité de l’effet « étoile » sur la texture
StarShader.attributes.add('starIntensity', {
    type: 'number',
    default: 0.5,
    title: 'Star Intensity',
    min: 0,
    max: 2
});

StarShader.prototype.initialize = function () {
    this.time = 0;
    var gd = this.app.graphicsDevice;

    // --- Vertex Shader ---
    var vertexShader = `
        attribute vec3 aPosition;
        attribute vec2 aUv0;
        uniform mat4 matrix_model;
        uniform mat4 matrix_viewProjection;
        varying vec2 vUv0;

        void main(void) {
            vUv0 = aUv0;
            gl_Position = matrix_viewProjection * matrix_model * vec4(aPosition, 1.0);
        }
    `;

    // --- Fragment Shader ---
    // - Ajout d'un sampler2D pour la texture diffuse (uDiffuseMap).
    // - Combinaison: colorFinal = diffuseColor + (starColor * starIntensity).
    var fragmentShader = `
        precision highp float;
        varying vec2 vUv0;

        uniform float uTime;
        uniform vec2 uResolution;
        uniform sampler2D uDiffuseMap;
        uniform float uStarIntensity;

        // Reprend la fonction "Cell" pour générer l’effet étoilé
        float Cell(vec2 c) {
            vec2 uv = fract(c);
            c -= uv;
            return (3.0 - length(uv)) * step(fract(sin(c.x + c.y * 100.0) * 1000.0), 0.04);
        }

        void main(void) {
            // Coordonnées "fragCoord" virtuelles
            vec2 fragCoord = vUv0 * uResolution;

            // Paramétrage identique au code Shadertoy
            vec2 projection = fragCoord / vec2(1024.0, 1024.0);
            float projX = fract(projection.x);
            float projY = pow(projection.y, 0.014);
            float time = uTime;
            vec3 starColor = vec3(0.0);

            for(int rgb = 0; rgb < 3; rgb++) {
                time -= 0.02;
                vec2 coord = vec2(projY, projX) * 256.0;
                vec2 delta = vec2(time * 7.0, 0.0);
                float c = Cell(coord - delta);
                c += Cell(coord - delta);
                starColor[rgb] = c * projection.y * 5.0;
            }

            // Récupère la couleur de la texture diffuse
            vec3 diffuseColor = texture2D(uDiffuseMap, vUv0).rgb;

            // Exemple de fusion : addition avec facteur
            // Ajustez à vos besoins : +, *, mix(), etc.
            vec3 finalColor = diffuseColor + starColor * uStarIntensity;

            gl_FragColor = vec4(finalColor, 1.0);
        }
    `;

    // Définition et création du shader
    this.shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vertexShader,
        fshader: fragmentShader
    };

    this.shader = new pc.Shader(gd, this.shaderDefinition);

    // Création du material
    this.material = new pc.Material();
    this.material.shader = this.shader;

    // Paramètres initiaux
    // (On convertit pc.Vec2 en tableau pour WebGL)
    this.material.setParameter('uResolution', [this.resolution.x, this.resolution.y]);
    this.material.setParameter('uTime', this.time);
    this.material.setParameter('uStarIntensity', this.starIntensity);

    // Si la texture est définie (non-null), on la transmet au shader
    if (this.diffuseMap) {
        this.material.setParameter('uDiffuseMap', this.diffuseMap.resource);
    }

    this.material.update();

    // Appliquer la matière à l’entité (qu’elle soit .model ou .render)
    if (this.entity.model && this.entity.model.meshInstances) {
        for (var i = 0; i < this.entity.model.meshInstances.length; i++) {
            this.entity.model.meshInstances[i].material = this.material;
        }
    } else if (this.entity.render && this.entity.render.meshInstances) {
        for (var r = 0; r < this.entity.render.meshInstances.length; r++) {
            this.entity.render.meshInstances[r].material = this.material;
        }
    }
};

StarShader.prototype.update = function (dt) {
    // Mise à jour du temps
    this.time += dt * this.speed;
    this.material.setParameter('uTime', this.time);

    // Met à jour la résolution si elle change en temps réel
    this.material.setParameter('uResolution', [this.resolution.x, this.resolution.y]);

    // Met à jour l'intensité des étoiles
    this.material.setParameter('uStarIntensity', this.starIntensity);

    // Mettre à jour la texture si elle change dynamiquement
    if (this.diffuseMap) {
        this.material.setParameter('uDiffuseMap', this.diffuseMap.resource);
    }
};
