var InfiniteDesert = pc.createScript('infiniteDesert');

// ----------------------------------------------------------------
// 1) Paramètres exposés dans l’éditeur
// ----------------------------------------------------------------

// (A) Déformation des dunes
InfiniteDesert.attributes.add('speed', {
    type: 'number',
    default: 0.2,
    title: 'Vitesse dunes'
});
InfiniteDesert.attributes.add('heightScale', {
    type: 'number',
    default: 2,
    title: 'Amplitude dunes'
});
InfiniteDesert.attributes.add('noiseScale', {
    type: 'number',
    default: 0.1,
    title: 'Échelle bruit (vertex)'
});

// (B) Texture sable
InfiniteDesert.attributes.add('sableTexture', {
    type: 'asset',
    assetType: 'texture',
    title: 'Texture de sable'
});
InfiniteDesert.attributes.add('uvTiling', {
    type: 'vec2',
    default: [1, 1],
    title: 'Tiling UV (répétition)'
});

// (C) Éclairage Lambert « maison »
InfiniteDesert.attributes.add('lightDirection', {
    type: 'vec3',
    default: [1, 1, 1],
    title: 'Direction lumière'
});
InfiniteDesert.attributes.add('lightColor', {
    type: 'rgb',
    default: [1, 1, 1],
    title: 'Couleur lumière'
});
InfiniteDesert.attributes.add('lightIntensity', {
    type: 'number',
    default: 1,
    title: 'Intensité lumière'
});
InfiniteDesert.attributes.add('ambientColor', {
    type: 'rgb',
    default: [0.3, 0.3, 0.3],
    title: 'Couleur ambiance'
});

// ----------------------------------------------------------------
// 2) Méthodes du script
// ----------------------------------------------------------------

InfiniteDesert.prototype.initialize = function() {

    // -------------------------
    // (A) Vertex Shader
    // -------------------------
    const vertexShader = `
    precision highp float;

    // Attributs
    attribute vec3 aPosition;
    attribute vec2 aUv0;

    // Matrices PlayCanvas
    uniform mat4 matrix_model;
    uniform mat4 matrix_viewProjection;

    // Uniforms pour la déformation
    uniform float uTime;
    uniform float uHeightScale;
    uniform float uNoiseScale;

    // Varyings (envoyés au fragment)
    varying vec2 vUv;
    // On stocke la position XZ après déformation 
    // (ou même avant, si on préfère) pour recalculer la hauteur dans le fragment
    varying vec2 vPosXZ;

    // -------------------- Fonctions de bruit
    float random(vec2 n) { 
        return fract(sin(dot(n, vec2(12.9898, 78.233))) * 43758.5453);
    }
    float noise2D(vec2 p){
        vec2 ip = floor(p);
        vec2 f  = fract(p);
        f = f*f*(3.0 - 2.0*f);

        float res = mix(
            mix(random(ip),         random(ip + vec2(1.0, 0.0)), f.x),
            mix(random(ip + vec2(0.0, 1.0)), random(ip + vec2(1.0, 1.0)), f.x),
            f.y
        );
        return res * res;
    }

    void main(void) {
        // Position locale
        vec3 pos = aPosition;

        // Déformation "dunes"
        vec2 noisePos = (pos.xz + vec2(uTime, 0.0)) * uNoiseScale;
        float duneH = noise2D(noisePos) * uHeightScale;
        pos.y += duneH;

        // Projection
        gl_Position = matrix_viewProjection * (matrix_model * vec4(pos, 1.0));

        // Sorties vers le fragment
        vUv = aUv0;
        // On stocke la position XZ en local (après déformation en Y, ou avant, 
        // ici c’est après – ça ne change pas XZ de toute façon)
        vPosXZ = pos.xz;
    }
    `;

    // -------------------------
    // (B) Fragment Shader
    // -------------------------
    const fragmentShader = `
    precision highp float;

    // Varyings
    varying vec2 vUv;
    varying vec2 vPosXZ;

    // -------- Uniforms dune (pour recalculer la hauteur => normale) 
    uniform float uTime;
    uniform float uHeightScale;
    uniform float uNoiseScale;

    // -------- Uniforms texture sable
    uniform sampler2D uSableTex; 
    uniform vec2 uUvTiling;

    // -------- Uniforms éclairage
    uniform vec3 uLightDir;
    uniform vec3 uLightColor;
    uniform float uLightIntensity;
    uniform vec3 uAmbientColor;

    // ------------------ Fonctions de bruit
    float random(vec2 n) { 
        return fract(sin(dot(n, vec2(12.9898, 78.233))) * 43758.5453);
    }
    float noise2D(vec2 p){
        vec2 ip = floor(p);
        vec2 f  = fract(p);
        f = f*f*(3.0 - 2.0*f);

        float res = mix(
            mix(random(ip), random(ip + vec2(1.0, 0.0)), f.x),
            mix(random(ip + vec2(0.0,1.0)), random(ip + vec2(1.0,1.0)), f.x),
            f.y
        );
        return res * res;
    }

    // ------------------ Hauteur de la dune (même algo que le vertex)
    float getDuneHeight(vec2 xz) {
        vec2 nPos = (xz + vec2(uTime, 0.0)) * uNoiseScale;
        return noise2D(nPos) * uHeightScale;
    }

    // ------------------ Approx. de la normale via dérivée
    vec3 computeNormal(vec2 xz) {
        float eps = 0.001;  // "pas" pour évaluer la pente
        float hL = getDuneHeight(xz + vec2(-eps, 0.0));
        float hR = getDuneHeight(xz + vec2(+eps, 0.0));
        float hD = getDuneHeight(xz + vec2(0.0, -eps));
        float hU = getDuneHeight(xz + vec2(0.0, +eps));

        float dx = (hR - hL);
        float dz = (hU - hD);

        // Normale = cross( dPos/dx, dPos/dz ), ou un trick direct
        // On normalise pour s'assurer d'un vecteur unitaire
        // Le "2.0 * eps" correspond à la différence en Y
        vec3 normal = normalize(vec3(-dx, 2.0 * eps, -dz));
        return normal;
    }

    void main(void) {
        // 1) Normale en fragment (Lambert)
        vec3 N = computeNormal(vPosXZ);
        // Direction de la lumière normalisée
        vec3 L = normalize(uLightDir);
        float NdotL = max(dot(N, L), 0.0);

        // 2) Échantillonnage de la texture
        //    On applique un tiling (répétition) 
        //    pour éviter l'étirement de la texture.
        vec2 uvScaled = vUv * uUvTiling;
        vec4 texColor = texture2D(uSableTex, uvScaled);

        // 3) Lambert simple
        vec3 lambert = texColor.rgb * uLightColor * (NdotL * uLightIntensity);
        // 4) Ambiance
        vec3 ambient = texColor.rgb * uAmbientColor;

        // 5) Couleur finale
        vec3 finalColor = lambert + ambient;

        gl_FragColor = vec4(finalColor, 1.0);
    }
    `;

    // -------------------------
    // (C) Création du shader
    // -------------------------
    const shaderDefinition = {
        attributes: {
            aPosition: pc.SEMANTIC_POSITION,
            aUv0: pc.SEMANTIC_TEXCOORD0
        },
        vshader: vertexShader,
        fshader: fragmentShader
    };
    this.shader = new pc.Shader(this.app.graphicsDevice, shaderDefinition);

    // -------------------------
    // (D) Création du matériau
    // -------------------------
    this.material = new pc.Material();
    this.material.shader = this.shader;
    this.material.update();

    // -------------------------
    // (E) Application du mat
    // -------------------------
    const render = this.entity.render;
    if (!render) {
        console.warn('Aucun composant Render sur cette entité');
        return;
    }
    const meshInstances = render.meshInstances;
    for (let i = 0; i < meshInstances.length; i++) {
        meshInstances[i].material = this.material;
    }

    // -------------------------
    // (F) Animation temps
    // -------------------------
    this.time = 0;
};

InfiniteDesert.prototype.update = function(dt) {
    // Avancement du temps
    this.time += dt * this.speed;

    // Passage des uniforms
    const meshInstances = this.entity.render.meshInstances;
    for (let i = 0; i < meshInstances.length; i++) {
        let mi = meshInstances[i];

        // (a) Vertex
        mi.setParameter('uTime', this.time);
        mi.setParameter('uHeightScale', this.heightScale);
        mi.setParameter('uNoiseScale', this.noiseScale);

        // (b) Texture sable (sampler2D)
        if (this.sableTexture && this.sableTexture.resource) {
            mi.setParameter('uSableTex', this.sableTexture.resource);
        }
        // Répétition UV
        mi.setParameter('uUvTiling', [this.uvTiling.x, this.uvTiling.y]);

        // (c) Éclairage
        //   Convertir lightDirection en simple tableau [x,y,z]
        mi.setParameter('uLightDir', [
            this.lightDirection.x,
            this.lightDirection.y,
            this.lightDirection.z
        ]);
        // Idem pour la couleur (RGB)
        mi.setParameter('uLightColor', [
            this.lightColor.r,
            this.lightColor.g,
            this.lightColor.b
        ]);
        mi.setParameter('uLightIntensity', this.lightIntensity);
        mi.setParameter('uAmbientColor', [
            this.ambientColor.r,
            this.ambientColor.g,
            this.ambientColor.b
        ]);
    }
};
