/**
 * @class
 * @name Bang2Effect
 * @classdesc PostEffect adapté du code Shadertoy, avec un mix progressif (factor).
 * @augments pc.PostEffect
 *
 * Propriétés manipulées dans bang2.js :
 *  - this.time    (avancé dans update)
 *  - this.factor  (0 => scène, 1 => fractal)
 */
function Bang2Effect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // On va animer le temps
    this.time = 0.0;
    // Mix entre la scène et le fractal
    this.factor = 0.0;

    // Fragment shader
    const fshader = /* glsl */ `
        varying vec2 vUv0;
        uniform sampler2D uColorBuffer; 
        uniform float uTime;    
        uniform float uFactor;  
        uniform vec2  uResolution; 

        // -- Code Shadertoy (adapté) --
        vec2 hash12(float t) {
            float x = fract(sin(t*674.31)*432.1)*6.2384;
            float y = fract(sin(t*472.1)*201.9);
            // on ajoute un offset "sin(uTime)"
            return vec2(sin(x)*cos(uTime), cos(x))*y + sin(uTime);
        }

        vec4 fractalEffect(vec2 fragCoord) {
            vec2 uv = (fragCoord - 0.5 * uResolution) / uResolution.y;

            float t = uTime;
            vec3 color = vec3(0.0);

            float dt = 0.5;
            vec3 baseCol = vec3(0.3*dt, 0.2, 0.88);

            for (float i=1000.0; i>0.0; i--) {
                float brightness = 0.0001 * (sin((i*3.0)+t) + 3.0);
                vec2 dir = hash12(i+1.0)*0.9;
                float tt = fract(uTime * 0.18) * 5.0;
                float distVal = length(uv * dir / tt);
                color += brightness / distVal;
            }

            vec3 finalCol = color * baseCol;
            return vec4(finalCol, 1.0);
        }

        void main() {
            vec4 sceneColor = texture2D(uColorBuffer, vUv0);

            // Calcul fractal
            vec2 fragCoord = vUv0 * uResolution;
            vec4 fractalCol = fractalEffect(fragCoord);

            // factor => 0 => 100% scène, 1 => 100% fractal
            vec3 finalColor = sceneColor.rgb+fractalCol.rgb* uFactor;

            gl_FragColor = vec4(finalColor, 1.0);
        }
    `;

    this.shader = pc.createShaderFromCode(
        graphicsDevice,
        pc.PostEffect.quadVertexShader,
        fshader,
        "Bang2Shader_Fade",
        { aPosition: pc.SEMANTIC_POSITION }
    );
}

Bang2Effect.prototype = Object.create(pc.PostEffect.prototype);
Bang2Effect.prototype.constructor = Bang2Effect;

Object.assign(Bang2Effect.prototype, {
    render: function(inputTarget, outputTarget, rect) {
        const scope = this.device.scope;

        // On passe la scène
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);

        // Résolution
        scope.resolve("uResolution").setValue([this.device.width, this.device.height]);

        // Temps
        scope.resolve("uTime").setValue(this.time);

        // factor [0..1]
        scope.resolve("uFactor").setValue(this.factor);

        this.drawQuad(outputTarget, this.shader, rect);
    }
});
