/**
 * @class
 * @name BangEffect
 * @classdesc PostEffect pour un "BANG" explosif style Shadertoy.
 * @augments pc.PostEffect
 *
 * Propriétés mises à jour depuis bang.js :
 *  - this.time  (avancé chaque frame)
 *  - this.active (booléen : explosif actif ou non)
 */
function BangEffect(graphicsDevice) {
    pc.PostEffect.call(this, graphicsDevice);

    // On va animer "time" depuis l'extérieur (bang.js)
    this.time = 0;

    // Indique si l'effet est actif
    this.active = false;

    // On mélange la scène + l'effet
    // Code Shadertoy (simplifié) : 
    //   iTime => uTime
    //   iResolution => uResolution
    //   ... 
    const fshader = /* glsl */ `
        // Pas de depth, on fait un effet 2D additif.
        varying vec2 vUv0;

        uniform sampler2D uColorBuffer; // la scène
        uniform float uTime;           // temps
        uniform vec2  uResolution;     // taille ecran
        uniform bool  uActive;         // si false => on n'affiche rien

        // Conversion de "fragCoord" => on part de vUv0, [0..1]
        // Dans la version Shadertoy, fragCoord ~ [0..iResolution], 
        // uv = (fragCoord - 0.5 * iRes) / iRes.y, etc.

        // Petit helper : hash12
        vec2 hash12(float t) {
            // "aléatoire" 
            float x = fract(sin(t*674.31)*432.1)*6.2384;
            float y = fract(sin(t*472.1)*201.9);
            return vec2(sin(x), cos(x))*y;
        }

        void main() {
            // Lecture de la couleur de la scène 
            vec4 sceneColor = texture2D(uColorBuffer, vUv0);

            // Si pas actif => on renvoie juste la scène
            if (!uActive) {
                gl_FragColor = sceneColor;
                return;
            }

            // Calcule uv façon Shadertoy
            vec2 fragCoord = vUv0 * uResolution;
            vec2 uv = (fragCoord - 0.5 * uResolution) / uResolution.y;

            // Récupère le temps 
            float t = uTime * 4.0; 

            vec3 color = vec3(0.0);

            // reprise du code d'origine
            // for i in [600..0], brightness=...
            for (float i = 600.0; i > 0.0; i--) {
                float brightness = 0.0001 * (sin((i*3.0) + t) + 3.0);
                vec2 dir = hash12(i+1.0)*2.0; 
                float tt = fract(uTime * 0.2) * 10.0;
                float dt = length(uv - dir * tt);

                color += brightness / dt;
            }

            // mélange additif avec la scène
            // plus color est grand, plus on éclaire la scène
            vec3 finalColor = sceneColor.rgb + color;

            gl_FragColor = vec4(finalColor, 1.0);
        }
    `;

    this.shader = pc.createShaderFromCode(
        graphicsDevice,
        pc.PostEffect.quadVertexShader,
        fshader,
        "BangShader",
        { aPosition: pc.SEMANTIC_POSITION }
    );
}

BangEffect.prototype = Object.create(pc.PostEffect.prototype);
BangEffect.prototype.constructor = BangEffect;

Object.assign(BangEffect.prototype, {
    render: function (inputTarget, outputTarget, rect) {
        // Dans bang.js => on met à jour this.time, this.active
        const scope = this.device.scope;

        // on envoie la scène
        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        // la résolution
        scope.resolve("uResolution").setValue([this.device.width, this.device.height]);
        // le temps
        scope.resolve("uTime").setValue(this.time);
        // actif ou pas
        scope.resolve("uActive").setValue(this.active);

        // on dessine 
        this.drawQuad(outputTarget, this.shader, rect);
    }
});
