var Fontaine = pc.createScript('fontaine');

/* ------------------------------------------
   ░░░ ATTRIBUTS DU SCRIPT ░░░
------------------------------------------ */

// 1 à 5 entités (chacune avec min/max scale)
Fontaine.attributes.add('entity1', { type: 'entity', title: "Entité 1" });
Fontaine.attributes.add('entity1ScaleMin', { type: 'number', default: 0.1 });
Fontaine.attributes.add('entity1ScaleMax', { type: 'number', default: 1.0 });

Fontaine.attributes.add('entity2', { type: 'entity', title: "Entité 2" });
Fontaine.attributes.add('entity2ScaleMin', { type: 'number', default: 0.1 });
Fontaine.attributes.add('entity2ScaleMax', { type: 'number', default: 1.0 });

Fontaine.attributes.add('entity3', { type: 'entity', title: "Entité 3" });
Fontaine.attributes.add('entity3ScaleMin', { type: 'number', default: 0.1 });
Fontaine.attributes.add('entity3ScaleMax', { type: 'number', default: 1.0 });

Fontaine.attributes.add('entity4', { type: 'entity', title: "Entité 4" });
Fontaine.attributes.add('entity4ScaleMin', { type: 'number', default: 0.1 });
Fontaine.attributes.add('entity4ScaleMax', { type: 'number', default: 1.0 });

Fontaine.attributes.add('entity5', { type: 'entity', title: "Entité 5" });
Fontaine.attributes.add('entity5ScaleMin', { type: 'number', default: 0.1 });
Fontaine.attributes.add('entity5ScaleMax', { type: 'number', default: 1.0 });

// Paramètres d'émission
Fontaine.attributes.add('emissionRate', {
    type: 'number',
    default: 0.2,
    title: "Taux d'émission (sec)"
});
Fontaine.attributes.add('batchSize', {
    type: 'number',
    default: 1,
    title: "Nb entités par rafale"
});
Fontaine.attributes.add('numRepetitions', {
    type: 'number',
    default: 10,
    title: "Nb répétitions total (par entité)"
});

// Contrôle du cône + vortex
Fontaine.attributes.add('angle', {
    type: 'number',
    default: 60,
    title: "Angle du cône (°)"
});
// 🆕 minDistance & maxDistance
Fontaine.attributes.add('minDistance', {
    type: 'number',
    default: 1,
    title: "Rayon minimal"
});
Fontaine.attributes.add('maxDistance', {
    type: 'number',
    default: 5,
    title: "Rayon maximal"
});
Fontaine.attributes.add('speed', {
    type: 'number',
    default: 1,
    title: "Vitesse d'expansion (rayon)"
});

// Rotation en vortex
Fontaine.attributes.add('swirlSpeedMin', {
    type: 'number',
    default: 30,
    title: "Vortex speed min (deg/s)"
});
Fontaine.attributes.add('swirlSpeedMax', {
    type: 'number',
    default: 60,
    title: "Vortex speed max (deg/s)"
});

// Rotation sur soi-même
Fontaine.attributes.add('selfRotMin', {
    type: 'number',
    default: -20,
    title: "Rot sur soi-même min (deg/s)"
});
Fontaine.attributes.add('selfRotMax', {
    type: 'number',
    default: 20,
    title: "Rot sur soi-même max (deg/s)"
});

// Événement de déclenchement
Fontaine.attributes.add('triggerEventName', {
    type: 'string',
    default: 'launchVortex',
    title: "Nom de l'événement déclencheur"
});

/* ------------------------------------------
   ░░░ INITIALISATION ░░░
------------------------------------------ */
Fontaine.prototype.initialize = function() {
    // Rassemble la liste des entités configurées
    this._templateConfigs = [];
    if (this.entity1) { this._templateConfigs.push({ entity: this.entity1, scaleMin: this.entity1ScaleMin, scaleMax: this.entity1ScaleMax }); }
    if (this.entity2) { this._templateConfigs.push({ entity: this.entity2, scaleMin: this.entity2ScaleMin, scaleMax: this.entity2ScaleMax }); }
    if (this.entity3) { this._templateConfigs.push({ entity: this.entity3, scaleMin: this.entity3ScaleMin, scaleMax: this.entity3ScaleMax }); }
    if (this.entity4) { this._templateConfigs.push({ entity: this.entity4, scaleMin: this.entity4ScaleMin, scaleMax: this.entity4ScaleMax }); }
    if (this.entity5) { this._templateConfigs.push({ entity: this.entity5, scaleMin: this.entity5ScaleMin, scaleMax: this.entity5ScaleMax }); }

    // Nombre total d'objets à émettre
    this._maxToSpawn = this._templateConfigs.length * this.numRepetitions;

    // État d'émission
    this._spawnCount = 0;
    this._spawnIndex = 0;
    this._spawnTimer = 0;
    this._activeParticles = [];
    this._emissionActive = false; // inactif tant qu'on n'a pas reçu l'événement

    // Écoute l'événement "triggerEventName"
    this.app.on(this.triggerEventName, this.startVortex, this);

    if (this._templateConfigs.length === 0) {
        console.warn("[fontaine vortex] Aucune entité configurée !");
    }
};

Fontaine.prototype.startVortex = function() {
    this._emissionActive = true;
    this._spawnCount = 0;
    this._spawnIndex = 0;
    this._spawnTimer = 0;

    // Optionnel: détruire d'anciennes particules si on relance
    // for (var i=0; i<this._activeParticles.length; i++){
    //     this._activeParticles[i].entity.destroy();
    // }
    this._activeParticles = [];

    console.log("[fontaine.js] Émission lancée ! ");
    

};

/* ------------------------------------------
   ░░░ UPDATE : émettre + animer vortex ░░░
------------------------------------------ */
Fontaine.prototype.update = function(dt) {
    if (!this._emissionActive) return;

    // A) Émission
    if (this._spawnCount < this._maxToSpawn && this._templateConfigs.length > 0) {
        this._spawnTimer += dt;
        if (this._spawnTimer >= this.emissionRate) {
            // On émet 'batchSize' particules d'un coup
            for (var i = 0; i < this.batchSize; i++) {
                this.spawnOneParticle();
                if (this._spawnCount >= this._maxToSpawn) break;
            }
            this._spawnTimer = 0;
        }
    }

    // B) Animer chaque particule en vortex
    for (var i = 0; i < this._activeParticles.length; i++) {
        var p = this._activeParticles[i];

        // 1) rotation sur soi-même
        p.selfEuler.x += p.selfRot.x * dt;
        p.selfEuler.y += p.selfRot.y * dt;
        p.selfEuler.z += p.selfRot.z * dt;
        var localQ = new pc.Quat();
        localQ.setFromEulerAngles(p.selfEuler.x, p.selfEuler.y, p.selfEuler.z);
        p.entity.setRotation(localQ);

        // 2) progression du temps de vie + fraction 0..1
        p.lifeTime += dt;
        var frac = p.lifeTime / p.duration;
        if (frac > 1) frac = 1;

        // 3) swirlAngle -> rotation autour de Y
        p.swirlAngle += p.swirlSpeed * dt;
        if (p.swirlAngle >= 360) p.swirlAngle -= 360;
        var swirlRad = p.swirlAngle * pc.math.DEG_TO_RAD;

        // 4) rayon (minDistance .. maxDistance)
        //    Au départ, rayon=0 => Non, on a choisi ci-dessous
        //    On a déjà un finalRadius; on applique frac
        var currentRadius = p.finalRadius * frac;

        // 5) angle polaire => p.polar
        var radPol = p.polar * pc.math.DEG_TO_RAD;
        var sinPol = Math.sin(radPol);
        var cosPol = Math.cos(radPol);

        // x = r * sinPol * cos(swirlRad)
        // y = r * cosPol
        // z = r * sinPol * sin(swirlRad)
        var x = currentRadius * sinPol * Math.cos(swirlRad);
        var y = currentRadius * cosPol;
        var z = currentRadius * sinPol * Math.sin(swirlRad);

        // position = origin + (x,y,z)
        var pos = new pc.Vec3(
            p.origin.x + x,
            p.origin.y + y,
            p.origin.z + z
        );
        p.entity.setPosition(pos);

        // 6) scale grandit de 0.. finalScale
        var s = frac; 
        p.entity.setLocalScale(
            p.finalScale.x * s,
            p.finalScale.y * s,
            p.finalScale.z * s
        );
    }
};

/* ------------------------------------------
   ░░░ spawnOneParticle : vortex immÉdiat ░░░
------------------------------------------ */
Fontaine.prototype.spawnOneParticle = function() {
    if (this._spawnCount >= this._maxToSpawn) return;

    var cfg = this._templateConfigs[this._spawnIndex];
    var template = cfg.entity;

    // Cloner
    var clone = template.clone();
    this.entity.getParent().addChild(clone);

    // Origine = entité script
    var origin = this.entity.getPosition().clone();

    // Choix random de l'angle polaire (0.. this.angle)
    var polar = pc.math.random(0, this.angle);

    // swirlAngle initial (0..360)
    var swirlAngle = pc.math.random(0, 360);

    // swirlSpeed (deg/s)
    var swirlSpeed = pc.math.random(this.swirlSpeedMin, this.swirlSpeedMax);

    // *** NOUVEAU *** : tirer un rayon via sqrt pour éviter la concentration au centre
    // radius in [minDistance, maxDistance], distribution = sqrt
    var u = pc.math.random(0, 1);    // random uniform [0..1]
    var t = Math.sqrt(u);           // => moins de points près de minDistance
    var radiusSpan = this.maxDistance - this.minDistance;
    var finalRadius = this.minDistance + radiusSpan * t;

    // durée = (finalRadius - 0) / speed
    var duration = (this.speed > 0) ? finalRadius / this.speed : 1;

    // rotation sur soi-même
    var rx = pc.math.random(this.selfRotMin, this.selfRotMax);
    var ry = pc.math.random(this.selfRotMin, this.selfRotMax);
    var rz = pc.math.random(this.selfRotMin, this.selfRotMax);

    // scale final
    var sx = pc.math.random(cfg.scaleMin, cfg.scaleMax);
    var sy = pc.math.random(cfg.scaleMin, cfg.scaleMax);
    var sz = pc.math.random(cfg.scaleMin, cfg.scaleMax);

    // init la particule
    clone.setPosition(origin);
    clone.setLocalScale(0.001, 0.001, 0.001);

    var pData = {
        entity: clone,
        origin: origin,

        lifeTime: 0,
        duration: duration,

        polar: polar,        // inclinaison
        swirlAngle: swirlAngle,
        swirlSpeed: swirlSpeed,
        finalRadius: finalRadius,

        selfEuler: new pc.Vec3(0,0,0),
        selfRot: new pc.Vec3(rx, ry, rz),

        finalScale: new pc.Vec3(sx, sy, sz)
    };
    this._activeParticles.push(pData);

    // incremente
    this._spawnCount++;
    this._spawnIndex = (this._spawnIndex + 1) % this._templateConfigs.length;
};

/* ------------------------------------------
   ░░░ EASING (facultatif) ░░░
------------------------------------------ */
Fontaine.prototype.easeOutQuad = function(t) {
    return t * (2 - t);
};
