var BurningEffect = pc.createScript("burningEffect");

/*
    Ce script crée un effet de "flamme/halo" localisé sur une entité, 
    en s’inspirant de la logique GodRays (2 passes) mais appliqué 
    uniquement à la texture qui rend cette entité sur un layer séparé,
    puis composité par-dessus la scène principale.

    Étapes:
    1) Crée un layer "BurningLayer" si inexistant
    2) Assigne l'entité à ce layer
    3) Crée une caméra secondaire qui ne voit QUE ce layer => rend en texture
    4) Applique un PostEffect GodRays-like à cette caméra secondaire
    5) Affiche la texture en overlay par-dessus la caméra principale

    Attributs configurables:
    - cameraEntity: la caméra principale
    - intensity, decay, exposure, color, etc.: paramètres du "halo enflammé"
*/

BurningEffect.attributes.add("cameraEntity", {
    type: "entity",
    title: "Main Camera"
});

// Paramètres du "GodRays" local (simplifiés)
BurningEffect.attributes.add("intensity", { 
    type: "number", 
    default: 2.0, 
    min: 0, 
    max: 5, 
    title: "Intensity" 
});
BurningEffect.attributes.add("decay", { 
    type: "number", 
    default: 0.01, 
    precision: 3, 
    min: 0, 
    max: 1, 
    title: "Decay" 
});
BurningEffect.attributes.add("exposure", { 
    type: "number", 
    default: 0.1, 
    min: 0, 
    max: 1, 
    title: "Exposure" 
});
BurningEffect.attributes.add("color", { 
    type: "rgb", 
    default: [1, 0.5, 0], 
    title: "Flame Color" 
});
BurningEffect.attributes.add("samples", {
    type: "number",
    default: 32,
    min: 1,
    max: 128,
    title: "Ray Samples"
});

// Classe simplifiée s’inspirant du GodRaysEffect
function LocalGodRaysEffect(device) {
    pc.PostEffect.call(this, device);

    this.needsDepthBuffer = false;  // On peut ignorer la depth si on ne fait pas d'occlusion
    this.cameraEntity = null;       // Sera setté depuis le script
    this.intensity = 2.0;
    this.decay = 0.01;
    this.exposure = 0.1;
    this.color = new pc.Color(1, 0.5, 0);
    this.samples = 32;

    var commonVert = [
        device.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3VS : "",
        "attribute vec2 aPosition;",
        "varying vec2 vUv0;",
        "void main() {",
        "    gl_Position = vec4(aPosition, 0.0, 1.0);",
        "    vUv0 = (aPosition.xy + 1.0) * 0.5;",
        "}"
    ].join("\n");

    // Pass unique (on peut combiner le halo + blur en un seul effet simplifié)
    this.godraysShader = new pc.Shader(device, {
        attributes: { aPosition: pc.SEMANTIC_POSITION },
        vshader: commonVert,
        fshader: [
            device.webgl2 ? "#version 300 es\n" + pc.shaderChunks.gles3PS : "",
            "precision " + device.precision + " float;",
            "",
            "#define MAX_SAMPLES 128",
            "varying vec2 vUv0;",
            "uniform sampler2D uColorBuffer;",
            "uniform float uSamples;",
            "uniform float uDecay;",
            "uniform float uExposure;",
            "uniform vec3  uColor;",
            "uniform float uIntensity;",
            "",
            "void main() {",
            "    // On lit la couleur de base",
            "    vec4 base = texture2D(uColorBuffer, vUv0);",
            "",
            "    // Raymarch radial depuis le centre du screen (0.5, 0.5) pour accentuer l’effet “halo”",
            "    // Si tu veux la source au centre de l’entité, plus sophistiqué => calcul 2D de la position",
            "    // Dans ce script simplifié, on prend le centre de l’écran comme source.",
            "",
            "    vec2 lightPos = vec2(0.5, 0.5);",
            "    vec2 delta = (vUv0 - lightPos) / uSamples;",
            "    float occ = 0.0;",
            "    float illumDecay = 1.0;",
            "    vec2 coord = vUv0;",
            "",
            "    for(int i=0; i<MAX_SAMPLES; i++) {",
            "        if(float(i) >= uSamples) break;",
            "        coord -= delta;",
            "        // Lis la luminance du pixel parcouru",
            "        float lum = dot(texture2D(uColorBuffer, coord).rgb, vec3(0.299, 0.587, 0.114));",
            "        occ += lum * illumDecay;",
            "        illumDecay *= uDecay;",
            "    }",
            "",
            "    // Applique l’intensité, l’exposition, la coloration",
            "    float rays = occ * uExposure * uIntensity;",
            "    vec3 haloColor = rays * uColor;",
            "",
            "    // Combine la couleur de base + halo en additif",
            "    gl_FragColor = base + vec4(haloColor, 1.0);",
            "}"
        ].join("\n")
    });
}
LocalGodRaysEffect.prototype = Object.create(pc.PostEffect.prototype);
LocalGodRaysEffect.prototype.constructor = LocalGodRaysEffect;

Object.assign(LocalGodRaysEffect.prototype, {
    render: function(inputTarget, outputTarget, rect) {
        var device = this.device;
        var scope  = device.scope;

        scope.resolve("uColorBuffer").setValue(inputTarget.colorBuffer);
        scope.resolve("uSamples").setValue(this.samples);
        scope.resolve("uDecay").setValue(1.0 - this.decay);
        scope.resolve("uExposure").setValue(this.exposure);
        scope.resolve("uIntensity").setValue(this.intensity);
        scope.resolve("uColor").setValue([this.color.r, this.color.g, this.color.b]);

        pc.drawFullscreenQuad(device, outputTarget, this.vertexBuffer, this.godraysShader, rect);
    }
});


// ------------------------------------------------------------------------
// Script principal "BurningEffect"
// ------------------------------------------------------------------------

BurningEffect.prototype.initialize = function () {
    // 1) Vérifier la caméra principale
    if (!this.cameraEntity || !this.cameraEntity.camera) {
        console.warn("BurningEffect: cameraEntity invalide");
        return;
    }

    // 2) Créer/récupérer le layer "BurningLayer"
    var app = this.app;
    var layerName = "BurningLayer";
    var layerComp = app.scene.layers;

    // Cherche un layer existant qui s’appelle "BurningLayer"
    var burningLayer = layerComp.getLayerByName(layerName);
    if (!burningLayer) {
        // Crée un layer
        burningLayer = new pc.Layer({
            name: layerName,
            opaqueSortMode: pc.SORTMODE_NONE,
            transparentSortMode: pc.SORTMODE_NONE
        });
        layerComp.insert(burningLayer, 1); // On l’insère quelque part dans la pile
    }

    // 3) Assigner l’entité courante à ce layer
    //    - Si c’est un RenderComponent => entity.render.layers = [ burningLayer.id ];
    //    - Si c’est un ModelComponent => entity.model.layers = [ burningLayer.id ];
    if (this.entity.render) {
        this.entity.render.layers = [burningLayer.id];
    } else if (this.entity.model) {
        this.entity.model.layers = [burningLayer.id];
    } else {
        console.warn("BurningEffect: l'entité n'a pas de render/model component:", this.entity.name);
        return;
    }

    // 4) Créer une caméra secondaire qui va rendre ONLY ce layer => texture
    this._burningCamera = new pc.Entity("BurningCamera");
    this.app.root.addChild(this._burningCamera);

    // Ajoute un composant camera
    this._burningCamera.addComponent("camera", {
        // On peut caler la même projection que la caméra principale, 
        // ou faire un orthographique si c’est un usage 2D...
        // Ici, on duplique la perspective de la caméra principale:
        fov: this.cameraEntity.camera.fov,
        nearClip: this.cameraEntity.camera.nearClip,
        farClip: this.cameraEntity.camera.farClip,
        clearColor: new pc.Color(0, 0, 0, 0),
        priority: this.cameraEntity.camera.priority - 1,   // On dessine avant la cam principale
        rect: this.cameraEntity.camera.rect.clone(),       // Même viewport
        layers: [burningLayer.id],                         // Seul le layer BurningLayer
        clearFlags: pc.CLEARFLAG_COLOR | pc.CLEARFLAG_DEPTH
    });

    // Optionnel: positionner la caméra secondaire au même endroit que la caméra principale
    // (pour que l'angle de vue soit identique)
    this._burningCamera.setPosition(this.cameraEntity.getPosition());
    this._burningCamera.setRotation(this.cameraEntity.getRotation());

    // 5) Rendu dans une texture
    var device = app.graphicsDevice;
    var rtTex = new pc.Texture(device, {
        width: device.width,
        height: device.height,
        format: pc.PIXELFORMAT_R8_G8_B8_A8,
        mipmaps: false
    });
    rtTex.minFilter = pc.FILTER_LINEAR;
    rtTex.magFilter = pc.FILTER_LINEAR;
    rtTex.addressU = pc.ADDRESS_CLAMP_TO_EDGE;
    rtTex.addressV = pc.ADDRESS_CLAMP_TO_EDGE;

    this._renderTarget = new pc.RenderTarget({
        colorBuffer: rtTex,
        depth: true
    });

    // Assignation
    this._burningCamera.camera.renderTarget = this._renderTarget;

    // 6) Ajoute notre post-effect local (GodRays simplifié) sur cette caméra
    this._localEffect = new LocalGodRaysEffect(device);
    this._localEffect.intensity = this.intensity;
    this._localEffect.decay = this.decay;
    this._localEffect.exposure = this.exposure;
    this._localEffect.color.set(this.color.r, this.color.g, this.color.b);
    this._localEffect.samples = this.samples;

    this._burningCamera.camera.postEffects.addEffect(this._localEffect);

    // 7) Créer un “quad” en overlay pour afficher la texture en plein écran 
    //    par-dessus la caméra principale (avec blending additif ou alpha).
    this._overlayEntity = new pc.Entity("BurningOverlay");
    this._overlayEntity.addComponent("element", {
        type: pc.ELEMENTTYPE_IMAGE,
        anchor: new pc.Vec4(0, 0, 1, 1),   // occupe tout l'écran
        // pivot: (0,0) => en haut à gauche, ou (0.5,0.5) => centré
        pivot: new pc.Vec2(0, 0),
        width: 1280,   // sera mis à l’échelle auto par anchor
        height: 720,
        material: null // on va créer un mat qui sample rtTex
    });

    // On place cette entité sur un layer UI/Overlay
    // (ou on peut la mettre dans un Screen 2D existant)
    var screenLayer = this.app.scene.layers.getLayerByName("UI") 
                  || this.app.scene.layers.getLayerByName("Environment") 
                  || this.app.scene.layers.getLayerByName("World");
    if (screenLayer) {
        this._overlayEntity.layer = screenLayer.id;
    }
    this.app.root.addChild(this._overlayEntity);

    // 8) Créer un material qui va sampler la texture du renderTarget, 
    //    en mode Additif ou Alpha < 1
    var overlayMat = new pc.StandardMaterial();
    overlayMat.emissiveMap = rtTex;
    overlayMat.emissive = pc.Color.WHITE;
    overlayMat.opacity = 1.0;
overlayMat.blendType = pc.BLEND_NONE;
    overlayMat.depthWrite = false;        // on ne veut pas écraser le Z-buffer
    overlayMat.depthTest = false;         // on veut un “overlay”
    overlayMat.update();

    this._overlayEntity.element.material = overlayMat;

    // On écoute les changements de taille d’écran pour redimensionner le RT
    this.app.graphicsDevice.on("resizecanvas", this._onResize, this);
};

BurningEffect.prototype._onResize = function (width, height) {
    if (!this._renderTarget) return;
    var device = this.app.graphicsDevice;

    // Recréation d’une texture à la nouvelle taille
    var rtTex = new pc.Texture(device, {
        width: width,
        height: height,
        format: pc.PIXELFORMAT_R8_G8_B8_A8,
        mipmaps: false
    });
    rtTex.minFilter = pc.FILTER_LINEAR;
    rtTex.magFilter = pc.FILTER_LINEAR;
    rtTex.addressU = pc.ADDRESS_CLAMP_TO_EDGE;
    rtTex.addressV = pc.ADDRESS_CLAMP_TO_EDGE;

    // Remplace l’ancienne
    this._renderTarget.destroy();
    this._renderTarget = new pc.RenderTarget({
        colorBuffer: rtTex,
        depth: true
    });

    // Assigne à la caméra
    this._burningCamera.camera.renderTarget = this._renderTarget;

    // Met à jour le material d’overlay
    var overlayMat = this._overlayEntity.element.material;
    overlayMat.emissiveMap = rtTex;
    overlayMat.update();
};

BurningEffect.prototype.update = function(dt) {
    // Si jamais tu veux animer l’intensité, la couleur, etc. en temps réel,
    // tu peux le faire ici. Par ex.:
    // this._localEffect.intensity = this.intensity;
};

// Nettoyage
BurningEffect.prototype.destroy = function() {
    this.app.graphicsDevice.off("resizecanvas", this._onResize, this);

    if (this._burningCamera && this._burningCamera.camera) {
        this._burningCamera.camera.postEffects.removeEffect(this._localEffect);
        this._burningCamera.camera.renderTarget = null;
    }
    if (this._renderTarget) {
        this._renderTarget.destroy();
        this._renderTarget = null;
    }
    if (this._overlayEntity) {
        this._overlayEntity.destroy();
        this._overlayEntity = null;
    }
    if (this._burningCamera) {
        this._burningCamera.destroy();
        this._burningCamera = null;
    }
};
