
/*

Timer based tween that is used for animations.
We can use this with different value types like number, vec2,
vec3, vec4, quat, color, array and object.
So, the types of the 'from' and 'to' must match.
In case of array or object, all elements and fields must be numbers.

Example:

MyScript.prototype.initialize = function() {
    const tween = this.createTween({
        duration: 1,
        from: new pc.Vec3(0, 0, 0),
        to: new pc.Vec3(0, 3, 0),
        active: true,
        loop: true,
        yoyo: true,
        func: Tween.POWER(3),
        onUpdate: (tween) => this.entity.setPosition(tween.value),
        onEnd: (tween) => console.log('loop')
    });
};

*/

class Tween extends Timer {

    constructor(script, {duration, active, loop, onUpdate, onEnd,
        from, to, yoyo = false, mirror = false, func = Tween.LINEAR}) {

        super(script, {duration, active, loop, onUpdate, onEnd});
        this._lerp = undefined;
        this.from = from;
        this.to = to;
        this.yoyo = yoyo;
        this.mirror = mirror;
        this.func = func;
    }

    _updateTween() {
        if(!this._lerp) this._init();

        let coef = this.progress;
        
        if(this.yoyo) coef *= 2;

        if(coef > 1) {
            coef = this.mirror ? this.func(2 - coef) : 1 - this.func(coef - 1);
        } else {
            coef = this.func(coef);
        }

        this._lerp(coef);
    }

    _init() {
        if(typeof this.from === 'number') {
            this._lerp = function(coef) {
                this._value = this.from + coef * (this.to - this.from);
            }
        } else if(typeof this.from === 'object') {
            this._value = new this.from.constructor();

            if(this.from instanceof pc.Vec2 ||
                this.from instanceof pc.Vec3 ||
                this.from instanceof pc.Vec4 ||
                this.from instanceof pc.Color) {

                this._lerp = function(coef) {
                    this._value.lerp(this.from, this.to, coef);
                }
            } else if(this.from instanceof pc.Quat) {
                this._lerp = function(coef) {
                    this._value.slerp(this.from, this.to, coef);
                }
            } else {
                this._lerp = function(coef) {
                    for(let key in this.from) {
                        this._value[key] = this.from[key] + coef * (this.to[key] - this.from[key]);
                    }
                }
            }
        }
    }

    get value() {
        return this._value;
    }

    // functions

    static LINEAR(x) {
        return x;
    }

    static SINE(x) {
        return Math.cos(x * Math.PI) / 2 + 0.5;
    }

    static POWER(exp) {
        return (x) => 1 - Math.pow(1 - x, exp);
    }

}

pc.ScriptType.prototype.createTween = function(params) {
    return new Tween(this, params);
};
