#version 410

#pragma include "filter_common.glsl"
#pragma include "../lib/math.glsl"
//#pragma include "spect.glsl"

uniform float u_Time;
uniform sampler3D u_LUT3D;  // 3D LUT uniform

uniform sampler2D u_CardTex;
uniform float u_Card;

uniform sampler2D u_BlackTex;

uniform sampler2D u_YoTex;
uniform float u_Yo;
uniform bool u_DoYo;

uniform sampler2D u_HeXeniumTex;
uniform float u_HeXenium;
uniform bool u_DoHeXenium;
uniform float u_Clean;

uniform bool u_DoGlitch;
uniform float vertJerkOpt;
uniform float vertMovementOpt;
uniform float bottomStaticOpt;
uniform float scalinesOpt;
uniform float rgbOffsetOpt;
uniform float horzFuzzOpt;

uniform sampler2D u_X26Tex;
uniform sampler2D u_DateTex;
uniform sampler2D u_PlaceTex;
uniform sampler2D u_GreetzTex;
uniform float u_Scroll;

uniform float u_Noise;

uniform float u_White;



// Constants as vec3 where used with fma (all args must match types)
const vec3 A   = vec3(0.15);
const vec3 B   = vec3(0.50);
const vec3 CB  = vec3(0.05);       // C*B = 0.10 * 0.50
const vec3 DE  = vec3(0.004);      // D*E = 0.20 * 0.02
const vec3 DF  = vec3(0.06);       // D*F = 0.20 * 0.30
const float EF = 0.06666667;       // E/F  = 0.02 / 0.30

// White normalization for W = 11.2 (precomputed)
const vec3 WHITE_SCALE = vec3(1.3790642466);
const vec3 EF_WS       = vec3(EF) * WHITE_SCALE;

//float rand(vec3 seed) {
//    return fract(sin(dot(seed, vec3(12.9898, 78.233, 45.164))) * 43758.5453123);
//}

vec3 applyToneMapUncharted2(vec3 x) {
    // Evaluate numerator and denominator via FMA: x*(A*x + k) + c
    vec3 num = fma(x, fma(x, A, CB), DE);
    vec3 den = fma(x, fma(x, A, B ), DF);

    // Multiply by reciprocal (drivers typically map to rcp+mul)
    vec3 res = num * (vec3(1.0) / den);

    // Apply scalar white normalization and subtract scaled EF (both as vec3 for FMA)
    return fma(res, WHITE_SCALE, -EF_WS);
}


// Author Attribution:
// This function is adapted from the WebGL noise implementation by Ashima Arts, which is a well-known
// source for procedural noise generation in GLSL. The mod289 function is used to wrap coordinates
// within a specific range, which is crucial for generating seamless noise patterns. The original
// code and related noise functions can be found here:
// https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
// Function to wrap a vector's components within the range [0, 289)
vec3 mod289(vec3 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}


// Author Attribution:
// This function is adapted from the WebGL noise implementation by Ashima Arts, which is widely used
// for procedural noise generation in GLSL. The `mod289` function is critical for ensuring that
// vector components are wrapped within a specific range, thereby enabling the creation of seamless
// noise patterns. The original code and related noise functions can be found here:
// https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
// Function to wrap a vector's components within the range [0, 289)
vec2 mod289(vec2 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}


// Author Attribution:
// This function is based on the noise generation techniques popularized by Ken Perlin and further
// developed in the WebGL community by Ashima Arts. The `permute` function is a crucial step in
// generating pseudo-random values used in procedural noise algorithms. It ensures that the input
// vector is shuffled in a way that contributes to the non-repetitive nature of noise. The original
// implementation can be found in the WebGL noise library by Ashima Arts:
// https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
// Function to permute vector components to ensure non-repetitive noise patterns
vec3 permute(vec3 x) {
  return mod289(((x * 34.0) + 1.0) * x);
}


// Author Attribution:
// This simplex noise function is adapted from a popular implementation by Ashima Arts,
// which itself draws from Ken Perlin's original work on noise functions. The algorithm
// generates a smooth, gradient noise pattern, which is often used in procedural
// texture generation and other computer graphics applications. This specific
// implementation is widely used in the graphics community and can be found
// in the WebGL noise repository by Ashima Arts:
// https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
// 2D Simplex noise function
float snoise(vec2 v) {
    // Constants for 2D simplex noise
    const vec4 C = vec4(
        0.211324865405187,  // (3.0-sqrt(3.0))/6.0
        0.366025403784439,  // 0.5*(sqrt(3.0)-1.0)
        -0.577350269189626, // -1.0 + 2.0 * C.x
        0.024390243902439   // 1.0 / 41.0
    );

    // First corner
    vec2 i  = floor(v + dot(v, C.yy));
    vec2 x0 = v - i + dot(i, C.xx);

    // Other corners
    vec2 i1 = (x0.x > x0.y) ? vec2(1.0, 0.0) : vec2(0.0, 1.0);
    vec4 x12 = x0.xyxy + C.xxzz;
    x12.xy -= i1;

    // Permutations to avoid truncation effects
    i = mod289(i);
    vec3 p = permute(permute(i.y + vec3(0.0, i1.y, 1.0)) + i.x + vec3(0.0, i1.x, 1.0));

    // Compute noise contributions from each corner
    vec3 m = max(0.5 - vec3(dot(x0, x0), dot(x12.xy, x12.xy), dot(x12.zw, x12.zw)), 0.0);
    m = m * m;
    m = m * m;

    // Compute gradient vectors
    vec3 x = 2.0 * fract(p * C.www) - 1.0;
    vec3 h = abs(x) - 0.5;
    vec3 ox = floor(x + 0.5);
    vec3 a0 = x - ox;

    // Normalize gradients
    m *= 1.79284291400159 - 0.85373472095314 * (a0 * a0 + h * h);

    // Compute final noise value
    vec3 g;
    g.x  = a0.x  * x0.x  + h.x  * x0.y;
    g.yz = a0.yz * x12.xz + h.yz * x12.yw;
    return 130.0 * dot(m, g);
}


// Author Attribution:
// This function combines simplex noise with procedural techniques to simulate static interference,
// a common visual effect in retro-style graphics and glitch art. The implementation uses 2D simplex noise
// (originally developed by Ken Perlin and adapted by Ashima Arts) to generate the random noise patterns.
// Additionally, the use of `step` and `pow` functions helps to control the distribution and intensity of the static effect.
// The techniques used here are inspired by shader coding practices found in various graphics communities,
// with particular reference to procedural noise generation.
// References:
// - Ken Perlin's Noise: https://cs.nyu.edu/~perlin/noise/
// - Ashima Arts Simplex Noise: https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
float static_v(vec2 uv) {
    // Calculate static parameters using simplex noise and time-based factors
    float static_height = snoise(vec2(9.0, u_Time * 1.2 + 3.0)) * 0.3 + 5.0;
    float static_amount = snoise(vec2(1.0, u_Time * 1.2 - 6.0)) * 0.1 + 0.3;
    float static_strength = snoise(vec2(-9.75, u_Time * 0.6 - 3.0)) * 2.0 + 2.0;

    // Generate static using a combination of noise and step functions
    float noise_value = snoise(vec2(
        5.0 * pow(u_Time, 2.0) + pow(uv.x * 7.0, 1.2),
        pow((mod(u_Time, 100.0) + 100.0) * uv.y * 0.3 + 3.0, static_height)
    ));

    // Calculate final static value by scaling and adjusting the noise
    float static_value = (1.0 - step(noise_value, static_amount)) * static_strength;

    return static_value;
}


// Author Attribution:
// This function combines various noise techniques to create a distorted visual effect reminiscent of analog TV or VHS
// artifacts. The implementation leverages simplex noise functions, which are an efficient alternative to Perlin noise,
// to generate random offsets and disruptions in the visual content. The techniques used here are inspired by shader
// programming practices often discussed in the context of retro-style visual effects and glitch art.
// References:
// - Ken Perlin's Noise: https://cs.nyu.edu/~perlin/noise/
// - Ashima Arts Simplex Noise: https://github.com/ashima/webgl-noise/blob/master/src/noise2D.glsl
// - ShaderToy examples: https://www.shadertoy.com/
vec4 distorted(sampler2D colortex, sampler2D boardtex, float colorFactor, float boardFactor, vec2 uv) {
    float my_time = u_Time + 8.0;

    // Calculate jerk and fuzz offsets using simplex noise
    float jerk_offset = (1.0 - step(snoise(vec2(my_time * 1.3, 5.0)), 0.8)) * 0.05;
    float fuzz_offset = snoise(vec2(my_time * 15.0, uv.y * 80.0)) * 0.003;
    float large_fuzz_offset = snoise(vec2(my_time * 1.0, uv.y * 25.0)) * 0.004;

    // Vertical movement and jerk effects
    float vert_movement_on = (1.0 - step(snoise(vec2(my_time * 0.2, 8.0)), 0.4)) * vertMovementOpt;
    float vert_jerk = (1.0 - step(snoise(vec2(my_time * 1.5, 5.0)), 0.6)) * vertJerkOpt;
    float vert_jerk2 = (1.0 - step(snoise(vec2(my_time * 5.5, 5.0)), 0.2)) * vertJerkOpt;
    float y_offset = abs(sin(my_time) * 4.0) * vert_movement_on + vert_jerk * vert_jerk2 * 0.3;
    float y = mod(uv.y + y_offset, 1.0);

    // Horizontal offset due to fuzz effects
    float x_offset = (fuzz_offset + large_fuzz_offset) * horzFuzzOpt;

    // Calculate static value
    float static_val = 0.0;
    for (float offset_y = -1.0; offset_y <= 1.0; offset_y += 1.0) {
        float max_dist = 5.0 / 200.0;
        float dist = offset_y / 200.0;
        static_val += static_v(vec2(uv.x, uv.y + dist)) * (max_dist - abs(dist)) * 1.5;
    }
    static_val *= bottomStaticOpt;

    // Sample the color texture with the calculated offsets
    float red   = texture(colortex, vec2(uv.x + x_offset - 0.01 * rgbOffsetOpt, y)).r + static_val;
    float green = texture(colortex, vec2(uv.x + x_offset, y)).g + static_val;
    float blue  = texture(colortex, vec2(uv.x + x_offset + 0.01 * rgbOffsetOpt, y)).b + static_val;

    float redB   = texture(boardtex, vec2(uv.x + x_offset - 0.01 * rgbOffsetOpt, y)).r + static_val;
    float greenB = texture(boardtex, vec2(uv.x + x_offset, y)).g + static_val;
    float blueN  = texture(boardtex, vec2(uv.x + x_offset + 0.01 * rgbOffsetOpt, y)).b + static_val;

//    red += redB * boardFactor;
//    green += greenB * boardFactor;
//    blue += blueN * boardFactor;

    red = red * colorFactor + redB * boardFactor;
    green = green * colorFactor + greenB * boardFactor;
    blue = blue * colorFactor + blueN * boardFactor;


    vec3 color = vec3(red, green, blue);

    // Apply scanline effect
    float scanline = sin(uv.y * 800.0) * 0.04 * scalinesOpt;
    color -= scanline;

    return vec4(color, 1.0);
}

//        color.rgb = 0.75 - color.rgb * 0.5;

vec3 applyStripTexture(
    sampler2D tex,
    vec2 uv,
    vec2 center,
    float H,
    vec3 baseColor,
    float offset
) {
    vec2 texSize = textureSize(tex, 0);

    float dateW = H * texSize.x / texSize.y * 0.6;

    float dateL = center.x - dateW * 0.5;
    float dateR = center.x + dateW * 0.5;
    float dateU = center.y + H * 0.5;
    float dateB = center.y - H * 0.5;

    vec3 color = baseColor;

    if (uv.x > dateL && uv.x < dateR &&
        uv.y > dateB && uv.y <= dateU) {

        vec2 tUV;
        tUV.x = (uv.x - dateL) / dateW + offset;
        tUV.y = (uv.y - dateB) / H;

        color = texture(tex, tUV).rgb;

    }

    return color;
}

void main() {
    vec4 color;

    // --- wybór bazowego koloru (bez wspólnego post-processingu) ---
    if (!u_DoGlitch) {
        // Ścieżka "normalna" – tu NA PEWNO nie wołamy distorted()
        vec3 base = texture(p3d_Texture0, uv).rgb;

        if (u_DoHeXenium) {
            vec3 hex = texture(u_HeXeniumTex, uv).rgb;
            color = vec4(base + hex * u_HeXenium, 1.0);
        } else {
            color = vec4(base, 1.0);
        }
    } else {
        // Ścieżka glitchowa – tylko tu używamy distorted()
        if (u_DoYo) {
            color = distorted(u_BlackTex, u_YoTex, 4.0, u_Yo * 4.0, uv);
//            color = vec4(distorted(u_BlackTex, u_YoTex, 1.0, u_Yo, uv).rgb, 1.0);
        } else {
            if (u_Clean > 0.0) {
                vec4 base = texture(p3d_Texture0, uv);
                color = mix(
                    distorted(p3d_Texture0, u_HeXeniumTex, 1.0, u_HeXenium * 7.0, uv),
                    base,
                    u_Clean
                );
            } else {
                color = distorted(p3d_Texture0, u_HeXeniumTex, 1.0, u_HeXenium * 7.0, uv);
            }

        }
//        color = distorted(p3d_Texture0, u_HeXeniumTex, 1.0, u_HeXenium, uv);


    }

//    color = texture(u_YoTex, uv);
//    color.rgb *= 4.0;

    // --- wspólny post-processing ---

    // Tone mapping (linear in -> linear out curve)
    color.rgb = applyToneMapUncharted2(color.rgb);

//    color = texture(u_YoTex, uv);

    // Apply 3D LUT here (after tone map)
//    color.rgb = texture(u_LUT3D, color.bgr).bgr;

    if (u_Card > 0.0) {
        vec4 card = texture(u_CardTex, uv);

        #define GU 0.239
        #define GB 0.168
        #define GL 0.376
        #define GR (1.0 - GL)
        #define H (0.239 - 0.168) + 0.003
        #define SHIFT_R 0.361
        #define DATE vec2(SHIFT_R, 0.5)
        #define PLACE vec2(1.0 - SHIFT_R, 0.5)
        #define X26 vec2(0.5, 0.870)
        #define GREETZ vec2(0.5, (0.239 + 0.168) * 0.5)

        card.rgb = applyStripTexture(u_DateTex, uv, DATE, H, card.rgb, 0.0);
        card.rgb = applyStripTexture(u_PlaceTex, uv, PLACE, H, card.rgb, 0.0);
        card.rgb = applyStripTexture(u_X26Tex, uv, X26, H, card.rgb, 0.0);

        if (uv.x > GL && uv.x < GR) {
//           card.rgb = applyStripTexture(u_GreetzTex, uv, GREETZ, H, card.rgb, -0.521);
//            card.rgb = applyStripTexture(u_GreetzTex, uv, GREETZ, H, card.rgb, +0.484);
            card.rgb = applyStripTexture(u_GreetzTex, uv, GREETZ, H, card.rgb, u_Scroll);
        }

        color.rgb = mix(color.rgb, card.rgb, u_Card);
    }

    if (u_Noise > 0.0) {
        color.rgb = mix(color.rgb, vec3(rand(vec3(uv, u_Time))), u_Noise);
    }

    // Final fade
    color.rgb *= u_MasterFade;

    color.rgb = mix(color.rgb, vec3(0.5), u_White);

    p3d_FragColor = color;
//    p3d_FragColor = vec4(color.rgb, 1.0);
}