#version 410
#pragma include "filter_common.glsl"

#define STEP_VS 15  // smaller = more accurate
//#define MAX_STEPS 776  // higher = longer reach
//#define MAX_STEPS 500  // higher = longer reach
//#define MAX_DIST_VS STEP_VS * MAX_STEPS * 1.5  // +50% margin for angles

//p3d_Texture0 : original_colortex
//p3d_Texture1 : original_depthtex
//p3d_Texture2 : reflectivity_colortex

uniform float u_MaxSteps;

uniform mat4 u_SceneProj;     // P
uniform mat4 u_SceneProjInv;  // P^-1

// === SSR CONSTS (pipeline invariants) =======================================
// Heurystyki marchingu / filtrowania
const float SSR_ThicknessVS   = 35.0;
const float SSR_StartBiasN    = 3.0;
const float SSR_StartBiasR    = 4.0;
const float SSR_ReflThreshold = 0.02;
const float SSR_FaceMin       = 0.15;
// ============================================================================


// Rekonstrukcja pozycji VS z depth 0..1
vec3 reconstructVS(vec2 tc, float depth01)
{
    float z = depth01 * 2.0 - 1.0;     // NDC.z
    vec4 clip = vec4(tc * 2.0 - 1.0, z, 1.0);
    vec4 vs   = u_SceneProjInv * clip;
    return vs.xyz / vs.w;
}

void fetchVSandNormal(in vec2 tc, float d0, out vec3 Pvs, out vec3 Nvs)
{
    // Reconstruct center position in view space
    Pvs = reconstructVS(tc, d0);

    // Texel size for depth sampling
    vec2 texel = 1.0 / vec2(textureSize(p3d_Texture1, 0));

    // Fetch depth at 4 neighboring texels (R, L, U, D)
    float dR = texture(p3d_Texture1, tc + vec2(texel.x, 0.0)).r;
    float dL = texture(p3d_Texture1, tc - vec2(texel.x, 0.0)).r;
    float dU = texture(p3d_Texture1, tc + vec2(0.0, texel.y)).r;
    float dD = texture(p3d_Texture1, tc - vec2(0.0, texel.y)).r;

    // Reconstruct 4 neighbor positions in view space
    vec3 Pr = reconstructVS(tc + vec2(texel.x, 0.0), dR);
    vec3 Pl = reconstructVS(tc - vec2(texel.x, 0.0), dL);
    vec3 Pu = reconstructVS(tc + vec2(0.0, texel.y), dU);
    vec3 Pd = reconstructVS(tc - vec2(0.0, texel.y), dD);

    // Compute gradients
    vec3 dx = Pr - Pl;
    vec3 dy = Pu - Pd;

    // Compute normal from depth gradients
    Nvs = normalize(cross(dx, dy));
}


// Jitter tylko na starcie (wygładza „schodki” bez kosztu)
float hash12(vec2 p) {
    // tani hash bez sin/cos
    vec3 p3  = fract(vec3(p.xyx) * 0.1031);
    p3 += dot(p3, p3.yzx + 33.33);
    return fract((p3.x + p3.y) * p3.z);
}

// ======= GŁÓWNY FRAGMENT =======
void main()
{
    // Initial reflectivity gate
    if (texture(p3d_Texture2, uv).r <= SSR_ReflThreshold) {
        p3d_FragColor = vec4(0.0);
        return;
    }

    // Jeśli bieżący piksel to "pustka" (depth 1.0), SSR nie ma sensu
    float d0 = texture(p3d_Texture1, uv).r;
    if (d0 >= 0.9999) {
        p3d_FragColor = vec4(0.0);
        return;
    }

    // Rekonstruuj punkt i normalną w VS z depth
    vec3 P, N;
    fetchVSandNormal(uv, d0, P, N);

    // Kierunek widoku (kamera w (0,0,0) VS):
    vec3 V = normalize(-P);

    // Odbicie kierunku widoku po normalnej:
    vec3 R = reflect(-V, N);

    // krótszy krok przy kątach stycznych
    float face0 = abs(dot(R, N));
    float stepLen = STEP_VS * (0.35 + face0 * 0.65);

    // Start promienia: tuż nad powierzchnią + minimalne "pchnięcie” wzdłuż R
    vec3 X = P + N * SSR_StartBiasN + R * SSR_StartBiasR;

    // Jitter tylko na starcie (wygładza "schodki” bez kosztu)
    float j = hash12(gl_FragCoord.xy);
    X += R * (j * stepLen); // przesunięcie startu o [0..1)*stepLen

    bool  hit = false;
    vec2  uvHit = uv;
    float traveled = 0.0;

    for (int i = 0; i < u_MaxSteps; ++i) {
        if (traveled > STEP_VS * 1.5 * u_MaxSteps) { break; }

        vec4 clip = u_SceneProj * vec4(X, 1.0);

        // Ray is behind camera? No point marching further!
        if (clip.w <= 0.0)
            break;

        // compute UV ONLY if clip.w > 0
        float iw = 0.5 / clip.w;
        vec2 uv2 = clip.xy * iw + 0.5;

        // UV out of bounds = also uselessm so break
        if (uv2.x <= 0.0 || uv2.x >= 1.0 || uv2.y <= 0.0 || uv2.y >= 1.0)
            break;

        // Pozycja sceny w tym UV (z depth)
        float dS = texture(p3d_Texture1, uv2).r;

        // Brak geometrii / skybox -> kontynuuj marsz
        if (dS >= 0.9999) {
            X += R * stepLen;
            traveled += stepLen;
            continue;
//            break;   // KONIEC, nie ma sensu iść dalej
        }

        vec3 QS = reconstructVS(uv2, dS);

        // Test przecięcia wzdłuż osi „głębi VS”:
        float rayG = -X.y;
        float sceneG = -QS.y;

        // Proste okno grubości
        float dz = sceneG - rayG;

        if (dz > 0.0 && dz <= SSR_ThicknessVS) {
            hit   = true;
            uvHit = uv2;
            break;
        }

        X        += R * stepLen;
        traveled += stepLen;

    }

    vec3 col = hit ? texture(p3d_Texture0, uvHit).rgb : vec3(0.0);
    p3d_FragColor = vec4(col, 1.0);
}

// FOR FUTURE COMMON WITH SSAO - DO NOT REMOVE
//void fetchVSandNormal(in vec2 tc, out vec3 Pvs, out vec3 Nvs)
//{
//    // Pozycja z depth – zostaje jak było
//    float d  = texture(p3d_Texture1, tc).r;
//    Pvs      = reconstructVS(tc, d);
//
//    // Normalna z tekstury normalnych
//    vec3 Nenc = texture(p3d_Texture3, tc).xyz;   // [0,1]
//    Nvs = Nenc * 2.0 - 1.0;                      // [-1,1]
//    Nvs = normalize(Nvs);
//
//    // Zachowanie „połowa przestrzeni: normalna w stronę kamery”
//    // zostaje bez zmian:
//    vec3 viewDir = -normalize(Pvs);
//    if (dot(Nvs, viewDir) < 0.0)
//        Nvs = -Nvs;
//
//
//}
