// Shadow filters (PCF and Poisson)
// - PCF = regular grid, stable
// - Poisson = random-like pattern, better quality per tap but less stable

// --- Hard ---
float sh_hard(sampler2DShadow smap, vec4 coord) {
    return textureProj(smap, coord);
}

// --- PCF 3x3 (9 taps) ---
float sh_pcf3(sampler2DShadow smap, vec4 coord, float spread) {
    float sh = 0.0;
    for (int si = -1; si <= 1; ++si)
    for (int sj = -1; sj <= 1; ++sj) {
        sh += textureProj(smap, coord + vec4(vec2(si, sj) * spread, vec2(0.0)));
    }
    return sh * (1.0 / 9.0);
}

// --- PCF 5x5 (25 taps) ---
float sh_pcf5(sampler2DShadow smap, vec4 coord, float spread) {
    float sh = 0.0;
    for (int si = -2; si <= 2; ++si)
    for (int sj = -2; sj <= 2; ++sj) {
        sh += textureProj(smap, coord + vec4(vec2(si, sj) * spread, vec2(0.0)));
    }
    return sh * (1.0 / 25.0);
}

// --- PCF 7x7 (49 taps) ---
float sh_pcf7(sampler2DShadow smap, vec4 coord, float spread) {
    float sh = 0.0;
    for (int si = -3; si <= 3; ++si)
    for (int sj = -3; sj <= 3; ++sj) {
        sh += textureProj(smap, coord + vec4(vec2(si, sj) * spread, vec2(0.0)));
    }
    return sh * (1.0 / 49.0);
}

// --- Poisson samples ---
// Precomputed disk offsets (normalized to [-1,1] range).
const vec2 POISSON8[8] = vec2[](
    vec2(-0.326, -0.406), vec2(-0.840, -0.074),
    vec2(-0.696,  0.457), vec2(-0.203,  0.621),
    vec2( 0.962, -0.195), vec2( 0.473, -0.480),
    vec2( 0.519,  0.767), vec2(-0.185, -0.893)
);

const vec2 POISSON16[16] = vec2[](
    vec2(-0.942, -0.399), vec2( 0.945, -0.768),
    vec2(-0.094, -0.929), vec2( 0.344,  0.293),
    vec2(-0.915,  0.793), vec2(-0.878, -0.684),
    vec2( 0.107, -0.965), vec2(-0.530, -0.213),
    vec2( 0.988,  0.463), vec2( 0.106,  0.355),
    vec2( 0.646, -0.606), vec2( 0.343, -0.879),
    vec2(-0.666,  0.555), vec2(-0.105, -0.818),
    vec2( 0.519,  0.767), vec2( 0.185, -0.893)
);

// --- Poisson 8 taps ---
float sh_pois8(sampler2DShadow smap, vec4 coord, float spread) {
    float sh = 0.0;
    for (int i = 0; i < 8; ++i) {
        sh += textureProj(smap, coord + vec4(POISSON8[i] * spread, 0.0, 0.0));
    }
    return sh * (1.0 / 8.0);
}

// --- Poisson 16 taps ---
float sh_pois16(sampler2DShadow smap, vec4 coord, float spread) {
    float sh = 0.0;
    for (int i = 0; i < 16; ++i) {
        sh += textureProj(smap, coord + vec4(POISSON16[i] * spread, 0.0, 0.0));
    }
    return sh * (1.0 / 16.0);
}

// Dispatcher by "samples" selector (float -> int mode)
// 0=hard, 1=pcf3, 2=pcf5, 3=pcf7, 4=pois8, 5=pois16
float sh(sampler2DShadow smap, vec4 coord, float spread, float samples) {
    int mode = int(samples); // CPU clamps to 0..5

    if (mode == 0) return sh_hard(smap, coord);
    if (mode == 1) return sh_pcf3(smap, coord, spread);
    if (mode == 2) return sh_pcf5(smap, coord, spread);
    if (mode == 3) return sh_pcf7(smap, coord, spread);
    if (mode == 4) return sh_pois8(smap, coord, spread * 1.8);   // Poisson8 gets a larger radius
    if (mode == 5) return sh_pois16(smap, coord, spread * 1.5);  // Poisson16 spread scaled
    return sh_hard(smap, coord);
}