#version 410

//#define MAX_LIGHTS 3

#pragma include "p3d/uniform.glsl"
#pragma include "lib/shadow.glsl"
#pragma include "lib/constants.glsl"
#pragma include "lib/color.glsl"
#pragma include "lib/math.glsl"

// Uniform inputs
uniform float u_MaxShininess;
uniform float u_RimLightIntensity;
uniform float u_ShadowSamples;
uniform float u_ShadowSpread;
uniform float u_SpecularBlendStrength;
uniform int u_NumberOfLights;
uniform float u_SpecularPower;
uniform float u_Roughness;
uniform float u_Brightness;
uniform float u_Saturation;
uniform float u_Reflectivity;

uniform sampler2D u_DemoTex;
uniform sampler2D u_CodeTex;
uniform sampler2D u_MsxTex;
uniform sampler2D u_GfxTex;
uniform int u_TexNo;

uniform vec3 u_TexOrder;

// Input from vertex shader
in vec4 vertInShadowSpaces[MAX_LIGHTS];
in vec4 vertColor;
in vec4 vertPosition;
in vec2 vertMultiTexCoord0;
in vec3 vertNormal;
in vec3 tangent;
in float neg;

in vec3 opos;

// Output to the screen
layout(location = 0) out vec4 fragColor;
layout(location = 1) out vec4 reflectivityOut;

#define MAX_FRESNEL_POWER 5.0
#define MAX_SHININESS     127.75

#define XMIN -0.91
#define XMAX +1.78
#define YMIN -2.25
#define YMAX +1.29
#define W (XMAX - XMIN)
#define H (YMAX - YMIN)

// Clamp value to [0,1] (short alias)
float sat(float x) { return clamp(x, 0.0, 1.0); }

void main() {

    vec4 albedo = texture(p3d_TextureModulate[0], vertMultiTexCoord0) * vertColor;
    vec4 diffuseColor = color_ops(albedo, u_Saturation, u_Brightness);

    if (
        opos.z > 1.0 &&
        diffuseColor.g > 0.05 &&
        opos.x > XMIN &&
        opos.x < XMAX &&
        opos.y > YMIN &&
        opos.y < YMAX
    ) {
        vec2 uv;
        uv.x = (opos.x - XMIN) / W;
        uv.y = (opos.y - YMIN) / H;
        float noise = rand(vec3(uv, osg_FrameTime));
//        float noise = rand(vec3(round(uv * 100.0) / 100.0, osg_FrameTime));
        vec3 probe;
        if (u_TexNo == 0)
            probe = texture(u_DemoTex, -uv).rgb;
        else if (u_TexNo == 1)
            probe = texture(u_CodeTex, -uv).rgb;
        else if (u_TexNo == 2)
            probe = texture(u_MsxTex, -uv).rgb;
        else if (u_TexNo == 3)
            probe = texture(u_GfxTex, -uv).rgb;
        else
            probe = vec3(0.0);
        diffuseColor.rgb = diffuseColor.rgb * 1.0 + probe * 6.0 + vec3(noise) * 0.5;
    }

    vec4 selector = texture(p3d_TextureSelector[0], vertMultiTexCoord0);
    float ao        = selector.r;
    float roughness = selector.g;
    float metallic  = selector.b;

    float baseReflectivity = 1.0 - roughness;
    baseReflectivity = mix(baseReflectivity, 1.0, metallic);
    float reflectivity = clamp(baseReflectivity * u_Reflectivity, 0.0, 1.0);

//    float dielectricF0 = 0.04;
//    float albedoLuma   = dot(albedo.rgb, vec3(0.2126, 0.7152, 0.0722));
//    float f0           = mix(dielectricF0, albedoLuma, metallic);
//    float glossFactor  = 1.0 - roughness;
//    float baseReflectivity = f0 * glossFactor;
//    float reflectivity     = clamp(baseReflectivity * u_Reflectivity, 0.0, 1.0);

//    float reflectivity = u_Reflectivity;

    float shininess = (1.0 - roughness) * MAX_SHININESS * u_MaxShininess;

    vec4 diffuse  = vec4(0.0, 0.0, 0.0, diffuseColor.a);
    vec4 specular = vec4(0.0, 0.0, 0.0, diffuseColor.a);
    vec4 rim      = vec4(0.0, 0.0, 0.0, diffuseColor.a);

    vec3 Nv = normalize(vertNormal);
    vec3 Tv = normalize(tangent - Nv * dot(tangent, Nv)); // upewnij T ⟂ N
    vec3 Bv = cross(Nv, Tv);                               // brak sign — prosto i szybko

    vec3 nMap = normalize(texture(p3d_TextureNormal[0], vertMultiTexCoord0).rgb * 2.0 - 1.0);
    vec3 normal = normalize(mat3(Tv, Bv, Nv) * nMap);

    vec3 eyeDirection = normalize(-vertPosition.xyz);

    vec3 test;

    for (int i = 0; i < u_NumberOfLights; ++i) {  // i = 0

        vec3 lightDirection = p3d_LightSource[i].position.xyz - vertPosition.xyz * p3d_LightSource[i].position.w;

        vec3 unitLightDirection = normalize(lightDirection);
        vec3 halfwayDirection   = normalize(unitLightDirection + eyeDirection);

        float lightDistance = length(lightDirection);

        float attenuation = 1 / (
            p3d_LightSource[i].constantAttenuation +
            p3d_LightSource[i].linearAttenuation * lightDistance +
            p3d_LightSource[i].quadraticAttenuation * (lightDistance * lightDistance)
        );

        if (attenuation <= 0.0) { continue; }

        // Diffuse
        float diffuseIntensity  = dot(normal, unitLightDirection);
        if (diffuseIntensity < 0.0) { continue; }
        vec4 diffuseTemp = vec4(
            clamp(diffuseColor.rgb * p3d_LightSource[i].diffuse.rgb * diffuseIntensity, 0.0 , 1.0), diffuseColor.a
        );


        // Specular
        float specularIntensity = sat(dot(normal, halfwayDirection));
        vec4 lightSpecularColor = p3d_LightSource[i].specular;
        vec4 materialSpecularColor = vec4(vec3(1.0 - roughness), diffuseColor.a);

        float fresnelFactor = dot(halfwayDirection, eyeDirection);
        fresnelFactor = max(fresnelFactor, 0.0);
        fresnelFactor = 1.0 - fresnelFactor;
        fresnelFactor = pow(fresnelFactor, MAX_FRESNEL_POWER);
        materialSpecularColor.rgb = mix(
            materialSpecularColor.rgb,
            vec3(u_SpecularBlendStrength),
            sat(fresnelFactor)
        );

        vec4 specularTemp = vec4(vec3(0.0), diffuseColor.a);
        specularTemp.rgb  = lightSpecularColor.rgb * pow(specularIntensity, shininess);
        specularTemp.rgb *= materialSpecularColor.rgb;
        specularTemp.rgb  = clamp(specularTemp.rgb, 0.0, 1.0);  // Comment to have more specular
        specularTemp.rgb *= u_SpecularPower;

        // Spot
        float unitLightDirectionDelta = dot(normalize(p3d_LightSource[i].spotDirection), -unitLightDirection);
        if (unitLightDirectionDelta < p3d_LightSource[i].spotCosCutoff) { continue; }
        float spotExponent = p3d_LightSource[i].spotExponent;
        diffuseTemp.rgb *= (spotExponent <= 0.0 ? 1.0 : pow(unitLightDirectionDelta, spotExponent));

        // Shadow
        float shadow = sh(
            p3d_LightSource[i].shadowMap,
            vertInShadowSpaces[i],
            u_ShadowSpread,
            u_ShadowSamples // float selector: 0..4
        );

        diffuseTemp.rgb  *= shadow;
        specularTemp.rgb *= shadow;

        diffuseTemp.rgb  *= attenuation;
        specularTemp.rgb *= attenuation;

        diffuse  += diffuseTemp;
        specular += specularTemp;

        // Rim
        vec4 rimLight = vec4(0.0);
        rimLight.rgb = vec3(1.0 - max(0.0, dot(normalize(-vertPosition.xyz), normalize(normal))));
        rimLight.rgb = pow(rimLight.rgb, vec3(2.0)) * u_RimLightIntensity;
        rimLight.rgb *= diffuse.rgb;
        rim += rimLight;

    }

    vec4 ambient = p3d_Material.ambient * p3d_LightModel.ambient * diffuseColor;

    vec3 color = diffuse.rgb + rim.rgb + specular.rgb + ambient.rgb + p3d_Material.emission.rgb;

    fragColor = vec4(color, diffuseColor.a);
//    reflectivityOut = vec4(vec3(u_Reflectivity), 1.0);
    reflectivityOut = vec4(vec3(reflectivity), 1.0);
}
