use anyhow::{Context, Result, bail};
use sdl3::gpu::TextureFormat;
use serde::Deserialize;
use std::{
    fs,
    path::{Path, PathBuf},
    str::FromStr,
};

const fn one() -> f32 {
    1.
}

#[derive(Debug, Clone, Deserialize)]
pub struct Audio {
    pub name: Option<PathBuf>,
    pub bpm: f32,
}

#[derive(Debug, Clone)]
pub struct RGB(f32, f32, f32);

impl Default for RGB {
    fn default() -> Self {
        Self(1., 1., 1.)
    }
}

impl FromStr for RGB {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> std::result::Result<Self, Self::Err> {
        if s.len() != 7 {
            bail!("RGB hex triplet must be 7 characters long");
        }
        if !s.starts_with('#') {
            bail!("RGB hex triplet must start with a '#'");
        }
        let parse = |s: &str| -> Result<f32> {
            u8::from_str_radix(s, 16)
                .map(|b| f32::from(b) / 255.)
                .context("RGB hex triplet must contain only hexadecimal")
        };
        let r = parse(&s[1..3])?;
        let g = parse(&s[3..5])?;
        let b = parse(&s[5..7])?;
        Ok(RGB(r, g, b))
    }
}

impl<'de> Deserialize<'de> for RGB {
    fn deserialize<D>(deserializer: D) -> std::result::Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let s = String::deserialize(deserializer)?;
        RGB::from_str(&s).map_err(serde::de::Error::custom)
    }
}

#[derive(Debug, Clone, Deserialize)]
pub struct RGBA(RGB, #[serde(default = "one")] f32);

impl Default for RGBA {
    fn default() -> Self {
        Self(RGB::default(), 1.)
    }
}

impl From<&RGBA> for glam::Vec4 {
    fn from(value: &RGBA) -> Self {
        glam::Vec4::new(value.0.0, value.0.1, value.0.2, value.1)
    }
}

#[derive(Debug, Clone, Copy, Deserialize)]
pub struct Resolution {
    pub width: u32,
    pub height: u32,
}

#[derive(Debug, Clone, Deserialize)]
pub enum TextSource {
    File { name: PathBuf },
    Inline { text: String },
    Stdin,
    Version,
    Fps,
    Command { cmd: String, args: Vec<String> },
}

#[derive(Debug, Clone, Deserialize)]
pub enum TextColor {
    Flat(RGBA),
    Gradient(RGBA, RGBA, RGBA, RGBA),
}

impl Default for TextColor {
    fn default() -> Self {
        Self::Flat(RGBA::default())
    }
}

#[derive(Debug, Clone, Default, Deserialize)]
pub struct TextLayer {
    #[serde(default)]
    pub offset: (f32, f32),
    #[serde(default)]
    pub color: TextColor,
}

#[derive(Debug, Clone, Deserialize)]
pub enum Texture {
    Framebuffer(usize),
}

#[derive(Debug, Clone, Deserialize)]
pub struct Shader {
    pub name: PathBuf,
    #[serde(default)]
    pub textures: Vec<Texture>,
}

#[derive(Debug, Clone, Deserialize)]
pub struct Text {
    pub layers: Vec<TextLayer>,
    pub sources: Vec<TextSource>,
    pub refresh: Option<f32>,
    #[serde(default)]
    pub cursor: bool,
}

#[derive(Debug, Clone, Deserialize)]
pub enum Draw {
    Shader(Shader),
    Text(Text),
}

#[derive(Debug, Clone, Deserialize)]
pub struct Pass {
    pub draw: Vec<Draw>,
    pub targets: Vec<usize>,
}

#[derive(Debug, Clone, Deserialize, Default)]
pub struct Output {
    pub draw: Vec<Draw>,
}

#[derive(Debug, Clone, Deserialize)]
pub enum FramebufferFormat {
    R8g8b8a8Unorm,
    R8Unorm,
    R16Float,
    R16g16Float,
    R16g16b16a16Float,
    R32Float,
    R32g32Float,
    R32g32b32a32Float,
}

impl From<&FramebufferFormat> for TextureFormat {
    fn from(value: &FramebufferFormat) -> Self {
        match value {
            FramebufferFormat::R8g8b8a8Unorm => Self::R8g8b8a8Unorm,
            FramebufferFormat::R8Unorm => Self::R8Unorm,
            FramebufferFormat::R16Float => Self::R16Float,
            FramebufferFormat::R16g16Float => Self::R16g16Float,
            FramebufferFormat::R16g16b16a16Float => Self::R16g16b16a16Float,
            FramebufferFormat::R32Float => Self::R32Float,
            FramebufferFormat::R32g32Float => Self::R32g32Float,
            FramebufferFormat::R32g32b32a32Float => Self::R32g32b32a32Float,
        }
    }
}

#[derive(Debug, Clone, Deserialize)]
pub struct Render {
    pub resolution: Resolution,
    #[serde(default)]
    pub framebuffers: Vec<FramebufferFormat>,
    #[serde(default)]
    pub passes: Vec<Pass>,
    pub output: Output,
}

#[derive(Debug, Clone, Deserialize)]
pub struct Config {
    pub audio: Audio,
    pub render: Render,
}

impl Config {
    pub fn load(path: impl AsRef<Path>) -> Result<Config> {
        let path = path.as_ref();
        let src =
            fs::read_to_string(path).with_context(|| format!("Can't access {}", path.display()))?;

        let conf: Config =
            ron::from_str(&src).with_context(|| format!("Can't parse {}", path.display()))?;

        Ok(conf)
    }
}

impl Default for Config {
    fn default() -> Self {
        Config {
            audio: Audio {
                name: None,
                bpm: 120.,
            },
            render: Render {
                resolution: Resolution {
                    width: 1280,
                    height: 720,
                },
                framebuffers: vec![],
                passes: vec![],
                output: Output::default(),
            },
        }
    }
}
