use std::{
    sync::{
        Arc,
        atomic::{AtomicU32, Ordering},
    },
    time::{Duration, Instant},
};

#[derive(Clone)]
pub struct FpsValue {
    fps: Arc<AtomicU32>,
}

impl FpsValue {
    pub fn get(&self) -> u32 {
        self.fps.load(Ordering::Acquire)
    }
}

#[derive(Clone)]
pub struct FpsCounter {
    frames: u32,
    timestamp: Instant,
    reading: FpsValue,
}

impl FpsCounter {
    pub fn new() -> Self {
        Self {
            frames: 0,
            timestamp: Instant::now(),
            reading: FpsValue {
                fps: Arc::new(AtomicU32::new(0)),
            },
        }
    }

    pub fn frame(&mut self) {
        if self.timestamp.elapsed() >= Duration::from_secs(1) {
            self.timestamp = Instant::now();
            self.reading.fps.store(self.frames, Ordering::Release);
            self.frames = 0;
        }
        self.frames += 1;
    }

    pub fn value(&self) -> &FpsValue {
        &self.reading
    }
}
