#ifndef VERTEX_MODIFIER_GERSTNER_WAVES_H
#define VERTEX_MODIFIER_GERSTNER_WAVES_H

#include <shaders/materials/noise/noise3Dgrad.glsl>
#include <shaders/materials/commons_curves.glsl>

struct VertexModifierGerstnerWavesParams
{
	mat4  transform_world_to_local;
	mat4  transform_local_to_world;
	mat4  transform_local;
	mat4  transform_local_inv;
	mat4  transform_parent;
	mat4  transform_parent_inv;

	uvec4 wave_directions[16];	// actually 64 directions. each is halffloat vec2

	vec4  noise_phase;
	vec3  domain_scale;
	int   _pad0;
	vec3  strength;
	int   _pad1;
	vec3  position;
	int   noise_relative_to_modifier;
	int   displacement_mode;  // 0 - along normal; 1 - radial
	float noise_value_base;
	int   noise_value_absolute;
	float wave_length;
	float wave_steepness;
	int   wave_octaves;
	float wave_octave_scale;
};

vec2 gerstner_evaluate_single(vec2 p, float t, float steepness, float wave_length, vec2 wave_direction)
{
	float k = 2.0 * M_PI / wave_length;
	float c = sqrt(9.8 / k);
	float f = k * (dot(wave_direction, p) - c * t);
	float a = steepness / k;

	return vec2(a * cos(f), a * sin(f));
}

void vertex_modifier_gerstner_waves_apply(VertexModifierGerstnerWavesParams params, inout ModifierFactor modifier_factor, inout VertexInput vtx)
{

	// for influence move object to the noise modifier local space
	vec3 modifier_position = vec3(0.0);

	vec3 base_pos = vtx.pos;

	// move to modifier local space
	//vec3 influence_pos = vector_transform_by_mat43(vtx.pos, params.transform_parent_inv * params.transform_local_inv * transform_params.mModel); // <- when disconnected, globals
	vec3 influence_pos = vtx.pos;
	vec3 noise_pos = influence_pos;

	if (params.noise_relative_to_modifier == 0)
		noise_pos = vector_transform_by_mat43(noise_pos, params.transform_local);

	//vec3 rel_coords;
	//vec3 influence_rel_coords;
	//rel_coords = base_pos;
	//influence_rel_coords = influence_pos;

	float influence = modifier_factor.factor;

	if (influence > 0.0)
	{
		vec2 wave = vec2(0.0, 0.0);
		float wave_length = params.wave_length;
		for(int wi = 0; wi < params.wave_octaves; wi++)
		{
			// wave length also shapes amplitude, no need to adjust it
			

			// wave direction
			vec2 wave_direction;
			#if 0
			if ((wi & 1) != 0)
				wave_direction = params.wave_directions[wi >> 1].xy;
			else
				wave_direction = params.wave_directions[wi >> 1].zw;
			#else
			if ((wi & 3) == 0)
				wave_direction = unpackHalf2x16(params.wave_directions[wi >> 2].x);
			if ((wi & 3) == 1)
				wave_direction = unpackHalf2x16(params.wave_directions[wi >> 2].y);
			if ((wi & 3) == 2)
				wave_direction = unpackHalf2x16(params.wave_directions[wi >> 2].z);
			if ((wi & 3) == 3)
				wave_direction = unpackHalf2x16(params.wave_directions[wi >> 2].w);
			#endif

			// wave computation makes sure we don't create loops but it assumes we don't rescale the vector later
			// NOTE: Also, for multiple waves we DO NOT perform normalization of steepness (it has to stal below 1)
			float steepness_modulated_by_strength = params.wave_steepness / params.strength.x;
			wave += gerstner_evaluate_single(
				noise_pos.xz * params.domain_scale.xy + params.domain_scale.xy * float(wi),
				params.noise_phase.x,
				steepness_modulated_by_strength,
				wave_length,
				wave_direction);

			wave_length *= params.wave_octave_scale;
		}

		wave += params.noise_value_base;
		vec3 local_norm = vtx.norm;

		if (params.displacement_mode == 1) // along normal
			local_norm = normalize(base_pos - vector_transform_by_mat43(modifier_position, params.transform_local));

		vec3 new_pos = base_pos;
		new_pos.x += wave.x * params.strength.x;
		new_pos.y += wave.y * params.strength.y;
		new_pos = mix(base_pos, new_pos, influence);
		vtx.pos = new_pos;
	}
	
	// debug influence
	if (false)
	{
		vtx.color.rgb = vec3(0.0);
		vtx.color.r   = max(0.0, influence);
	}

}

#endif // VERTEX_MODIFIER_GERSTNER_WAVES_H


