#version 450
#extension GL_EXT_debug_printf : enable

#define LIGHT_PROPERTIES_BINDING 1
#define MATERIAL_PROPERTIES_BINDING 1

#define USE_AMBIENT_OCCLUSION_TERM

uniform sampler2D   sDepth;
uniform usampler2D  sNormalMaterial;
uniform sampler2D   sAlbedo;
uniform sampler2D   sEmissive;
uniform sampler2D   sIBL;
uniform usampler2D  sMetalnessRoughnessMaterialTags;	// metalness roughness material index
uniform sampler2D   sRaytrace;
uniform usampler2D  sRaytraceHitInfo;

uniform sampler2D   sScreenSpaceOcclusion;
uniform sampler2D   sVoxelLighting;
uniform sampler2D   sVoxelOcclusion;

#include <shaders/materials/commons_deferred.glsl>
#include <shaders/materials/commons.glsl>

struct DeferredRenderLightsParams
{
	vec2  frustum_shift;		// TODO: Remove this. It is still here because of the non-standard way this code is invoked
	vec2  resolution;
	int   lights_num;
	float env_map_intensity;
	float raytrace_scaling_factor;
	float raytrace_strength;
	
};

layout(std140, row_major) uniform BasicDeferredParamsBuffer{
	BasicDeferredParams basic_params;
};

layout(std140, row_major) uniform DeferredRenderLightsParamsBuffer {
	DeferredRenderLightsParams render_lights_params;
};

layout (std140, row_major) uniform DeferredCompositeSetupBuffer {
	DeferredCompositeSetup composite_setup;
};

uniform sampler2D      s_LTC1;
uniform sampler2D      s_LTC2;
uniform sampler2D      s_BRDF;
uniform sampler2DArray s_BlueNoise;

in vec2 vTexcoord0;
in vec4 vFrustum;

// output

out vec4 outColor;

#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/commons.glsl>
#include <shaders/deferred/lighting/lighting_support.glsl>

float linearizeDepth(float d)
{
	return basic_params.camera_near_far_plane.z / (basic_params.camera_near_far_plane.y + basic_params.camera_near_far_plane.x - d * basic_params.camera_near_far_plane.w);
}

vec3 positionFromDepth(vec3 vDirection, float depth)
{
	return vDirection.xyz * depth;
}

float sampleShadowDist(in sampler2DShadow smpl, in vec4 coords, out float in_frustum, out vec4 projector_color, out float shadow_dist)
{
	in_frustum = 0.0;

	// no shading if fragment behind
	if (coords.w <= 0.0)
		return 0.0;

	// no shading if fragment outside of light frustum
	if (coords.x < -coords.w || coords.x > coords.w || coords.y < -coords.w || coords.y > coords.w)
		return 1.0;

	in_frustum = 1.0;

	coords.xy = coords.xy * vec2(0.5) + vec2(0.5) * coords.w;
	vec3 samp = coords.xyz / coords.w;

	float shadow = textureProj(smpl, coords);

	// TODO: Reimplement
	projector_color = vec4(0.0);
	shadow_dist = 0.0;
	//projector_color = texture(LightProjectorSamplers[light.projector_sampler], coords.xy / coords.w);
	//shadow_dist = texture(LightShadowmapCmpSamplers[light.projector_sampler], coords.xy / coords.w).r - coords.z / coords.w;
	return shadow;
}

vec4 sampleProjectorTexture(in sampler2D smpl, in vec4 coords)
{
	vec4 color = vec4(0.0);

	// no shading if fragment behind
	if (coords.w <= 0.0)
		return color;

	// no shading if fragment outside of light frustum
	if (coords.x < -coords.w || coords.x > coords.w || coords.y < -coords.w || coords.y > coords.w)
		return color;

	coords.xy = coords.xy * vec2(0.5) + vec2(0.5) * coords.w;

#ifdef SPIRV_VULKAN
	coords.y = coords.y /coords.w;
	coords.y = 1.0 - coords.y;
	coords.y = coords.y * coords.w;
#endif

	vec3 samp = coords.xyz / coords.w;
	color = texture(smpl, coords.xy / coords.w);
	//shadow_dist = texture(LightShadowmapCmpSamplers[light.projector_sampler], coords.xy / coords.w).r - coords.z / coords.w;
	return color;
}

float calculate_shadow_for_position(in LightProperties light, vec3 world, float depth, out vec4 cascadeColor, out vec4 projector_color)
{
	vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);
	float in_frustum;
	cascadeColor = vec4(1.0, 0.1, 0.1, 1.0);
	float shadow = sampleShadow(LightShadowmapCmpSamplers[light.shadowmap_sampler0], vShadowCoords, in_frustum, projector_color);
	shadow *= in_frustum;
	return shadow;
}

float calculate_shadow_dist_for_position(in LightProperties light, vec3 world, float depth, out vec4 cascadeColor, out vec4 projector_color, out float shadow_dist)
{
	vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);
	float in_frustum;
	cascadeColor = vec4(1.0, 0.1, 0.1, 1.0);
	float shadow = sampleShadowDist(LightShadowmapCmpSamplers[light.shadowmap_sampler0], vShadowCoords, in_frustum, projector_color, shadow_dist);
	shadow *= in_frustum;
	projector_color *= in_frustum;
	return shadow;
}

// ray-cone intersection. could be removed when we finally use the bounding primitive
// code based on https://www.shadertoy.com/view/MtcXWr

struct Cone
{
	float cosa;	// half cone angle
	float h;	// height
	vec3 c;		// tip position
	vec3 v;		// axis
};

struct Ray
{
	vec3 o;		// origin
	vec3 d;		// direction
};

bool inside_light_cone(vec3 p0, vec3 p, float angle, float height)
{
	float hsquared = height*height;
	float cosangle = angle;
	float cDistance = dot(p, p);
	return cDistance<=hsquared && dot(p0, p) >= sqrt(cDistance)*cosangle;
}

bool intersect_cone(Cone s, Ray r, out float v)
{
	v = 0.0;

	if (inside_light_cone(s.v, r.o - s.c, s.cosa, s.h))
	{
	//	v = 0.5;
		return true;
	}

	vec3 co = r.o - s.c;

	float a = dot(r.d,s.v)*dot(r.d,s.v) - s.cosa*s.cosa;
	float b = 2. * (dot(r.d,s.v)*dot(co,s.v) - dot(r.d,co)*s.cosa*s.cosa);
	float c = dot(co,s.v)*dot(co,s.v) - dot(co,co)*s.cosa*s.cosa;

	float det = b*b - 4.*a*c;
	if (det < 0.)
	{
		//v = 0.7;
		return false;
	}

	det = sqrt(det);
	float t1 = (-b - det) / (2. * a);
	float t2 = (-b + det) / (2. * a);

	// This is a bit messy; there ought to be a more elegant solution.
	float t = t1;
	//	if (t < 0.) t = t2;

	if (t < 0. || t2 > 0. && t2 < t) t = t2;
	//if (t < 0. || t2 > t) t = t2;		// we actualy test for further intersection
	//if (t < 0.) return false;

	if (t < 0.0)
		return false;

	vec3 cp = r.o + t*r.d - s.c;
	float h = dot(cp, s.v);
	if (h < 0.0)
	{
		// can happen if nearest intersection is for the 'negative cone'
		cp = r.o + max(t1, t2)*r.d - s.c;
		h = dot(cp, s.v);
		if (h < 0.0)
			return false;
	}
	if (h < s.h)
		return true;
	
	// check for far intersection if exists
	{
		cp = r.o + max(t1, t2)*r.d - s.c;
		h = dot(cp, s.v);
		if (h > 0.0 && h < s.h)
			return true;
	}

	//v = 0.1;
	return false;
	
	// cap the cone with a plane. NOTE: not needed because we also check the further intersection which seems to be enough
#if 0
	{
		vec3 P0 = s.c + s.h * s.v;
		float d = dot(normalize(s.v), r.d);
		if (d > 0.0)
			return false;

		//float tp = -(dot(r.o, normalize(s.v)) + length(P0)) / d;
		float tp = -(dot(r.o - P0, normalize(s.v))) / d;

		if (t1 > t2)
		{
			float st = t1;
			t1 = t2;
			t2 = st;
		}

		if (tp < t1 || tp > t2)
			return false;
	}
	return true;
#endif
}

#include <shaders/materials/noise/noise3d.glsl>
#include "volumetric_component.glsl"

// all the positions are in world coords
vec3 calculate_lighting_world(LightProperties light, in vec3 pos, in vec3 normal, in vec3 light_pos, in vec3 cam_pos, in float NdotL)
{
	float d = NdotL;
	if (d < 0.0)
		d = 0.0;
	
	vec3 specular = vec3(0.0);
	if (d > 0.0)
		specular = pow(max(0.0, dot(reflect(normalize(light_pos - pos), normalize(normal)), -normalize(cam_pos - pos))), 14.0) * light.diffuse.xyz;

	return vec3(vec3(d) * light.diffuse.xyz + specular);
}

//#include <support/ltc_line.glsl>
#include "ltc_quad.glsl"

#if 0
void getPBRParams(PBRFactors params, out float3 diffuseColor, out float3 specularColor, out float perceptualRoughness, out float alpha)
{
    // Metallic and Roughness material properties are packed together
    // In glTF, these factors can be specified by fixed scalar values
    // or from a metallic-roughness map
    alpha = 0.0;
    perceptualRoughness = 0.0;
    diffuseColor = float3(0.0, 0.0, 0.0);
    specularColor = float3(0.0, 0.0, 0.0);
    float metallic = 0.0;
    float3 f0 = float3(0.04, 0.04, 0.04);

    float4 baseColor = getBaseColor(Input, params);

#ifdef MATERIAL_SPECULARGLOSSINESS
    float4 sgSample = getSpecularGlossinessTexture(Input);
    perceptualRoughness = (1.0 - sgSample.a * params.glossinessFactor); // glossiness to roughness
    f0 = sgSample.rgb * params.specularFactor; // specular

    // f0 = specular
    specularColor = f0;
    float oneMinusSpecularStrength = 1.0 - max(max(f0.r, f0.g), f0.b);
    diffuseColor = baseColor.rgb * oneMinusSpecularStrength;

#ifdef DEBUG_METALLIC
    // do conversion between metallic M-R and S-G metallic
    metallic = solveMetallic(baseColor.rgb, specularColor, oneMinusSpecularStrength);
#endif // ! DEBUG_METALLIC
#endif // ! MATERIAL_SPECULARGLOSSINESS

#ifdef MATERIAL_METALLICROUGHNESS
    // Roughness is stored in the 'g' channel, metallic is stored in the 'b' channel.
    // This layout intentionally reserves the 'r' channel for (optional) occlusion map data
    float4 mrSample = getMetallicRoughnessTexture(Input);
    perceptualRoughness = mrSample.g * params.roughnessFactor;
    metallic = mrSample.b * params.metallicFactor;

    diffuseColor = baseColor.rgb * (float3(1.0, 1.0, 1.0) - f0) * (1.0 - metallic);
    specularColor = lerp(f0, baseColor.rgb, metallic);
#endif // ! MATERIAL_METALLICROUGHNESS

    perceptualRoughness = clamp(perceptualRoughness, 0.0, 1.0);

    alpha = baseColor.a;
}
#endif

vec4 sample_rt(uvec2 pixel_pos, vec2 p, float rt_scaling_factor, float depth, vec3 ref_normal)
{
	float ref_metalness;
	float ref_roughness;
	uint  ref_materialIndex;
	decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, ivec2(pixel_pos), 0).rg, ref_metalness, ref_roughness, ref_materialIndex);
	MaterialPropertiesGPU ref_material = materials.material_properties[ref_materialIndex];
	bool is_raytraced = (ref_material.flags & MaterialFlag_Reflective) != 0;

	if (is_raytraced == false)
		return vec4(0.0);

	const float KERNEL_5[5] = 
	{
		1.0/16.0, 1.0/4.0, 3.0/8.0, 1.0/4.0, 1.0/16.0
	};

	const float KERNEL_7[7] = 
	{
		0.07130343198685299,
		0.13151412084312236,
		0.1898792328888381,
		0.214606428562373,
		0.1898792328888381,
		0.13151412084312236,
		0.07130343198685299
	};

	const float KERNEL_11[11] = 
	{
		0.009300040045324049,
		0.028001560233780885,
		0.06598396774984912,
		0.12170274650962626,
		0.17571363439579307,
		0.19859610213125314,
		0.17571363439579307,
		0.12170274650962626,
		0.06598396774984912,
		0.028001560233780885,
		0.009300040045324049
	};

	p *= rt_scaling_factor;
	uvec2 pixel_pos_scaled = pixel_pos / uvec2(1.0 / rt_scaling_factor);

#if 1
#define R 3
//#define KERNEL KERNEL_5
#define KERNEL KERNEL_7
//#define KERNEL KERNEL_11

	vec4 v = vec4(0.0);
	float total_w = 0.0;

	for(int y = -R; y <= R; y+=1)
	{
		for(int x = -R; x <= R; x+=1)
		{
			ivec2 sampling_pos =  ivec2(pixel_pos) + ivec2(x, y);
			vec2 sampling_pos_scaled =  vec2(pixel_pos_scaled) + vec2(x, y) * rt_scaling_factor + 0.5;

			float w = KERNEL[x + R] * KERNEL[y + R];
			{
				float metalness;
				float roughness;
				uint materialIndex;
				decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, sampling_pos, 0).rg, metalness, roughness, materialIndex);

				if (materialIndex != ref_materialIndex)
					w = 0.0;

				MaterialPropertiesGPU material = materials.material_properties[materialIndex];
				bool is_raytraced = (material.flags & MaterialFlag_Reflective) != 0;
				if (is_raytraced == false)
					w = 0.0;

				uint encoded_normal_material = texelFetch(sNormalMaterial, sampling_pos, 0).r;
				vec3 normal = decode_normal(encoded_normal_material);

				float normal_diff = clamp(dot(ref_normal, normal), 0.0, 1.0);
				w *= normal_diff;

				{
					vec4 pxl = texture(sRaytrace, sampling_pos_scaled / render_lights_params.resolution);
					//vec4 pxl = texelFetch(sRaytrace, sampling_pos_scaled, 0);

					v += pxl * w;
					total_w += w;
				}
			}
		}
	}

	return v / total_w;
	//return vec4(total_w, total_w, total_w, s);

#undef R
#else
	return texelFetch(sRaytrace, ivec2(pixel_pos_scaled), 0);
#endif

}

vec3 get_view_direction(vec2 screen_pos)
{
	vec2 vd_pos = screen_pos - render_lights_params.frustum_shift.xy * render_lights_params.resolution.xy * vec2(0.5, -0.5);
	vec3 view_direction;

	view_direction.x = -basic_params.camera_projection_params.z + basic_params.camera_projection_params.x * vd_pos.x / render_lights_params.resolution.x;
	view_direction.y = -basic_params.camera_projection_params.w + basic_params.camera_projection_params.y * vd_pos.y / render_lights_params.resolution.y;
	view_direction.z = 1.0;

	#ifdef SPIRV_VULKAN
	view_direction.y = -view_direction.y;
	#endif

	return view_direction;
}

void main() {
	ivec2 screen_pos = ivec2(gl_FragCoord.xy);

	vec4  base_color              = texelFetch(sAlbedo, screen_pos, 0);
	uint  encoded_normal_material = texelFetch(sNormalMaterial, screen_pos, 0).r;
	vec3  vNorm                   = decode_normal(encoded_normal_material);
	int   materialId              = decode_material(encoded_normal_material);

	float depth                   = linearizeDepth(texelFetch(sDepth, screen_pos, 0).r);
	vec3  view_direction          = get_view_direction(vec2(screen_pos));
	//vec3  world                   = basic_params.camera_position.xyz + positionFromDepth(vFrustum.xyz, depth);
	vec3  world                   = (basic_params.mModel * vec4(positionFromDepth(view_direction, depth), 1.0)).xyz;

	//outColor.rgb = fract(world * 0.01);
	//return;

	outColor = vec4(0.0);

	MetalnessRoughnessMeterialTags metalness_roughness_material_tags;
	metalness_roughness_material_tags = decode_metalness_roughness_material_tags(texelFetch(sMetalnessRoughnessMaterialTags, screen_pos, 0).rgba);

	float metalness = metalness_roughness_material_tags.metalness;
	float roughness = metalness_roughness_material_tags.roughness;
	uint material   = metalness_roughness_material_tags.material_index;

	uint material_flags = materials.material_properties[material].flags;
	// also check material overrides. only include 4 attributes
	if ((metalness_roughness_material_tags.material_flag_overrides & MaterialFlag_OverrideFlags) != 0)
	{
		material_flags &= ~0xf;
		material_flags |= metalness_roughness_material_tags.material_flag_overrides & 0xf;
	}

	vec3 specularColor;
	vec3 diffuseColor;

	if ((material_flags & MaterialFlag_DisableLighting) != 0)
	{
		outColor = base_color;
		outColor.a = 0.0;
		return;
	}

	// NOTE: because the model of PBR we use, and it allowing for 0...1 value for base color it doesn't play
	// well with our totally mixed up pipe. For RT we do a hacky job and use color and magnitude separately.
	// How this is going to play? No idea

	vec3 base_emissive = texelFetch(sEmissive, screen_pos, 0).rgb * materials.material_properties[material].emissive.rgb;
	bool is_background = (materialId & MATERIAL_ID_MASK_ATTR) == ATTR_BACKGROUND;
	bool is_raytraced  = (material_flags & (MaterialFlag_Reflective)) != 0;
	bool is_particle   = (material_flags & (MaterialFlag_ParticleLighting)) != 0;

	// NOTE: Currently conside color to be premultiplied by diffuse
	if (is_particle == false)
	{
		//base_color.rgb *= materials.material_properties[material].diffuse.rgb;
	}
	else
	{
		outColor       = base_color;
		outColor.a     = 0.0;
		base_color.rgb = materials.material_properties[material].diffuse.rgb;
	}

	if (false)
	if (is_raytraced)
	{
		float color_magnitude = length(base_color.rgb);
		if (color_magnitude > 1.0)
		{
			vec3 emissive_from_base_color = base_color.rgb - base_color.rgb / color_magnitude;
			base_color.rgb -= emissive_from_base_color;
			base_emissive += emissive_from_base_color;
		}
		base_color.rgb = saturate(base_color.rgb);
	}

	{
		vec3 f0 = vec3(0.04, 0.04, 0.04);
		diffuseColor = base_color.rgb * (vec3(1.0, 1.0, 1.0) - f0) * (1.0 - metalness);
		specularColor = mix(f0, base_color.rgb, metalness);
	}

	// Roughness is authored as perceptual roughness; as is convention,
    // convert to material roughness by squaring the perceptual roughness [2].
	float alphaRoughness = roughness * roughness;

	vec3 specularEnvironmentR0 = specularColor.rgb;
    // Anything less than 2% is physically impossible and is instead considered to be shadowing. Compare to "Real-Time-Rendering" 4th editon on page 325.
    float reflectance = max(max(specularColor.r, specularColor.g), specularColor.b);
    vec3 specularEnvironmentR90 = vec3(1.0, 1.0, 1.0) * clamp(reflectance * 25.0, 0.0, 1.0);

	MaterialInfo materialInfo =
    {
        roughness,
        specularEnvironmentR0,
        alphaRoughness,
        diffuseColor,
        specularEnvironmentR90,
        specularColor
    };

	vec3 view = normalize(basic_params.camera_position.xyz - world.xyz);
	vec3 outLightColor = vec3(0.0);

	// iterate through lights
	if (is_particle == false)
	{
		for(int light_idx = 0; light_idx < render_lights_params.lights_num; light_idx++)
		{
			LightProperties light = lights.light_properties[light_idx];
			vec3 light_color = vec3(0.0);

			// remove when we use the proxy object... -- kiero
			// NOTE: also, dont check for non-volumetric lights

			if ((light.type & LightType_Spot) != 0)
			{
				if ((light.type & LightType_Volumetric) != 0)
				{
					Cone cone = Cone(light.cutoff * 1.0, light.range * 1.5, light.position.xyz, light.direction.xyz);
					Ray ray = Ray(basic_params.camera_position.xyz, normalize(vFrustum.xyz));
					float vv = 0.0;
					if (intersect_cone(cone, ray, vv) == false)
						continue;
				}
			}

			vec3 pointToLight = light.position.xyz - world.xyz;
			if ((light.type & LightType_Directional) != 0)
				pointToLight = -light.direction.xyz;

			pointToLight = normalize(pointToLight);
			float NdotL = dot(vNorm, pointToLight);

			// for particles don't include directional component
			if ((materialId & MATERIAL_ID_MASK_ATTR) == ATTR_PARTICLE)
				NdotL = 1.0;

			//vec3 lighting = calculate_lighting_world(light, world, vNorm, light.position.xyz, basic_params.camera_position.xyz, NdotL);
			vec3 lighting = getPointShade(pointToLight, materialInfo, vNorm, view) * light.intensity * light.diffuse.rgb;

			float shadow = 0.0;

			bool calculate_shadows = ((material_flags & MaterialFlag_ShadowReceive) != 0)
			  && ((light.type & LightType_Shadowcasting) != 0);

			if (is_background)
			{
				shadow = 0.0;
				lighting = vec3(1.0);
				calculate_shadows = false;
			}

			if ((light.type & (LightType_Spot | LightType_Directional)) != 0 && calculate_shadows)
			{
				vec4 vShadowCoords = light.mat_shadow_p[0] * light.mat_shadow_mv * vec4(world.xyz, 1.0);

				// shadows
				float in_frustum = 0.0;

				// TODO: Make this configurable. We don't need contact shadows for all the lights as they are expensive
				vec2 penumbraHash = texelFetch(s_BlueNoise, ivec3(screen_pos.xy & ivec2(127), globals.monotonic & 15), 0).rg;

				float light_size        = length(light.dimensions);
				int penumbra_iterations = max(6, min(10, int(light_size * 0.0025)));

				float penumbra = light.roughness_modifier / 1024.0;

				if ((light.type & LightType_Spot) != 0)
					penumbra = PenumbraFromOccluderSearch(
						LightShadowmapSamplers[light.shadowmap_sampler0],
						light_size,
						penumbraHash,
						vShadowCoords,
						//max(0.5, light.roughness_modifier) / 1024.0,
						max(0.25, light_size * 0.2) / 1024.0,	// NOTE: This still needs tweaking. for small amounf of samples and large penumbra there is still banding
						penumbra_iterations,	// 8 is good compromise where the edge is smooth. 3 is fine in some cases but might produce an edge between shadow and penumbra...
						materials.material_properties[material].shadowmap_bias);
				
				// NOTE: About the sample count. For larger lights increasing penumbra computation sample count
				// is actually advised. Might consider separate code path for high quality (less noise) penumbras

				//penumbra = 0.01;
				//if (penumbra < 5.0)
				//	penumbra = 0.0;

				int light_iterations = max(4, min(12, int(penumbra * light.roughness_modifier * 0.25)));

				//shadow = sampleShadowPCF(LightShadowmapCmpSamplers[light.shadowmap_sampler0], vShadowCoords, in_frustum, 8);
				shadow = sampleShadowPCFNoiseOffset(
					LightShadowmapCmpSamplers[light.shadowmap_sampler0],
					vShadowCoords,
					screen_pos,
					in_frustum,
					light_iterations,
					max(0.25, penumbra * light.roughness_modifier) / 1024.0,	// 100 is just for some semi-okish scaling. TODO: check if this works with all scene scales
					materials.material_properties[material].shadowmap_bias);

				//lighting.rgb = vec3(penumbra) + 0.01;
				//shadow = 0.0;
				//shadow = clamp(shadow, 0.0, 1.0);
				//light_color.rgb = vec3(penumbra);
				//shadow = penumbra * 1.0;

				if (NdotL <= 0.0)
					shadow = max(shadow, smoothstep(0.0, -0.05, NdotL));

				//shadow = 0.0;
			}

			// NOTE: This is a kludge for very bright lights until we get temporal/spatial filtering for
			// shadows. This kindof tweaks the falloff for the filtering. It only adjusts the curve, so
			// should not affect fully lit/shadowed surfaces!
			//if (shadow > 0.01)
			//	shadow = pow(shadow, 1.0 / max(1.0, 0.0005 * light.intensity));
			light_color = lighting.rgb * (1.0 - shadow);
			//light_color = vec3(NdotL);
			//light_color = vec3(shadow);

			{
				if ((light.type & LightType_Projector) != 0)
				{
					vec4 vShadowCoords = light.mat_shadow_p[0] * light.mat_shadow_mv * vec4(world.xyz, 1.0);

					vec3 projector_color = sampleProjectorTexture(
						LightProjectorSamplers[light.projector_sampler],
						vShadowCoords).rgb * light.projector_intensity;

						//projector_color = clamp(projector_color, 0.0, 1.0);
					light_color.rgb *= projector_color;
				}
			}

			if ((light.type & (LightType_Spot | LightType_Attenuating)) == (LightType_Spot | LightType_Attenuating))
			{
				float attenuation = light_calculate_spot_attenuation(light, world.xyz);
				light_color.rgb *= attenuation;
				//light_color.rgb = vec3(attenuation);
			}
			else if ((light.type & (LightType_Point | LightType_Attenuating)) == (LightType_Point | LightType_Attenuating))
			{
				float attenuation = light_calculate_point_attenuation(light, world.xyz);
				light_color.rgb *= attenuation;
				//light_color.rgb = vec3(attenuation);
			}

			// TODO: Externalize. Experimental LTC lighting support. Move to external shader and apply then in bulk? -- kiero
			if (light.is_area != 0)
			{
				if (screen_pos.x == 0 && screen_pos.y == 0)
				{
					//debugPrintfEXT("dims:%f, %f, %f, %f\\n", light.dimensions.x, light.dimensions.y, length(light.right), length(light.up));
				}

				if ((materialId & MATERIAL_ID_MASK_ATTR) == ATTR_PARTICLE)
				{
					//light_color.rgb = normalize(light.diffuse.rgb) * light.intensity;
				}
				else
				{
					LTCRect rect;
					InitRect(light, rect);

					vec3 points[4];
					InitRectPoints(rect, points);

					float ltc_roughness = roughness;
					float ltc_intensity = light.intensity;

					vec3 dcol = light.diffuse.rgb;
					vec3 scol = dcol;

					vec3 col = vec3(0.0);
					vec3 pos = world.xyz;

					vec3 N = vNorm;
					vec3 V = normalize(basic_params.camera_position.xyz - world.xyz);

					float ndotv = clamp(dot(N, V), 0.05, 1.0);

					vec2 uv = vec2(ltc_roughness, sqrt(1.0 - ndotv));
					uv = uv * LTC_LUT_SCALE + LTC_LUT_BIAS;

					vec4 t1 = texture(s_LTC1, uv);
					vec4 t2 = texture(s_LTC2, uv);
			#if 0 // line handling
					ltc_Minv = mat3(
						vec3(t1.x, 0, t1.y),
						vec3(  0,  1,    0),
						vec3(t1.z, 0, t1.w)
					);

					vec3 spec = LTC_Evaluate(N, V, pos);
					// BRDF shadowing and Fresnel
					spec *= scol*t2.x + (vec3(1.0) - scol)*t2.y;

					ltc_Minv = mat3(1);
					vec3 diff = LTC_Evaluate(N, V, pos);
					col  = lcol*(spec + dcol*diff);
					col /= 2.0*PI;
			#else // quad handling
					mat3 Minv = mat3(
						vec3(t1.x, 0, t1.y),
						vec3(  0,  1,    0),
						vec3(t1.z, 0, t1.w)
					);

					vec3 spec = LTC_Evaluate(N, V, pos, Minv, points, twoSided);
					// BRDF shadowing and Fresnel
					spec *= scol * t2.x + (1.0 - scol) * t2.y;

					//vec3 diff = LTC_Evaluate_DiffuseOnly(N, pos, mat3(1), points, twoSided);
					//col  = ltc_intensity * (dcol * diff);
					vec3 diff = LTC_Evaluate(N, V, pos, mat3(1), points, twoSided);
					col  = ltc_intensity * (spec * specularColor.rgb + dcol * diff * diffuseColor.rgb);
			#endif
		
					light_color.rgb = col;
				}

				if ((light.type & (LightType_Attenuating)) == (LightType_Attenuating))
				{
					float attenuation = light_calculate_area_attenuation(light, world.xyz);
					light_color.rgb *= attenuation;
				}
			}

			if ((light.type & LightType_Volumetric) != 0)
			{
				// experimental lightshafts (volumetric)
				float attenuation = 0.0;
				vec4 projector_color = vec4(0.0);
				float volume_attenuation = volumetric_sample_shadow_spot(light, vFrustum.xyz, depth, attenuation, projector_color);
				//light_color.a = projector_color.r * volume_attenuation;
				outColor.a += light.diffuse.a * volume_attenuation;
				//light_color.a = pow(volume_attenuation * light.diffuse.a, 2.0);
			}

			outLightColor.rgb += light_color;
		}
	
		// raytyraced surfaces are hacky as fuck. they already had reflections traced, so if we are in shadow, we still contribute
		if (true)
		if (is_raytraced)
		{
			vec2 p = vTexcoord0 * render_lights_params.raytrace_scaling_factor;

			#if 1
			vec4 raytraceColor = texture(sRaytrace, p).rgba;
			#else
			vec4 raytraceColor = sample_rt(screen_pos, p, render_lights_params.raytrace_scaling_factor, 0.0, vNorm);
			#endif

			// idea is to fake roughness by using is as a blend. more rough->less rt
			raytraceColor.rgb *= render_lights_params.raytrace_strength;
			outLightColor.rgb = outLightColor.rgb + specularColor.rgb * raytraceColor.rgb * (1.0 - roughness);

			outLightColor.rgb = mix(outLightColor.rgb, raytraceColor.rgb, raytraceColor.a);
			//outLightColor.rgb = vec3(raytraceColor.a);
			//light_color.rgb = light_color.rgb  + specularColor.rgb * raytraceColor.rgb;
			//light_color.rgb = specularColor.rgb * raytraceColor.rgb;
		}
	}

	outLightColor.rgb += base_emissive.rgb;

	// IBL contribution
	if (is_background == false)
	{
		vec4 ibl_occlusion = texelFetch(sIBL, screen_pos, 0).rgba;
		vec3 ibl = ibl_occlusion.rgb;
		float occlusion = ibl_occlusion.a;
		outLightColor.rgb += ibl;

		//outLightColor.rgb = vec3(1.0 - (1.0 - exp(1.0 - occlusion)) * composite_setup.occlusion_base_strength);
		outLightColor.rgb *= vec3(1.0 - (pow(occlusion, 0.25) * composite_setup.occlusion_base_strength));
		//outLightColor.rgb = vec3(occlusion);

		// DEBUG: only IBL component
		//outLightColor.rgb = vec3(0.0);
		//outColor.rgb = ibl;
	}

	outColor.rgb += outLightColor.rgb;
	if (is_background == false)
	{
		// This is 'artsy' as fuck
		outColor.rgb *= materials.material_properties[material].emissive.rgb * materials.material_properties[material].emissive_factor * base_color.rgb + 1.0;
		outColor.rgb += materials.material_properties[material].emissive.rgb * materials.material_properties[material].emissive_factor * base_color.rgb;
	}

	vec3 noisyColor = outColor.rgb - outLightColor.rgb;
	//outColor.rgb = clamp(outLightColor + 3.0, vec3(0.0), vec3(1.0));

	//outColor = vec4(0.5, 0.5, 0.5, 0.0);//outColor * colorDiffuse;
	//outColor = vec4(1.0);
	//outColor = vec4(texture(sDepth, vTexcoord0).r);
	//outColor = vec4(world.x);
	//outColor.rgb = vec3(roughness);
	//outColor.rgb = metalness.rrr;
}
