//This shader provides an automatic adaptive bloom effect based on screen brightness to scale the bloom effect
//Shader code by Mark Blosser and Uzi Idiot
//Camera exposure code by Krzysztof Narkowicz
//email: mjblosser@gmail.com
//website: www.mjblosser.com
//released under creative commons license by attribution: http://creativecommons.org/licenses/by-sa/3.0/
string Description = "Camera Post Processing Shader by Uzi Idiot";
string Thumbnail = "Blur.png";

float2 ViewSize : ViewSize;
float time : Time;

float BloomWeights[5] = {1, 1, 1, 1, 1};
float AvgWeights[9] = {1, 1, 1, 1, 1, 1, 1, 1, 1};
    
//-----TWEAKABLES--------------------------------------------------------------------------------------//

float exposure 
<
	string UIWidget = "slider";
	float UIMax = 10.0;
	float UIMin = -10.0;
	float UIStep = 1.0;
> = 0.000000;

float2 aperture  = { 2.100000, 8.300000 };
float2 iso  = { 100.000000, 100.000000 };
float2 shutterSpeed  = { 100.000000, 100.000000 };

float2 sensitivity  = { 1.000000, 1.000000 };

float2 BloomThreshold  = { 0.000000, 1.000000 };

float PostBloomBoost 
<
	string UIWidget = "slider";
	float UIMax = 4.0;
	float UIMin = 0.0;
	float UIStep = 0.1;
> = 0.250000;

float TonemapMode <
    string UIWidget = "slider";
    string UIName = "Exposure";
    float UIMin = 0.0;
    float UIMax = 6.0;
    float UIStep = 1;
> = 5.000000;

float Whitepoint <
    string UIWidget = "slider";
    float UIMin = 0.0;
    float UIMax = 5.0;
    float UIStep = 0.1;
> = 9.000000;

float width 
<
	string UIWidget = "slider";
	float UIMax = 10.0;
	float UIMin = 1;
	float UIStep = 0.01;
> = 2.500000;

float gamma  = 2.200000;

//9 sample gauss filter, declare in pixel offsets convert to texel offsets in PS
float4 GaussFilter[9] =
{
    { -1,  -1, 0,  0.0625 },
    { -1,   1, 0,  0.0625 },
    {  1,  -1, 0,  0.0625 },
    {  1,   1, 0,  0.0625 },
    { -1,   0, 0,  0.125  },
    {  1,   0, 0,  0.125  },
    {  0,  -1, 0,  0.125 },
    {  0,   1, 0,  0.125 },
    {  0,   0, 0,  0.25 },
};

//9 sample gauss filter, declare in pixel offsets convert to texel offsets in PS
float2 GaussFilter2[5] =
{
    { -2, 0.125 },
    { -1, 0.25 },
    { -0, 0.5 },
    { 1, 0.25 },
    { 2, 0.125 },
};

//9 sample gauss filter, declare in pixel offsets convert to texel offsets in PS
float2 GaussFilter3[7] =
{
    { -3, 0.125 },
    { -2, 0.125 },
    { -1, 0.25 },
    { -0, 0.5 },
    {  1, 0.25 },
    {  2, 0.125 },
    {  3, 0.125 },
};



//---------------RENDER TARGET TEXTURES-------------------------------------------------------------------//

//starting scene image
texture frame : RENDERCOLORTARGET
< 
	string ResourceName = "";
	float2 ViewportRatio = {1.0,1.0 };
	
>;

sampler2D frameSamp = sampler_state {
    Texture = < frame >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//2x2 average luminosity texture using point sampling
texture AvgLum2x2Img : RENDERCOLORTARGET 
< 
	string ResourceName = ""; 
	//float2 ViewportRatio = { 0.5, 0.5 };
	int width = 9;
	int height = 9;
>;

sampler2D AvgLum2x2ImgSamp = sampler_state {
    Texture = < AvgLum2x2Img >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//Average scene luminosity stored in 1x1 texture 
texture AvgLumFinal : RENDERCOLORTARGET 
< 
	string ResourceName = ""; 
	//float2 ViewportRatio = { 0.5, 0.5 };
	int width = 1;
	int height = 1;
>;

sampler2D AvgLumFinalSamp = sampler_state {
    Texture = < AvgLumFinal >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//reduce image to 1/8 size for brightpass
texture BrightpassImg : RENDERCOLORTARGET
< 
	string ResourceName = "";
	//float2 ViewportRatio = {0.125,0.125 };
	int width = 256;
	int height = 192;
	
>;

sampler2D BrightpassImgSamp = sampler_state {
    Texture = < BrightpassImg >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//blur texture 1
texture Blur1Img : RENDERCOLORTARGET
< 
	string ResourceName = "";
	//float2 ViewportRatio = {0.125,0.125 };
	int width = 128;
	int height = 96;
	
>;

sampler2D Blur1ImgSamp = sampler_state {
    Texture = < Blur1Img >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//blur texture 2
texture Blur2Img : RENDERCOLORTARGET
< 
	string ResourceName = "";
	//float2 ViewportRatio = {0.125,0.125 };
	int width = 64;
	int height = 48;
	
>;

sampler2D Blur2ImgSamp = sampler_state {
    Texture = < Blur2Img >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//blur texture 2
texture Blur3Img : RENDERCOLORTARGET
< 
	string ResourceName = "";
	//float2 ViewportRatio = {0.125,0.125 };
	int width = 32;
	int height = 24;
	
>;

sampler2D Blur3ImgSamp = sampler_state {
    Texture = < Blur3Img >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//blur texture 2
texture Blur4Img : RENDERCOLORTARGET
< 
	string ResourceName = "";
	//float2 ViewportRatio = {0.125,0.125 };
	int width = 16;
	int height = 12;
	
>;

sampler2D Blur4ImgSamp = sampler_state {
    Texture = < Blur4Img >;
    MinFilter = Linear; MagFilter = Linear; MipFilter = Linear;
    AddressU = Clamp; AddressV = Clamp;
};

//-------------STRUCTS-------------------------------------------------------------

struct input 
{
	float4 pos : POSITION;
	float2 uv : TEXCOORD0;
};
 
struct output {

	float4 pos: POSITION;
	float2 uv: TEXCOORD0;

};

//-----------VERTEX SHADER------------------------------------------------------------//
output VS( input IN ) 
{
	output OUT;

	//quad needs to be shifted by half a pixel.
    //Go here for more info: http://www.sjbrown.co.uk/?article=directx_texels
    
	float4 oPos = float4( IN.pos.xy + float2( -1.0f/ViewSize.x, 1.0f/ViewSize.y ),0.0,1.0 );
	OUT.pos = oPos;

	float2 uv = (IN.pos.xy + 1.0) / 2.0;
	uv.y = 1 - uv.y; 
	OUT.uv = uv;
	
	return OUT;	
}

//---------PIXEL SHADERS--------------------------------------------------------------//

//takes original frame image and outputs to 2x2
float4 PSReduce( output IN, uniform sampler2D srcTex ) : COLOR
{
    float4 color = 0;    
    color= tex2D( srcTex, IN.uv );    
    return color;    
}

//-----------------computes average luminosity for scene-----------------------------
float4 PSGlareAmount( output IN, uniform sampler2D srcTex ) : COLOR
{
    float4 GlareAmount = 0;
    
    //sample texture 4 times with offset texture coordinates
    GlareAmount+= tex2D( srcTex, IN.uv + float2(-0.0, -0.0) )*AvgWeights[0];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(-0.0, 0.66) )*AvgWeights[1];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(0.0, -0.66) )*AvgWeights[2];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(-0.66, -0.0) )*AvgWeights[3];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(0.66, -0.0) )*AvgWeights[4];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(-0.66, 0.66) )*AvgWeights[5];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(-0.66, -0.66) )*AvgWeights[6];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(0.66, 0.66) )*AvgWeights[7];
    GlareAmount+= tex2D( srcTex, IN.uv + float2(0.66, -0.66) )*AvgWeights[8];
    GlareAmount/= AvgWeights[0]+AvgWeights[1]+AvgWeights[2]+AvgWeights[3]+AvgWeights[4]+AvgWeights[5]+AvgWeights[6]+AvgWeights[7]+AvgWeights[8];
    
    //average these samples
    //float3 AvgColor = (color1.xyz+color2.xyz+color3.xyz+color4.xyz+color5.xyz+color6.xyz+color7.xyz+color8.xyz+color9.xyz)/9;
    
    //AvgColor = (color1+color3+color4)/3;
    
    //convert to luminance
    //AvgColor = lerp(dot(float3(0.3,0.59,0.11), AvgColor),AvgColor,WhiteBalance);
    
    GlareAmount = pow(dot(float3(0.3,0.59,0.11), GlareAmount),sensitivity.x);
    
    //GlareAmount.xyz = AvgColor;
    
    //interpolation value to blend with previous frames
    GlareAmount.w = 0.03*sensitivity.y;
       
    return GlareAmount;    
}


//-------------brightpass pixel shader, removes low luminance pixels--------------------------
float4 PSBrightpass( output IN, uniform sampler2D srcTex, uniform sampler2D srcTex2  ) : COLOR
{
    float4 screen = tex2D(srcTex, IN.uv);  //original screen texture;
    float4 glaretex = tex2D(srcTex2, IN.uv);  //glareamount from 1x1 in previous pass;
    
    //remove low luminance pixels, keeping only brightest
    float3 Brightest = saturate(screen.xyz - BloomThreshold.x)/(1-BloomThreshold.x);
    Brightest = pow(Brightest,BloomThreshold.y);
    
    //apply glare adaption
    //float3 adapt = Brightest.xyz * (1-glaretex.xyz);
    
    float4 color = float4(Brightest.xyz, 1);
    
    return color;    
}

float4 PSBlur( output IN, uniform sampler2D srcTex ) : COLOR
{
    float4 color = float4(0,0,0,0);
    
    //inverse view for correct pixel to texel mapping
    float2 ViewInv = float2( 1/ViewSize.x,1/ViewSize.y);   
    /*
    //sample and output average colors using gauss filter samples
    for(int i=0;i<9;i++)
    
    { 
    float4 col = GaussFilter[i].w * tex2D(srcTex,IN.uv + float2(GaussFilter[i].x * ViewInv.x*width, GaussFilter[i].y *ViewInv.y*width));  
    color+=col;
    } 
    
    
    //sample and output average colors using gauss filter samples
    for(int i=0;i<5;i++)
    
    {
    float4 col = tex2D(srcTex,IN.uv + float2(GaussFilter2[i].x * ViewInv.x, GaussFilter2[i].x *ViewInv.y));  
    col += tex2D(srcTex,IN.uv + float2(-GaussFilter2[i].x * ViewInv.x, GaussFilter2[i].x *ViewInv.y));  
    color+=GaussFilter2[i].y * col/2;
    }
    */
    
    
    //sample and output average colors using gauss filter samples
    for(int i=0;i<7;i++)
    
    {
    float4 col = tex2D(srcTex,IN.uv + float2(GaussFilter3[i].x * ViewInv.x, GaussFilter3[i].x *ViewInv.y));  
    col += tex2D(srcTex,IN.uv + float2(-GaussFilter3[i].x * ViewInv.x, GaussFilter3[i].x *ViewInv.y));  
    color+=GaussFilter3[i].y * col/3;
    }
    
    return color;
}

float4 BloomCombine( output IN, uniform sampler2D srcTex, uniform sampler2D srcTex1, uniform sampler2D srcTex2, uniform sampler2D srcTex3, uniform sampler2D srcTex4  ) : COLOR
{
    float4 screen = tex2D(srcTex, IN.uv)*BloomWeights[0]; 
    screen += tex2D(srcTex1, IN.uv)*BloomWeights[1]; 
    screen += tex2D(srcTex2, IN.uv)*BloomWeights[2]; 
    screen += tex2D(srcTex3, IN.uv)*BloomWeights[3]; 
    screen += tex2D(srcTex4, IN.uv)*BloomWeights[4]; 
    screen /= BloomWeights[0]+BloomWeights[1]+BloomWeights[2]+BloomWeights[3]+BloomWeights[4]; 
    
    
    return screen;    
}

/*
* Get an exposure using the Saturation-based Speed method.
*/
float getSaturationBasedExposure(float aperture,
                                 float shutterSpeed,
                                 float iso)
{
    float l_max = (7800.0f / 65.0f) * (aperture*aperture) / (iso * shutterSpeed);
    return 1.0f / l_max;
}
 
/*
* Get an exposure using the Standard Output Sensitivity method.
* Accepts an additional parameter of the target middle grey.
*/
float getStandardOutputBasedExposure(float aperture,
                                     float shutterSpeed,
                                     float iso,
                                     float middleGrey = 0.18f)
{
    float l_avg = (1000.0f / 65.0f) * (aperture*aperture) / (iso * shutterSpeed);
    return middleGrey / l_avg;
}


float3 Reinhard(float3 x)
{
   return x/(1+x);
}

float3 ReinhardLuminance(float3 x)
{
    float Luminance = dot(x, float3(0.2126, 0.7152, 0.0722));;
	float3 Tone = Reinhard(Luminance);
	return ((Tone / Luminance)*x.rgb);
}

float3 ReinhardLumWhite(float3 x)
{
	float luma = dot(x, float3(0.2126, 0.7152, 0.0722));
	float toneMappedLuma = luma * (1. + luma / ((Whitepoint)*(Whitepoint))) / (1. + luma);
	x *= toneMappedLuma / luma;
	//x = pow(x, vec3(1. / gamma));
	return x;
}


float3 Uncharted2Tonemap(float3 x)
{
   float A = 0.15;
   float B = 0.50;
   float C = 0.10;
   float D = 0.20;
   float E = 0.02;
   float F = 0.30;
   return ((x*(A*x+C*B)+D*E)/(x*(A*x+B)+D*F))-E/F;
}

float3 FilmicTonemap(float3 x)
{
   x = max(0,x-0.004);
   float3 retColor = (x*(6.2*x+.5))/(x*(6.2*x+1.7)+0.06);
   return pow(retColor,2.2);
}

float3 ACESFilm( float3 x )
{
    float a = 2.51f;
    float b = 0.03f;
    float c = 2.43f;
    float d = 0.59f;
    float e = 0.14f;
    return saturate((x*(a*x+b))/(x*(c*x+d)+e));
}


//combine adaptive bloom texture with original scene image
float4 PSPresent( output IN, uniform sampler2D srcTex, uniform sampler2D srcTex2, uniform sampler2D srcTex3 ) : COLOR
{
    float4 color = float4(0,0,0,0);
    
    //sample screen texture and bloom texture
    float4 ScreenMap=pow(tex2D(srcTex, IN.uv),gamma);
    float4 BloomMap=pow(tex2D(srcTex2, IN.uv),gamma);
    float4 AmtMap=pow(tex2D(srcTex3, IN.uv),1);
    /*
    ScreenMap = (ScreenMap)/(1.01-ScreenMap);
    BloomMap = (BloomMap)/(1.01-BloomMap);
    AmtMap = (AmtMap)/(1.01-AmtMap);
    */
    
    aperture = lerp(aperture.x,aperture.y,AmtMap);
    iso = lerp(iso.x,iso.y,AmtMap);
    shutterSpeed = lerp(shutterSpeed.x,shutterSpeed.y,AmtMap);
    
    shutterSpeed = 1/shutterSpeed;
    
    
    //technique 1: just add results
    
    float Luminance = dot(ScreenMap.xyz, float3(0.3, 0.59, 0.11));
    float3 mod = lerp(PostBloomBoost,0,AmtMap.x);
    float3 final = ScreenMap + PostBloomBoost*BloomMap;
    
    
    final *= 1000;
    
    //final *= getSaturationBasedExposure(aperture,shutterSpeed,iso);
    final *= getStandardOutputBasedExposure(aperture,shutterSpeed,iso);
    
    final *= pow(2,exposure);
    /*
    final = (2*final)/(2*final+1);
    final = pow(final,1.5);
    final *= 1.25;
    final -= 0.125;
    */
    
    //tonemapping - DONT USE MORE THAN ONE AT A TIME
    
    final *= 2;
    float2 vig = IN.uv-.5;
    //vig *= 1/2;
    final *= 1-dot(vig,vig);
    
     if (TonemapMode==0)
    {
    //final = ((2*final)/(Whitepoint)); //no tonemap, lots of clipping
    final /= 2; //no tonemap, lots of clipping
    }
    
    else if (TonemapMode==1)
    {
    final = Reinhard(final); //reinhard tonemap, no clipping but washed out
    }
    
    else if (TonemapMode==2)
    {
    final = ReinhardLuminance(final); //reinhard tonemap on luminance only, very saturated
    }
    
    else if (TonemapMode==3)
    {
    final = ReinhardLumWhite(final); //reinhard tonemap on luminance, but with whitepoint
    }
    
    else if (TonemapMode==4)
    {
    float ExposureBias = 2.0f;
    float3 curr = Uncharted2Tonemap(ExposureBias*final);
    float3 whiteScale = 1.0f/Uncharted2Tonemap(Whitepoint*2);
    final = curr*whiteScale; //Uncharted 2 filmic tonemap, less washed out
    }
	else if (TonemapMode==5)
    {
    final =FilmicTonemap(final/2)/FilmicTonemap(Whitepoint/2); //Filmic tonemap by Jim Hejl and Richard Burgess-Dawson, higher contrast
    }
    
	else if (TonemapMode==6)
    {
    final = ACESFilm(final/2)/ACESFilm(Whitepoint/2);///ACESFilm(Whitepoint*2); //ACES filmic tonemap, high contrast
    }
	
	//-------------prevent time value from becoming too large--------------
	//float framespersec = 25;
	float looptime = 1000; //looptime in seconds			
	float loopcounter  = floor(time/looptime); //increments by one every 50 seconds (or whatever "looptime" is)
	float offset ;  //initialize offset value used below	
	offset = looptime*loopcounter; //offset time value -increments every looptime
	//float speed =(time*framespersec) - (offset*framespersec) ;	 
	float speed =(time) - (offset) ;
	//------------------------------------------------------------------------
	
    color = float4(final,1);
	
	// sample the source
	float4 cTextureScreen = color;
	float4 combine = color;
	float FilmgrainAmount = iso/6400;
	//cTextureScreen = pow(cTextureScreen,Gamma);

	
	float3 x = IN.uv.x * IN.uv.y *(float3(speed,speed*2,speed*3)*500);
	x = fmod(x, 13) * fmod(x, 123);	
	float3 dx = fmod(x, 0.01);
	
    float3 cResult = cTextureScreen;

	cResult.rgb = combine.rgb * (pow(combine,1/5) + dx.xxx * 100);
	
	// interpolate between source and result by intensity
	color = float4(lerp(combine, cResult, FilmgrainAmount),1);
	
    
	
    
    return pow(color,1/gamma);
    
    
}

//-------TECHNIQUE------------------------------------------------------------------
  
technique AdaptiveBloom
<
	//specify where we want the original image to be put
	string RenderColorTarget = "frame";
>
{
	
	//1. first reduce to 2x2 and save in AvgLum2x2Img
	pass Reduce2x2
	<
		string RenderColorTarget = "AvgLum2x2Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( frameSamp );
		
	}
	
	//2. reduce to 1x1 and save in AvgLumFinalImg, using alpha blending to blend with previous frames
	pass Reduce1x1
	<
		string RenderColorTarget = "AvgLumFinal";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSGlareAmount( AvgLum2x2ImgSamp );
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
	}
	
	//3. remove low luminance pixels keeping only brightest for blurring in next pass
	pass Brightpass
	<
		string RenderColorTarget = "BrightpassImg";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBrightpass( frameSamp, AvgLumFinalSamp );
		
	}
	
	
	
	
	pass Blur1
	<
		string RenderColorTarget = "BrightpassImg";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( BrightpassImgSamp );
		
	}
	
	//4. blur texture and save in Blur1Img
	pass Blur1
	<
		string RenderColorTarget = "Blur1Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( BrightpassImgSamp );
		
	}
	
	//5. repeat blur texture and save in Blur2Img
	pass Blur2
	<
		string RenderColorTarget = "Blur2Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur1ImgSamp );
		
	}
	
	//6. repeat blur again
	pass Blur3
	<
		string RenderColorTarget = "Blur3Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur2ImgSamp );
		
	}
	
	pass Blur4
	<
		string RenderColorTarget = "Blur4Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur3ImgSamp );
		
	}
	
	
	
	
	pass Blur1
	<
		string RenderColorTarget = "BrightpassImg";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( BrightpassImgSamp );
		
	}
	
	pass Blur2
	<
		string RenderColorTarget = "Blur1Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur1ImgSamp );
		
	}
	
	pass Blur3
	<
		string RenderColorTarget = "Blur2Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur2ImgSamp );
		
	}
	
	pass Blur4
	<
		string RenderColorTarget = "Blur3Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur3ImgSamp );
		
	}
	
	pass Blur5
	<
		string RenderColorTarget = "Blur4Img";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( Blur4ImgSamp );
		
	}
	
	
	
	pass BloomCombine
	<
		string RenderColorTarget = "BrightpassImg";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 BloomCombine( BrightpassImgSamp, Blur1ImgSamp, Blur2ImgSamp, Blur3ImgSamp, Blur4ImgSamp );
		
	}
	
	
	
	
	pass Blur1
	<
		string RenderColorTarget = "BrightpassImg";
	>
	{
		ZEnable = False;
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_0 PSBlur( BrightpassImgSamp );
	}
	
	
	
	//send the combined image to the screen
	pass Present
	<
		string RenderColorTarget = "";
	>
	{
		VertexShader = compile vs_1_1 VS();
		PixelShader = compile ps_2_b PSPresent( frameSamp, BrightpassImgSamp, AvgLumFinalSamp );
	}
}

