#version 460

in vec2 vTexcoord0;
uniform sampler2D s_texture0;		// color

layout(location = 0) out vec4 outColor;

struct ColorMixerParams
{
	vec2  resolution;
	vec2  inv_resolution;
	vec4  r_input;
	vec4  g_input;
	vec4  b_input;
	float hue_shift;
};

layout (std140, row_major) uniform ColorMixerParamsBuffer {
	ColorMixerParams params;
};

// All components are in the range [01], including hue.
vec3 rgb2hsv(vec3 c)
{
	vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
	vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
	vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));

	float d = q.x - min(q.w, q.y);
	float e = 1.0e-10;
	return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}
 

// All components are in the range [01], including hue.
vec3 hsv2rgb(vec3 c)
{
	vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
	vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
	return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

float luminance(vec3 x)
{
	return max(0.0, dot(x, vec3(0.2127, 0.7152, 0.0722)));
}

void main()
{
	vec4 base_color = max(vec4(0.0), texelFetch(s_texture0, ivec2(vTexcoord0 * params.resolution), 0));

	vec3 color;
	color.r = dot(base_color.rgb, params.r_input.rgb);
	color.g = dot(base_color.rgb, params.g_input.rgb);
	color.b = dot(base_color.rgb, params.b_input.rgb);

	if (params.hue_shift != 0.0)	// don't do double conversion if not required. should be better for HDR signals where we do the naive tonemap
	{
		float l = luminance(color);
		if (l > 0.0)
		{
			vec3 color_01 = color / (1.0 + l);
			vec3 hsv = rgb2hsv(color_01);

			hsv.x += params.hue_shift;
			if (hsv.x < 0.0)
				hsv.x += 1.0;
			if (hsv.x > 1.0)
				hsv.x -= 1.0;

			color_01 = hsv2rgb(hsv);
			color = color_01 / max(0.0001, (1.0 - luminance(color_01)));
		}
	}

	base_color.rgb = color;
	outColor = base_color;
}

