#include "pass.h"
#include "SDL2/SDL_log.h"
#include "config.h"
#include "demo.h"
#include "shader.h"

// A constant vertex shader, which uses gl_VertexID to output
// a viewport-filling quad. No buffers or Input Assembly needed.
static const char *vertex_shader_src =
    "out vec2 FragCoord;\n"
    "void main() {\n"
    "    vec2 c = vec2(-1, 1);\n"
    "    vec4 coords[4] = vec4[4](c.xxyy, c.yxyy, c.xyyy, c.yyyy);\n"
    "    FragCoord = coords[gl_VertexID].xy;\n"
    "    gl_Position = coords[gl_VertexID];\n"
    "}\n";

pass_renderer_t *pass_renderer_init(void) {
    pass_renderer_t *renderer = calloc(1, sizeof(pass_renderer_t));
    // VAO is a must in GL 3.3 core. Don't think about it too much.
    glGenVertexArrays(1, &renderer->vao);
    glBindVertexArray(renderer->vao);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindVertexArray(0);

    renderer->vertex_shader = compile_shader(
        vertex_shader_src, strlen(vertex_shader_src), "vert", NULL, 0);

    return renderer;
}

void pass_renderer_deinit(pass_renderer_t *renderer) {
    if (renderer) {
        if (renderer->vertex_shader) {
            shader_deinit(renderer->vertex_shader);
        }
        if (renderer->vao) {
            glDeleteVertexArrays(1, &renderer->vao);
        }
        free(renderer);
    }
}

// This function returns a corresponding rocket track name for an uniform.
// Argument `c` is a "component suffix" such as x, y, z or w.
//
// Example: rocket_track_suffix(ufm, 'x') -> "Cam:Pos.x"
static const char *rocket_track_name(const uniform_t *ufm, char c) {
    static char trackname[UFM_NAME_MAX];

    // Adjust for r_ -prefix
    const char *name = ufm->name + 2;
    size_t name_len = ufm->name_len - 2;

    memcpy(trackname, name, name_len + 1);

    // Replace first dot with colon(tab) when possible
    char *instance = memchr(trackname, '.', name_len);
    if (instance) {
        *instance = ':';
    }

    // Add component suffix when requested
    if (c) {
        trackname[name_len] = '.';
        trackname[name_len + 1] = c;
        trackname[name_len + 2] = 0;
    }

    return trackname;
}

// This iterates all uniforms in program, and calls the appropriate
// rocket functions and glUniform functions to glue them together.
// In addition to glUniform, it also supports block uniform buffers.
// This is unnecessarily complex and might get reduced to support
// only block uniforms in a later release.
void set_rocket_uniforms(const program_t *program, struct sync_device *rocket,
                         double rocket_row) {
    // Iterate every active uniform in program
    for (size_t i = 0; i < program->uniform_count; i++) {
        uniform_t *ufm = program->uniforms + i;
        // This tag is used to mark uniform's base type (null, float or int)
        static enum { N = 0, F, I } tag = 0;
        // Staging buffer to be uploaded to uniform memory in OpenGL
        static union {
            GLfloat f[4];
            GLint i;
        } staging;
        // Number of elements (e.g. vec3 => 3)
        GLsizeiptr size = 0;

        // Check for r_ -prefix
        if (ufm->name_len < 3 || ufm->name[0] != 'r' || ufm->name[1] != '_') {
            continue;
        }

        // Fill the staging buffer and set tag and size
        switch (ufm->type) {
        case GL_FLOAT_VEC4:
            staging.f[3] = GET_VALUE(rocket_track_name(ufm, 'w'));
            size = 4;
            // fall through
        case GL_FLOAT_VEC3:
            staging.f[2] = GET_VALUE(rocket_track_name(ufm, 'z'));
            size = size ? size : 3;
            // fall through
        case GL_FLOAT_VEC2:
            staging.f[1] = GET_VALUE(rocket_track_name(ufm, 'y'));
            staging.f[0] = GET_VALUE(rocket_track_name(ufm, 'x'));
            size = size ? size : 2;
            tag = F;
            break;
        case GL_FLOAT:
            staging.f[0] = GET_VALUE(rocket_track_name(ufm, 0));
            size = 1;
            tag = F;
            break;
        case GL_INT:
        case GL_SAMPLER_2D:
            staging.i = (GLint)GET_VALUE(rocket_track_name(ufm, 0));
            size = 1;
            tag = I;
            break;
        default:
            SDL_Log("Unsupported shader uniform type: %d.\n", ufm->type);
            SDL_Log("Go add support for it in " __FILE__
                    " (static void set_non_block_rocket_uniform())\n");
        }

        // Dispatch OpenGL calls to upload the staging buffer
        if (ufm->block_index == -1) {
            // Non-block uniform
            GLint location = glGetUniformLocation(program->handle, ufm->name);
            if (tag == F && size == 1) {
                glUniform1fv(location, 1, (GLfloat *)&staging);
            } else if (tag == F && size == 2) {
                glUniform2fv(location, 1, (GLfloat *)&staging);
            } else if (tag == F && size == 3) {
                glUniform3fv(location, 1, (GLfloat *)&staging);
            } else if (tag == F && size == 4) {
                glUniform4fv(location, 1, (GLfloat *)&staging);
            } else if (tag == I && size == 1) {
                glUniform1iv(location, 1, (GLint *)&staging);
            } else {
                SDL_Log("Oops, check" __FILE__ " line %d\n", __LINE__);
            }
        } else {
            // Block uniform
            GLint index = ufm->block_index;
            GLuint buffer = program->block_buffers[index];
            glBindBuffer(GL_UNIFORM_BUFFER, buffer);
            if (tag == F) {
                size *= sizeof(GLfloat);
            } else if (tag == I) {
                size *= sizeof(GLint);
            } else {
                SDL_Log("Oops, check" __FILE__ " line %d\n", __LINE__);
            }
            glBufferSubData(GL_UNIFORM_BUFFER, ufm->offset, size, &staging);
            glBindBuffer(GL_UNIFORM_BUFFER, 0);

            glUniformBlockBinding(program->handle, index, index);
            glBindBufferBase(GL_UNIFORM_BUFFER, index, buffer);
        }
    }
}

// Uses a shader program, rocket, input textures etc. to draw a shader
// to an output (`draw_fb`).
void pass_render(const pass_render_parameters_t *p) {
    glBindFramebuffer(GL_DRAW_FRAMEBUFFER,
                      p->draw_fb ? p->draw_fb->framebuffer : 0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glDepthMask(GL_FALSE);

    glUseProgram(p->program->handle);

    if (p->viewport) {
        glViewport(p->viewport->x0, p->viewport->y0, p->viewport->x1,
                   p->viewport->y1);
        glUniform2f(glGetUniformLocation(p->program->handle, "u_Resolution"),
                    p->viewport->x1, p->viewport->y1);
    } else if (p->draw_fb) {
        glViewport(0, 0, p->draw_fb->width, p->draw_fb->height);
        glUniform2f(glGetUniformLocation(p->program->handle, "u_Resolution"),
                    p->draw_fb->width, p->draw_fb->height);
    }

    if (p->rocket) {
        set_rocket_uniforms(p->program, p->rocket, p->rocket_row);
        glUniform1f(glGetUniformLocation(p->program->handle, "u_RocketRow"),
                    p->rocket_row);
    }
    glUniform1i(glGetUniformLocation(p->program->handle, "u_NoiseSize"),
                NOISE_SIZE);

    // Bind textures for upcoming draw operation
    for (size_t i = 0; i < p->n_textures; i++) {
        glActiveTexture(GL_TEXTURE0 + i);
        glBindTexture(GL_TEXTURE_2D, p->textures[i]);
        glUniform1i(
            glGetUniformLocation(p->program->handle, p->sampler_ufm_names[i]),
            i);
    }

    // Draw a screen-filling quad with the shader!
    glBindVertexArray(p->renderer->vao);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    glBindVertexArray(0);

    glDepthMask(GL_TRUE);
}
