#include "SDL2/SDL_video.h"
#include "config.h"
#include "demo.h"
#include "gl.h"
#include "music_player.h"
#include "sync.h"
#include <SDL2/SDL.h>
#include <stdlib.h>

// The surrounding () parentheses are actually important!
// Without them, the expression could be changed by it's surroundings
// after the macro is "inlined" in the preprocessor.
#define ROW_RATE ((BEATS_PER_MINUTE / 60.) * ROWS_PER_BEAT)

// The following functions and sync_cb struct are used to glue rocket to
// our music player.
#ifdef DEBUG
static void set_row(void *d, int row) {
    music_player_t *player = (music_player_t *)d;
    player_set_time(player, row / ROW_RATE);
}

static void pause(void *d, int flag) {
    music_player_t *player = (music_player_t *)d;
    player_pause(player, flag);
}

static int is_playing(void *d) {
    music_player_t *player = (music_player_t *)d;
    return player_is_playing(player);
}

static struct sync_cb rocket_callbacks = {
    .pause = pause,
    .set_row = set_row,
    .is_playing = is_playing,
};
#endif

int find_display_mode(int display, int w, int h, SDL_DisplayMode *mode) {
    int modes = SDL_GetNumDisplayModes(display);
    if (modes < 1) {
        SDL_Log("%s\n", SDL_GetError());
        return -1;
    }

    for (int i = 0; i < modes; i++) {
        if (SDL_GetDisplayMode(display, i, mode) != 0) {
            SDL_Log("%s\n", SDL_GetError());
            return -1;
        }

        SDL_Log("\nMode %d:\n", i);
        SDL_Log("\twidth %d\n", mode->w);
        SDL_Log("\theight %d\n", mode->h);
        SDL_Log("\trate %d\n", mode->refresh_rate);

        if ((mode->w == w && mode->h >= h) || (mode->h == h && mode->w >= w)) {
            return 0;
        }
    }

    return -1;
}

typedef struct {
    int flags;
    SDL_DisplayMode mode;
} fullscreen_config_t;

static fullscreen_config_t configure_fullscreen(double scale,
                                                SDL_Window *window) {
    const int w = WIDTH * scale, h = HEIGHT * scale;
    fullscreen_config_t config = {.flags = SDL_WINDOW_FULLSCREEN_DESKTOP,
                                  .mode = {.w = w, .h = h}};

    if (!window) {
        return config;
    }

    int display = SDL_GetWindowDisplayIndex(window);
    if (display < 0) {
        SDL_Log("Failed to query display mode. %s\n", SDL_GetError());
        return config;
    }

    // Configure real/emulated fullscreen modes
    config.flags = SDL_WINDOW_FULLSCREEN;

    // For everything else than wayland, find a normal fullscreen mode
    const char *driver = SDL_GetCurrentVideoDriver();
    SDL_Log("Using %s video driver\n", driver);
    if (strcmp(driver, "wayland") != 0) {
        if (find_display_mode(display, w, h, &config.mode)) {
            SDL_Log("No compatible display mode found.\n");
            return configure_fullscreen(scale, NULL);
        }
        return config;
    }

    // Because SDL's wayland driver emulates fullscreen with wp_viewporter
    // which always stretches the image, generate a mode with padding
    SDL_Log("Wayland detected, using direct viewporter output\n");
    if (SDL_GetDesktopDisplayMode(display, &config.mode) != 0) {
        SDL_Log("Failed to query display mode. %s\n", SDL_GetError());
        return configure_fullscreen(scale, NULL);
    }

    // Emulated fullscreen does not work when window is larger than display
    if (w > config.mode.w || h > config.mode.h) {
        SDL_Log("Display too small\n");
        return configure_fullscreen(scale, NULL);
    }

    double desktop_aspect = (double)config.mode.w / (double)config.mode.h;
    double output_aspect = (double)w / (double)h;

    config.mode.w = w;
    config.mode.h = h;

    if (desktop_aspect > output_aspect) {
        config.mode.w = h * desktop_aspect;
    } else {
        config.mode.h = w / desktop_aspect;
    }

    return config;
}

static void fullscreen_toggle(SDL_Window *window, demo_t *demo) {
    if (!window) {
        return;
    }

    double *scale = SDL_GetWindowData(window, "scale");

    // If in fullscreen, un-fullscreen the window
    if (SDL_GetWindowFlags(window) & SDL_WINDOW_FULLSCREEN) {
        SDL_SetWindowFullscreen(window, 0);
        int width, height;
        SDL_GL_GetDrawableSize(window, &width, &height);
        demo_resize(demo, width, height);
        SDL_Log("Fullscreen disabled.\n");
        return;
    }

    fullscreen_config_t config = configure_fullscreen(*scale, window);
    if (SDL_SetWindowDisplayMode(window, &config.mode) != 0) {
        SDL_Log("Cannot set display mode. %s\n", SDL_GetError());
        SDL_Log("Fall back to soft scaling\n");
        config = configure_fullscreen(*scale, NULL);
    }

    if (SDL_SetWindowFullscreen(window, config.flags) != 0) {
        // Extra fallback
        config = configure_fullscreen(*scale, NULL);
        if (SDL_SetWindowFullscreen(window, config.flags) != 0) {
            SDL_Log("Fullscreen failed. %s\n", SDL_GetError());
            return;
        }
    }

    demo_resize(demo, config.mode.w, config.mode.h);
    SDL_Log("Fullscreen enabled.\n");
}

// This handles SDL2 events. Returns 1 to "keep running" or 0 to "stop"/exit.
static int poll_events(demo_t *demo, struct sync_device *rocket) {
    static SDL_Event e;

    // Get SDL events, such as keyboard presses or quit-signals
    while (SDL_PollEvent(&e)) {
        if (e.type == SDL_QUIT) {
            return 0;
        } else if (e.type == SDL_KEYDOWN) {
            if (e.key.keysym.sym == SDLK_ESCAPE || e.key.keysym.sym == SDLK_q) {
                return 0;
            }
#ifdef DEBUG
            if (e.key.keysym.sym == SDLK_s) {
                sync_save_tracks(rocket);
                SDL_Log("Tracks saved.\n");
            }
            if (e.key.keysym.sym == SDLK_r) {
                demo_reload(demo);
                SDL_Log("Shaders reloaded.\n");
            }
            if (e.key.keysym.sym == SDLK_f) {
                SDL_Window *window = SDL_GetWindowFromID(e.window.windowID);
                fullscreen_toggle(window, demo);
            }
#endif
        } else if (e.type == SDL_WINDOWEVENT) {
            if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED) {
                SDL_Window *window = SDL_GetWindowFromID(e.window.windowID);
                int w, h;
                SDL_GL_GetDrawableSize(window, &w, &h);
                SDL_Log("Window size changed. New size: %d * %d\n", w, h);
                demo_resize(demo, w, h);
            }
        }
    }

    return 1;
}

// Connects rocket while keeping SDL events polled. Returns 1 when successful,
// 0 when unsuccessful.
#ifdef DEBUG
static int connect_rocket(struct sync_device *rocket, demo_t *demo) {
    SDL_Log("Connecting to Rocket editor...\n");
    while (sync_tcp_connect(rocket, "localhost", SYNC_DEFAULT_PORT)) {
        // Check exit events while waiting for Rocket connection
        if (!poll_events(demo, rocket)) {
            return 0;
        }
        SDL_Delay(200);
    }
    SDL_Log("Connected.\n");
    return 1;
}
#endif

int main(int argc, char *argv[]) {
    // Prefer wayland
    SDL_setenv("SDL_VIDEODRIVER", "wayland,x11,windows", 0);

    // Initialize SDL
    // This is required to get OpenGL and audio to work
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO)) {
        SDL_Log("SDL2 failed to initialize: %s\n", SDL_GetError());
        return 1;
    }

    // Set OpenGL version (ES 3.1 when GLES configured in make)
#ifdef GLES
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 1);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_PROFILE_MASK, SDL_GL_CONTEXT_PROFILE_ES);
#else
    // Set OpenGL version (3.3 core is the default)
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_PROFILE_MASK,
                        SDL_GL_CONTEXT_PROFILE_CORE);
#endif

    // Set OpenGL default framebuffer to sRGB without depth buffer
    // We won't need a depth buffer for running GLSL shaders
    SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 0);
    SDL_GL_SetAttribute(SDL_GL_FRAMEBUFFER_SRGB_CAPABLE, 1);

    // Create a window
    // This is what the demo gets rendered to.
    SDL_Window *window = SDL_CreateWindow(
        "demo", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, WIDTH, HEIGHT,
        SDL_WINDOW_OPENGL | SDL_WINDOW_RESIZABLE);
    if (!window) {
        SDL_Log("SDL2 failed to initialize a window: %s\n", SDL_GetError());
        return 1;
    }

    // Get an OpenGL context
    // This is needed to connect the OpenGL driver to the window we just created
    SDL_GLContext gl_context = SDL_GL_CreateContext(window);
    if (!gl_context) {
        SDL_Log("SDL2 failed to create an OpenGL context: %s\n",
                SDL_GetError());
        return 1;
    }

#ifdef __MINGW64__
    // On windows, we need to actually load/"wrangle" some OpenGL functions
    // at runtime. We use glew for that, because it's a hassle without using
    // a library.
    glewExperimental = GL_TRUE;
    GLenum err = glewInit();
    if (err != GLEW_OK) {
        SDL_Log("glew initialization failed.\n %s\n", glewGetErrorString(err));
        return 1;
    }
#endif

    // Initialize music player
    music_player_t *player = music_player_init("data/music.ogg");
    if (!player) {
        return 1;
    }

    // Initialize demo rendering
    double scale = RESOLUTION_SCALE;
    if (argc > 1) {
        double s = strtod(argv[1], NULL);
        if (s != 0.0) {
            scale = s;
        }
    }
    SDL_SetWindowData(window, "scale", &scale);
    demo_t *demo = demo_init(WIDTH * scale, HEIGHT * scale);
    if (!demo) {
        return 1;
    }

    // Resize demo to fit the window we actually got
    int width, height;
    SDL_GL_GetDrawableSize(window, &width, &height);
    demo_resize(demo, width, height);

#ifndef DEBUG
    // Put window in fullscreen when building a non-debug build
    fullscreen_toggle(window, demo);
    SDL_ShowCursor(SDL_DISABLE);
#endif

    // Initialize rocket
    struct sync_device *rocket = sync_create_device("data/sync");
    if (!rocket) {
        SDL_Log("Rocket initialization failed\n");
        return 1;
    }

#ifdef DEBUG
    // Connect rocket
    if (!connect_rocket(rocket, demo)) {
        return 0;
    }

    // Set up framerate counting
    uint64_t frames = 0;
    uint64_t frame_check_time = SDL_GetTicks64();
    uint64_t max_frame_time = 0;
    uint64_t timestamp = 0;
#endif

    // Here starts the demo's main loop
    player_pause(player, 0);
    while (poll_events(demo, rocket)) {
        // Get time from music player
        double time = player_get_time(player);
        double rocket_row = time * ROW_RATE;

#ifdef DEBUG
        // Update rocket in a loop until timed out or row changes.
        // This can cause a net delay up to n * m milliseconds, where
        // n = 10 (hardcoded loop) and m = 20 (hardcoded SDL_Delay call).
        // Total 200 ms may be blocked to save power when nothing is changing
        // in the editor.
        for (int i = 0; i < 10; i++) {
            if (sync_update(rocket, (int)rocket_row, &rocket_callbacks,
                            (void *)player)) {
                SDL_Log("Rocket disconnected\n");
                if (!connect_rocket(rocket, demo)) {
                    return 0; // how many layers of nesting are you on
                    // like, maybe 5, or 6 right now. my dude
                }
            }
            // After previous update, if player time has changed (due to seek
            // or unpause), don't continue delaying, go render.
            if (player_is_playing(player) || player_get_time(player) != time) {
                break;
            }
            // Delay
            SDL_Delay(20);
        }

        // Print FPS reading every so often
        uint64_t ct = SDL_GetTicks64();
        uint64_t ft = ct - timestamp;
        max_frame_time = ft > max_frame_time ? ft : max_frame_time;
        timestamp = ct;
        if (frame_check_time + 5000 <= ct) {
            SDL_Log("FPS: %.1f, max frametime: %lu ms\n",
                    frames * 1000. / (double)(ct - frame_check_time),
                    max_frame_time);
            frames = 0;
            max_frame_time = 0;
            frame_check_time = ct;
        }
        frames++;
#else
        // Quit the demo when music ends
        if (player_at_end(player)) {
            break;
        }
#endif

        // Render. This does draw calls.
        demo_render(demo, rocket, rocket_row);

        // Swap the render result to window, so that it becomes visible
        SDL_GL_SwapWindow(window);
    }

#ifdef DEBUG
    sync_save_tracks(rocket);
    SDL_Log("Tracks saved.\n");
#endif

    demo_deinit(demo);
    music_player_deinit(player);
    SDL_Quit();
    return 0;
}
