#version 430

#include <shaders/materials/commons.glsl>

in vec2 vTexcoord0;
uniform sampler2D sColor;
uniform sampler2D sDepth;
uniform sampler2D sCoCDepth;

// color
// output for 2 buffers

struct PostDepthOfFieldParams
{
	vec2   resolution;
	vec2   inv_resolution;

	vec4   camera_projection_params;
	vec4   near_far_plane;

	float  focus_point;
	float  focus_scale;
	float  far_plane;
	int    debug_blur_size;
};

layout (std140, row_major) uniform PostDepthOfFieldParamsBuffer {
	PostDepthOfFieldParams dof_params;
};


layout(location = 0) out vec4 outColor;

// https://blog.voxagon.se/2018/05/04/bokeh-depth-of-field-in-single-pass.html
// using simple prepass for depth/coc computation

const float GOLDEN_ANGLE = 2.39996323; 
const float MAX_BLUR_SIZE = 8.0; 
const float RAD_SCALE = 0.45; // Smaller = nicer blur, larger = faster

#if 0

#if 1
float linearizeDepth(float d)
{
	return dof_params.near_far_plane.z / (dof_params.near_far_plane.y + dof_params.near_far_plane.x - d * dof_params.near_far_plane.w);
}
#else
float linearizeDepth(float d)
{
	return d * dof_params.far_plane;
}
#endif

float getBlurSize(float depth, float focusPoint, float focusScale)
{
	float coc = clamp((1.0 / focusPoint - 1.0 / depth)*focusScale, -1.0, 1.0);
	return abs(coc) * MAX_BLUR_SIZE;
}

vec3 depthOfField(vec2 texCoord, float focusPoint, float focusScale)
{
	float centerDepth = linearizeDepth(texture(sDepth, texCoord).r);// * dof_params.far_plane;
	float centerSize = getBlurSize(centerDepth, focusPoint, focusScale);
	vec3 color = texture(sColor, texCoord).rgb;

	if (dof_params.debug_blur_size != 0)
		return vec3(centerSize * 0.01);

	if (focusPoint == 0.0)
	{
		return color;
	}
	
	float tot = 1.0;
	float radius = RAD_SCALE;
	for (float ang = 0.0; radius<MAX_BLUR_SIZE; ang += GOLDEN_ANGLE)
	{
		vec2 tc = texCoord + vec2(cos(ang), sin(ang)) * dof_params.inv_resolution * radius;
		vec3 sampleColor = texture(sColor, tc).rgb;
		//float sampleDepth = texture(sDepth, tc).r * dof_params.far_plane;
		float sampleDepth = linearizeDepth(texture(sDepth, tc).r);// * dof_params.far_plane;
		float sampleSize = getBlurSize(sampleDepth, focusPoint, focusScale);
		if (sampleDepth > centerDepth)
			sampleSize = clamp(sampleSize, 0.0, centerSize*2.0);
		float m = smoothstep(radius-0.5, radius+0.5, sampleSize);
		color += mix(color/tot, sampleColor, m);
		tot += 1.0;   radius += RAD_SCALE/radius;
	}
	return color /= tot;
}
#else
vec3 depthOfField(vec2 texCoord, float focusPoint, float focusScale)
{
	vec2 coc_depth = texture(sCoCDepth, texCoord).rg;

	float centerDepth = coc_depth.g * dof_params.far_plane;
	float centerSize = coc_depth.r;
	vec3 color = texture(sColor, texCoord).rgb;

	if (dof_params.debug_blur_size != 0)
		return vec3(centerSize * 0.01);

	if (focusPoint == 0.0)
	{
		return color;
	}
	
	float tot = 1.0;
	float radius = RAD_SCALE;
	for (float ang = 0.0; radius<MAX_BLUR_SIZE; ang += GOLDEN_ANGLE)
	{
		vec2 tc = texCoord + vec2(cos(ang), sin(ang)) * dof_params.inv_resolution * radius;
		vec3 sampleColor = texture(sColor, tc).rgb;
		//float sampleDepth = texture(sDepth, tc).r * dof_params.far_plane;
		coc_depth = texture(sCoCDepth, tc).rg;
		float sampleDepth = coc_depth.g * dof_params.far_plane;
		float sampleSize = coc_depth.r;
		if (sampleDepth > centerDepth)
			sampleSize = clamp(sampleSize, 0.0, centerSize*2.0);
		float m = smoothstep(radius-0.5, radius+0.5, sampleSize);
		color += mix(color/tot, sampleColor, m);
		tot += 1.0;
		radius += RAD_SCALE/radius;
	}
	return color /= tot;
}

#endif

void main()
{
	vec3 c = depthOfField(vTexcoord0, dof_params.focus_point, dof_params.focus_scale);
	outColor = vec4(c, 1.0);
}

