#version 460

#ifndef SPIRV_ENABLED
#extension GL_NV_gpu_shader5 : enable
#extension GL_NV_shader_thread_group : enable
#extension GL_ARB_shader_ballot : enable
#else
#extension GL_EXT_shader_explicit_arithmetic_types : enable
#extension GL_ARB_shader_ballot : enable
#extension GL_KHR_shader_subgroup_ballot : enable
#extension GL_EXT_control_flow_attributes : enable
#define HAS_16BIT_TYPES
#endif

layout(early_fragment_tests) in;

#define RT_READ_ONLY 1

uniform sampler2DArray s_BlueNoise;
uniform sampler2D      s_BRDF;

#include <shaders/materials/commons.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/commons_sphere_sampling.glsl>
#include <shaders/geometry_partitioning/raytrace_buffers.glsl>
#include <shaders/geometry_partitioning/raytrace_commons.glsl>
#include <shaders/deferred/lighting_support.glsl>

#include <shaders/materials/mapping/triplanar.glsl>

#include <shaders/materials/raytrace_simple_traversal.glsl>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

//#pragma optionNV(fastmath on)
//#pragma optionNV(fastprecision off)
//#pragma optionNV(ifcvt none)
//#pragma optionNV(inline all)
//#pragma optionNV(strict on)
//#pragma optionNV(unroll 5)

#ifndef SPIRV_VULKAN
in Vertex
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vCameraRelativeWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_input;
#else
layout(location = 1) in struct
{
	vec3 vCoords;
	vec3 vNorm;
	vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vCameraRelativeWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_input;
#endif

// Marks rays which didn't hit solid object
//#define MARK_UNFINISHED 1
//#define VISUALIZE_GRID 1
//#define VISUALIZE_HEATMAP
//#define INNER_REFLECTION
#define INTERPOLATE_NORMALS 1

#ifndef MAX_BOUNCES
#define MAX_BOUNCES 4
#endif

#ifndef MAX_TRACE_LENGTH
#define MAX_TRACE_LENGTH 1024
#endif

uniform samplerCube s_reflection;

struct RTSetup
{
	mat4  mat_projection;
	mat4  mat_model;
	vec3  camera_position;
	int   screen_sampling_scale;
	vec4  camera_projection_params;
	vec4  near_far_plane;

	float trace_range_primary;
	float trace_range_secondary;

	float roughness_clamp;
	float env_map_intensity;

	int   lights_num;

	float initial_face_start_distance;
};

layout (std140, row_major) uniform RTSetupBuffer
{
	RTSetup rt_setup;
};

// outputs depending on the pass
// -- RaytracePass
#if defined(RAYTRACE_PASS)

#ifndef GLASS_REFLECTION_PASS
layout(location = 0) out vec4  outAlbedo;
layout(location = 1) out uvec4 outHitInfo;
uniform sampler2D sFresnelReflection;
#else
layout(location = 0) out vec4 outFresnelReflection;
#endif // GLASS_REFLECTION_PASS

layout(r32ui)    uniform readonly uimage2D imPrimitiveId;
layout(r32ui)    uniform readonly uimage2D imNormalMaterial;
layout(rgba16ui) uniform readonly uimage2D imMetalnessRoughnessMaterialTags;
layout(rgba16f)  uniform readonly image2D imAlbedo;
uniform sampler2D sTextureDepth;

float linearizeDepth(in float d)
{
	return rt_setup.near_far_plane.z / (rt_setup.near_far_plane.y + rt_setup.near_far_plane.x - d * rt_setup.near_far_plane.w);
}

vec3 positionFromDepth(vec3 vDirection, float depth)
{
	return vDirection.xyz * depth;
}


#endif // RAYTRACE_PASS
//

//
#define GRID_SIZE in_bbox_data.grid_size_raytrace

vec3 TurboColormap(in float x)
{
  const vec4 kRedVec4 = vec4(0.13572138, 4.61539260, -42.66032258, 132.13108234);
  const vec4 kGreenVec4 = vec4(0.09140261, 2.19418839, 4.84296658, -14.18503333);
  const vec4 kBlueVec4 = vec4(0.10667330, 12.64194608, -60.58204836, 110.36276771);
  const vec2 kRedVec2 = vec2(-152.94239396, 59.28637943);
  const vec2 kGreenVec2 = vec2(4.27729857, 2.82956604);
  const vec2 kBlueVec2 = vec2(-89.90310912, 27.34824973);
  
  x = clamp(x, 0.0, 1.0);
  vec4 v4 = vec4( 1.0, x, x * x, x * x * x);
  vec2 v2 = v4.zw * v4.z;
  return vec3(
	dot(v4, kRedVec4)   + dot(v2, kRedVec2),
	dot(v4, kGreenVec4) + dot(v2, kGreenVec2),
	dot(v4, kBlueVec4)  + dot(v2, kBlueVec2)
  );
}

vec3 glass_refract(vec3 v, f16vec3 n)
{
	// TODO: add param for air->glass vs glass->air
	// Here value for air->glass
	//return refract(v, n, 1.0/1.5);
	//return n;

	float s = dot(f16vec3(v), n) < 0.0 ? 1.0 : -1.0;
	vec3 new_v = refract(v, n * s, 1.0 / 1.25);

	//return v;
	if (dot(new_v, new_v) == 0.0)
	{
		return v;
	}

	return new_v;
}

f16vec4 sample_env_map(in vec3 r, float roughness)
{
	vec4 v = textureLod(s_reflection, r, 0.5f + roughness * 15.0f);
	return f16vec4(max(vec4(0.0f), v));
}

vec3 calculate_lighting_world(LightProperties light, in vec3 pos, in f16vec3 normal, in vec3 light_pos, in float16_t NdotL)
{
	float16_t d = NdotL;
	if (d < 0.0)
		d = float16_t(0.0);
	return light.diffuse.xyz * d;
}

struct IBLOutput
{	
	vec3 color_weighted;
	vec3 color_raw;
};

IBLOutput ibl(vec3 n, vec3 v, vec3 diffuseColor, vec3 specularColor, vec3 f0, vec3 f90, float perceptual_roughness)
{
	IBLOutput ibl;

	float NdotV = clamp(dot(n, v), 0.0, 1.0);
	vec3 r = reflect(-v, n);

	float mip_count = 7.5;
	float lod       = clamp(perceptual_roughness * mip_count, 0.0, float(mip_count));
	vec2 brdf_sample_point = clamp(float2(NdotV, 1.0 - perceptual_roughness), vec2(0.0, 0.0), vec2(1.0, 1.0));

	vec2 brdf          = texture(s_BRDF, brdf_sample_point).rg;
	vec3 diffuseLight  = textureLod(s_reflection, n, 5.0).rgb;
	vec3 specularLight = textureLod(s_reflection, r, lod).rgb * (brdf.x + brdf.y);

	// output two values, used for noise separation. diffuse and color are/can be noisy
	ibl.color_weighted = (diffuseColor * diffuseLight + specularColor * specularLight);
	ibl.color_raw      = (diffuseLight + specularLight);
	return ibl;
}

vec3 sample_projector_texture(in sampler2D smpl, in vec4 coords)
{
	vec3 color = vec3(0.0);

	// no shading if fragment behind
	if (coords.w <= 0.0)
		return color;

	// no shading if fragment outside of light frustum
	if (coords.x < -coords.w || coords.x > coords.w || coords.y < -coords.w || coords.y > coords.w)
		return color;

	coords.xy = coords.xy * vec2(0.5) + vec2(0.5) * coords.w;

#ifdef SPIRV_VULKAN
	coords.y = coords.y /coords.w;
	coords.y = 1.0 - coords.y;
	coords.y = coords.y * coords.w;
#endif

	vec3 samp = coords.xyz / coords.w;
	color = texture(smpl, coords.xy / coords.w).rgb;
	//shadow_dist = texture(LightShadowmapCmpSamplers[light.projector_sampler], coords.xy / coords.w).r - coords.z / coords.w;
	return color.rgb;
}

// VERY simple version
#if 0
vec3 evaluate_lighting(vec3 p, vec3 o, f16vec3 n, int material, f16vec3 albedo, f16vec3 emissive)
{
	vec3 c = vec3(0.0);
	vec3 world = p;

	f16vec3 f0 = f16vec3(0.04, 0.04, 0.04);
	float16_t metallic = float16_t(materials.material_properties[material].metalness);
	f16vec3 baseColor = f16vec3(materials.material_properties[material].diffuse.rgb) * albedo.rgb;
	f16vec3 diffuseColor = baseColor.rgb * (f16vec3(1.0) - f0) * (1.0 - metallic);
	f16vec3 specularColor = mix(f0, baseColor.rgb, metallic);

	// iterate through lights. no specular component, only diffuse
	for(int light_idx = 0; light_idx < rt_setup.lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];

		vec3 light_color = vec3(0.0);
		vec4 projector_color = vec4(0.0);

		float16_t NdotL = dot(n, f16vec3(normalize(light.position.xyz - world.xyz)));

		// For transparents we assume light passed. Bit silly, but...
		if ((materials.material_properties[material].flags & MaterialFlag_Transparent) != 0)
			NdotL = abs(NdotL);
		vec3 lighting = calculate_lighting_world(light, world, n, light.position.xyz, NdotL);
		
		c += lighting;
	}

	c += emissive * materials.material_properties[material].emissive_factor;

	return c;
}

#else

f16vec3 evaluate_lighting(vec3 p, vec3 o, f16vec3 n, int material, f16vec3 albedo, f16vec3 emissive)
{
	vec3 c     = vec3(0.0);
	vec3 world = p;
	vec3 view  = normalize(o - p);

	f16vec3 f0            = f16vec3(0.04f, 0.04f, 0.04f);
	float16_t metallic    = float16_t(materials.material_properties[material].metalness);
	f16vec3 baseColor     = f16vec3(clamp(materials.material_properties[material].diffuse.rgb * albedo.rgb, vec3(0.0), vec3(1.0)));
	f16vec3 diffuseColor  = baseColor.rgb * (f16vec3(1.0f) - f0) * (float16_t(1.0) - metallic);
	f16vec3 specularColor = f16vec3(mix(f0, baseColor.rgb, metallic));

	f16vec3 specularEnvironmentR0 = specularColor.rgb;
	// Anything less than 2% is physically impossible and is instead considered to be shadowing. Compare to "Real-Time-Rendering" 4th editon on page 325.
	float reflectance             = float16_t(max(max(specularColor.r, specularColor.g), specularColor.b));
	vec3 specularEnvironmentR90   = vec3(1.0, 1.0, 1.0) * clamp(reflectance * 25.0, 0.0, 1.0);

	float roughness = materials.material_properties[material].roughness;
	float alphaRoughness = roughness * roughness;

	MaterialInfo materialInfo =
	{
		roughness,
		specularEnvironmentR0,
		alphaRoughness,
		diffuseColor,
		specularEnvironmentR90,
		specularColor
	};

	// iterate through lights. no specular component, only diffuse
	for(int light_idx = 0; light_idx < rt_setup.lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];
		if ((light.type & (LightType_Spot| LightType_Directional)) != 0)
		//if ((light.type & (LightType_Spot)) != 0)
		{
			vec3 pointToLight = light.position.xyz - world.xyz;
			if ((light.type & LightType_Directional) != 0)
				pointToLight = -light.direction.xyz;

			pointToLight = normalize(pointToLight);
			float NdotL = dot(vec3(n), pointToLight);

			vec3 lighting = vec3(0.0);
			if (NdotL > 0.0)
			{
				float attenuation = 1.0;
				if ((light.type & LightType_Spot) != 0)
					attenuation = light_calculate_spot_attenuation(light, world.xyz);
			
				if (attenuation > 0.0)
				{
					vec4 vShadowCoords = light.mat_shadow_p[0] * light.mat_shadow_mv * vec4(world.xyz, 1.0);
					lighting = getPointShade(pointToLight, materialInfo, n, view) * (light.intensity * attenuation) * light.diffuse.rgb;

					if ((light.type & LightType_Projector) != 0)
					{
						vec4 vShadowCoords = light.mat_shadow_p[0] * light.mat_shadow_mv * vec4(world.xyz, 1.0);
						float shadow = 0.0;

						lighting *= sample_projector_texture(
							LightProjectorSamplers[light.projector_sampler],
							vShadowCoords).rgb * ((1.0 - shadow) * light.projector_intensity);
					}

					c += lighting;
				}
			}
		}
	}

	IBLOutput ibl = ibl(n, view, diffuseColor, specularColor, specularEnvironmentR0, specularEnvironmentR90, roughness);
	c += ibl.color_weighted * rt_setup.env_map_intensity;

	c += emissive * materials.material_properties[material].emissive_factor;
	//c = world * 0.0001;
	//c = n * 0.5 + 0.5;
	return f16vec3(c);
	//return f16vec3(1.0);
}

#endif


f16vec4 evaluate_albedo(int16_t materialIndex, int fi, f16vec2 bc)
{
	f16vec4 c = interpolate_color_from_bc_yz(fi, bc);
	if (materials.material_properties[materialIndex].albedo_sampler >= 0)
	{
		f16vec2 uv = interpolate_uv_from_bc_yz(fi, bc);
		c *= f16vec4(textureLod(material_textures[materials.material_properties[materialIndex].albedo_sampler], uv, 1.0).rgba);
		//c.rg = f16vec2(uv);
		//c.b = 1.0;
	}

	return c;
}

f16vec3 evaluate_emissive(int16_t materialIndex, int fi, f16vec2 bc)
{
	f16vec3 c = f16vec3(1.0);
	if (materials.material_properties[materialIndex].emissive_sampler >= 0)
	{
		f16vec2 uv = interpolate_uv_from_bc_yz(fi, bc);
		c = f16vec3(textureLod(material_textures[materials.material_properties[materialIndex].emissive_sampler], uv, 1.0).rgb);
	}

	return c;
}

vec3 yCgCo2rgb(vec3 ycc)
{
	float R = ycc.x - ycc.y + ycc.z;
	float G = ycc.x + ycc.y;
	float B = ycc.x - ycc.y - ycc.z;
	return vec3(R,G,B);
}

vec3 spectrum_offset_ycgco( float t )
{
	//vec3 ygo = vec3( 1.0, 1.5*t, 0.0 ); //green-pink
	//vec3 ygo = vec3( 1.0, -1.5*t, 0.0 ); //green-purple
	vec3 ygo = vec3( 1.0, 0.0, -1.25*t ); //cyan-orange
	//vec3 ygo = vec3( 1.0, 0.0, 1.5*t ); //brownyello-blue
	return yCgCo2rgb( ygo );
}

vec3 spectrum_offset_rgb( float t )
{
	float t0 = 3.0 * t - 1.5;
	vec3 ret = clamp( vec3( -t0, 1.0-abs(t0), t0), vec3(0.0), vec3(1.0));
	return ret;
}

void evaluate_material(in out ray_state state, in vec3 prev_state_origin, int hit_face, uint hit_material_flags, f16vec2 bc, bool flip_normal_on_glass)
{
	int16_t hit_material = int16_t(rt_get_triangle_material(hit_face));
	f16vec4 hit_albedo = evaluate_albedo(hit_material, hit_face, bc);
	f16vec3 hit_emissive = evaluate_emissive(hit_material, hit_face, bc) * f16vec3(materials.material_properties[hit_material].emissive.rgb);

	//f16vec3 hit_color = evaluate_lighting(state.origin, prev_state_origin, state.normal, hit_material, hit_albedo.rgb, hit_emissive);
	f16vec3 hit_color;
	{
		// When crossing the glass, when computing the light we need to pretend we are on the other side of the point
		//

		vec3 camera_position = rt_setup.camera_position.xyz;
		if (flip_normal_on_glass)
		{
			vec3 d = camera_position - state.origin;
			camera_position = reflect(d, -state.normal) + state.origin;
		}
		hit_color = evaluate_lighting(state.origin, camera_position, state.normal, hit_material, hit_albedo.rgb, hit_emissive);
	}
	//vec3 hit_color = vec3(bc, 1.0);//evaluate_lighting(state.origin, prev_state_origin, state.normal, hit_material, hit_albedo, hit_emissive);
	//vec3 hit_color = materials.material_properties[hit_material].diffuse.rgb;// evaluate_lighting(state.origin, prev_state_origin, state.normal, hit_material, hit_albedo, hit_emissive);

	// NOTE: Why not using _Reflective property? Because this instruments render passes if the material should
	// be raytraced and we might not want that. Maybe it should be separated into two fields actually?

	bool hit_material_reflective = materials.material_properties[hit_material].roughness < 1.0;
	if ((materials.material_properties[state.material].flags & MaterialFlag_Transparent) == 0)
	{
		// NOTE: This is some old mixing code which seems completely off/broken....
		//hit_color   = f16vec3(mix(state.color * hit_color, state.color, float16_t(materials.material_properties[state.material].roughness)));
		hit_color   = f16vec3(state.color * hit_color);

		state.color = f16vec3(mix(state.color, hit_color, float16_t(state.transparency)));
		//state.transparency = mix(state.transparency, 0.0, (1.0 - materials.material_properties[hit_material].transparency));
		//state.transparency = min(state.transparency, materials.material_properties[hit_material].transparency);
		//state.transparency = mix(state.transparency, 1.0, 1.0 - materials.material_properties[hit_material].transparency);
		// NOTE: don't modulate transparency as we want to keep current one for this ray
		state.hit   = state.hit || (state.bounces == MAX_BOUNCES);

	}
	else if (hit_material_reflective == false && (hit_material_flags & MaterialFlag_Transparent) == 0) // diffuse, no reflection, abort
	{
		state.color   = f16vec3(mix(state.color, hit_color, state.transparency));
		state.running = false;
		state.hit     = true;
	}
	else if (hit_material_reflective == true && (hit_material_flags & MaterialFlag_Transparent) == 0) // reflective
	{
		// reflective material
		state.color = f16vec3(mix(state.color, hit_color, state.transparency));
		state.hit   = state.hit || (state.bounces == MAX_BOUNCES);
	}
	else //if (materials.material_properties[state.material].transparency > 0.0)
	{
		// beer's (approx, VERY rough, very high transmitance)
		float alpha = 0.25 * (1.0 - materials.material_properties[state.material].transparency);
		float absorbance = alpha * -length(prev_state_origin - state.origin);
		float attenuation = exp(absorbance);
		//if (state.inside_transparent == false)

		bool is_current_transparent_and_doublesided = 
			(materials.material_properties[state.material].flags & (MaterialFlag_Transparent | MaterialFlag_Doublesided)) == (MaterialFlag_Transparent | MaterialFlag_Doublesided);
		if (is_current_transparent_and_doublesided == false)
			attenuation = 1.0;

		float hit_transparency = 1.0 - (1.0 - materials.material_properties[hit_material].transparency) * hit_albedo.a;
		state.transparency = state.transparency * float16_t(hit_transparency) * attenuation;
		state.color        = f16vec3(mix(state.color, hit_color, state.transparency));
	}

	if ((hit_material_flags & MaterialFlag_Transparent) == 0)
	{
		state.dir = reflect(state.dir, vec3(state.normal));
		state.inside_transparent = false;
	}
	else
	{
		vec3 refracted_dir = glass_refract(state.dir, state.normal);
		float angle = dot(refracted_dir, state.dir);

		#if 1
		float threshold = 0.;
		angle = 1.0 - abs(angle);
		{
			//angle = angle * angle;
			//angle = sqrt(angle);
			float factor = clamp(angle * 5.0, 0.0, 1.0);
			angle = angle * 10.5;
			//angle = clamp(angle, 0.0, 1.0);
			//state.color = mix(state.color, spectrum_offset_rgb(angle), factor);
			state.color = mix(state.color, spectrum_offset_ycgco(angle), factor);
			//state.color = spectrum_offset_rgb(angle * 10.0);
			//state.color = vec3(factor);
		}
		#endif
		state.dir                = refracted_dir;
		state.inside_transparent = !state.inside_transparent;
		//state.color.rgb = vec3(hit_albedo.a);
	}

	if ((hit_material_flags & MaterialFlag_Reflective) == 0
		&& (hit_material_flags & MaterialFlag_Transparent) == 0)
	{
		state.running = false;
		state.hit = true;
	}

	//state.color = hit_color;//state.normal * 0.5 + 0.5;
	//state.running = false;
	//state.hit = true;

	state.material = hit_material;
}

//#define findClosest findClosestNaive
#define findClosest findClosestDDA

vec3 rt_randomize_dir_for_roughness(ivec2 screen_pos, vec3 dir, vec3 n, float roughness, out float bounce_throughput)
{
	bounce_throughput = 1.0;
#if 1
	// NOTE: This is REALLY costly when divergens goes to hell, so for now because we don't cluster
	// rays just try to limit the roughness...
	if (roughness > 0.0f)
	{
		roughness = min(rt_setup.roughness_clamp, roughness);
		roughness = dot(dir, n) * roughness;
		bounce_throughput = bounce_throughput * max(0.0, 1.0 - roughness);

		const float golden_ratio = 1.61803398875;
		int frame = globals.monotonic & 127;
		float clamped_roughness = roughness * roughness;

		screen_pos = screen_pos & ivec2(127);
		vec2 noise = texelFetch(s_BlueNoise, ivec3(screen_pos, 0), 0).rg;
		vec2 hash = fract(noise + frame * golden_ratio);
		//vec2 hash = fract(texelFetch(s_BlueNoise, ivec3(screen_pos, 0), 0).rg);

		vec3 d = CosineSampleHemisphere(hash.x * clamped_roughness, hash.y);
		mat3 vecSpace = matrixFromVector(dir);
		d = vecSpace * d;

		// check if d potentially traces 'into the floor'. this happens with high roughness and hemisphere that is highly rotated and goes into the floor...
		float VdotN = dot(d, n);
		if (VdotN < 0.0 && true)
		{
			// try another one... if this correct even? 'Stochastic Screen-Space Reflections' by Stachowiak (siggraph, 2015, page 43 mention they re-generate)
			// seems to help to some degree at least
			hash = fract(noise + (frame + 10) * golden_ratio);
			d = CosineSampleHemisphere(hash.x * clamped_roughness, hash.y);
			d = vecSpace * d;
			VdotN = dot(d, n);
			if (VdotN > 0.0)
				dir = d;
		}
		else
		{
			dir = d;
		}
	}
#endif

	return dir;
}

vec3 rt_reflect_dir_for_roughness_sample_ggx(ivec2 screen_pos, vec3 dir, vec3 n, float roughness, out float bounce_throughput)
{
	const float golden_ratio = 1.61803398875;
	int frame = globals.monotonic & 127;
	float clamped_roughness = roughness;
	//clamped_roughness = clamped_roughness * clamped_roughness;
	clamped_roughness = min(rt_setup.roughness_clamp, clamped_roughness * clamped_roughness);

	screen_pos = screen_pos & ivec2(127);
	vec2 noise = texelFetch(s_BlueNoise, ivec3(screen_pos, 0), 0).rg;
	vec2 hash = fract(noise + frame * golden_ratio);

	//mat3 basis = matrixFromVector(n);
	mat3 basis = construct_ONB_frisvad(n);	// NOTE: Has to be this one...

	vec3 V = dir;
	vec3 H = normalize(basis * ImportanceSampleGGX_VNDF(hash, clamped_roughness, V, basis));
	vec3 L = reflect(V, H);

	float NoL = max(0.0, dot(n, L));
	float NoV = max(0.0, -dot(n, V));
	float VoH = max(0, -dot(V, H));

	if (NoL < 0.0 || NoV < 0.0)
	{
		hash = fract(noise + (frame + 10) * golden_ratio);
		H = normalize(basis * ImportanceSampleGGX_VNDF(hash, clamped_roughness, V, basis));
		L = reflect(V, H);

		NoL = max(0.0, dot(n, L));
		NoV = max(0.0, -dot(n, V));
		VoH = max(0, -dot(V, H));
	}

	if (NoL > 0.0 && NoV > 0.0)
	{
		// See the Heitz paper referenced above for the estimator explanation.
		//   (BRDF / PDF) = F * G2(V, L) / G1(V)
		// Assume G2 = G1(V) * G1(L) here and simplify that expression to just G1(L).
					
		float G1_NoL = G1_Smith(clamped_roughness, NoL);
		vec3 F = schlick_fresnel(vec3(0.04, 0.04, 0.04), VoH, 1.0);

		bounce_throughput = length(G1_NoL * F);
		return normalize(L);
	}

	bounce_throughput = 0.0;
	return dir;
}


void main() {

#ifndef RAYTRACE_PASS
#else

	#ifndef GLASS_REFLECTION_PASS
	outHitInfo.r = 0;
	//outAlbedo = vec4(1.0);
	//return;
	#endif

	ivec2 scaled_sample_pos = ivec2(gl_FragCoord.xy) * rt_setup.screen_sampling_scale;
	ivec2 native_sample_pos =  ivec2(gl_FragCoord.xy);

#ifdef SPIRV_VULKAN
	//scaled_sample_pos.y = 1080 - scaled_sample_pos.y;
	//native_sample_pos.y = 1080 - native_sample_pos.y;
#endif

	vec4 color = vec4(1.0);

	// for now also trace to the point we just hit. we would need some way to identify the face we are rasterizing
	// 

	vec3 worldPos = vtx_input.vCameraRelativeWorldPos.xyz;// - transform_params.vCameraPosition.xyz;
	vec3 dir = -normalize(transform_params.vCameraPosition.xyz - worldPos);	// TODO: simplify with the above
	vec3 origin = transform_params.vCameraPosition.xyz;
	f16vec3 normal;

	int closest_fi = -1;
	int16_t material = int16_t(0);

	// if not running with stencil we simply discard based on material
	{
		float metalness;
		float roughness;
		uint materialIndex;
		decode_metalness_roughness_material(imageLoad(imMetalnessRoughnessMaterialTags, scaled_sample_pos).rg, metalness, roughness, materialIndex);
		//MaterialPropertiesGPU material = materials.material_properties[materialIndex];
		if (materialIndex == 1) // || materials.material_properties[materialIndex].raytrace == 0.0)
		{
			//discard;
			//return;
		}

		material = int16_t(materialIndex);
	}

	vec3 view_direction;
	view_direction.x = -rt_setup.camera_projection_params.x * 0.5 + rt_setup.camera_projection_params.x * scaled_sample_pos.x / 1920.0;
	view_direction.y = -rt_setup.camera_projection_params.y * 0.5 + rt_setup.camera_projection_params.y * scaled_sample_pos.y / 1080.0;
	view_direction.z = 1.0;

#ifdef SPIRV_VULKAN
	view_direction.y = -view_direction.y;
#endif

	float depth = linearizeDepth(texelFetch(sTextureDepth, native_sample_pos, 0).r);
	vec3 view_coords = positionFromDepth(view_direction, depth);
	view_coords = (rt_setup.mat_model * vec4(view_coords, 1.0)).xyz;

	dir = -normalize(transform_params.vCameraPosition.xyz - view_coords.xyz);

	//outAlbedo.rgb = vec3(fract(view_coords.xyz * 0.1));
	//outAlbedo.rgb = vec3(fract(dir.xyz));
	//outAlbedo.rgb = vec3(TurboColormap(fract(float(closest_fi) * 0.001)));
	//return;

	float closest_it;

	// TODO: optimize with simple length

	f16vec3 worldNorm;
	{
		closest_it = length(origin - view_coords.xyz);
		uint encoded_normal_material = imageLoad(imNormalMaterial, scaled_sample_pos).r;
		normal = f16vec3(normalize(decode_normal(encoded_normal_material)));
		worldNorm = normal;

		//outNormalMaterial = encode_normal_material(normalize(normal), 0);
		//outAlbedo.rgb = normal.rgb * 2.5 + 0.5;
		//return;
	}

#ifndef GLASS_REFLECTION_PASS
	{
		vec3 ro = origin + dir * closest_it - in_bbox_data.bbox_raytrace_min.xyz;
		ivec3 icell = ivec3(floor(ro / vec3(GRID_SIZE)));
		if (icell.x >=0 && icell.y >= 0 && icell.z >= 0 && icell.x < GRID_RES && icell.y < GRID_RES && icell.z < GRID_RES)
		{
			uint icell_idx = icell.z * (GRID_RES * GRID_RES) + icell.y * GRID_RES + icell.x;
			//outAlbedo.rgb = vec3(in_buckets.sizes[icell_idx]) / 10.0;
			//return;
		}
		else
		{

		}
	}
#endif

#ifdef GLASS_REFLECTION_PASS
	outFresnelReflection = vec4(0.0);
#endif

	{
		origin = origin + dir * closest_it;

		// NOTE: Start with albedo color. lighting for the primary hit is calculated in normal lighting pass
		// not here, so it will be multiplied later

		ray_state state;
		ray_traversal_params traversal_params;

		traversal_params.trace_range_primary   = rt_setup.trace_range_primary;
		traversal_params.trace_range_secondary = rt_setup.trace_range_secondary;

		{
			// albedo holds opacity. 0 = 100% transparent
			vec4 albedo_value = imageLoad(imAlbedo, scaled_sample_pos).rgba;
			state.color        = f16vec3(1.0);
			state.transparency = float16_t(materials.material_properties[material].transparency == 0.0 ? 1.0 : materials.material_properties[material].transparency);
			state.transparency = 1.0 - (1.0 - state.transparency) * albedo_value.a;

			//if (state.transparency < 1.0)
			//	state.color = albedo_value.rgb;
		}
		state.normal                 = normal;
		state.material               = material;
		state.running                = true;
		state.dir                    = dir;
		state.origin                 = origin;
		state.bounces                = int16_t(1);
		state.hit                    = false;
		state.left                   = false;
		state.inside_transparent     = false;
		state.tests                  = 0;
		state.face_tests             = 0;
		state.active_threads_factor  = 0;
		state.active_threads_samples = 0;
		state.final_color_factor     = 1.0;

		bool pre_step_along_ray    = false;
		bool pre_step_along_normal = !pre_step_along_ray;

		if (materials.material_properties[material].roughness == 1.0 && (materials.material_properties[material].flags & MaterialFlag_Transparent) == 0)
		{
			dir = reflect(dir, vec3(normal));
			//state.color = materials.material_properties[material].diffuse.rgb;
			state.hit = true;
		}
		else
		{
			// glass pass:
			// only one look for the reflection. we will limit fresnel so it is not too visible
			if ((materials.material_properties[material].flags & MaterialFlag_Transparent) != 0)
			{

#ifdef GLASS_REFLECTION_PASS

				float fresnel = pow(1.0f - max(0.0f, dot(vec3(worldNorm), normalize(transform_params.vCameraPosition - view_coords.xyz))), 4.0f);
				float reflection =  mix(1.0 - state.transparency, 1.0, fresnel);

				vec3 glass_reflection = vec3(0.0f);
				const float reflection_threshold = 0.001f;
				if (reflection > reflection_threshold) // no point tracing below that
				{
					if (pre_step_along_normal)
						state.origin += vec3(normal) * rt_setup.initial_face_start_distance;

					vec3 ref_dir = reflect(dir, vec3(normal));
					// NOTE: This is REALLY costly when divergens goes to hell, so for now because we don't cluster
					// rays just try to limit the roughness...
					if (materials.material_properties[state.material].roughness > 0.0f)
					{
						ref_dir = rt_randomize_dir_for_roughness(ivec2(native_sample_pos), ref_dir, vec3(normal), materials.material_properties[state.material].roughness, state.final_color_factor);
					}

					int ref_closest_fi;
					float ref_closest_it;

					state.dir = ref_dir;

					if (pre_step_along_ray)
						state.origin += state.dir * rt_setup.initial_face_start_distance;

					findClosestDDAMultibounce(traversal_params, state, closest_fi, ref_closest_fi, ref_closest_it, 1);

					if (ref_closest_fi != -1)
						glass_reflection = state.color;
					else
						glass_reflection = sample_env_map(normal, materials.material_properties[state.material].roughness).rgb;

					reflection = (reflection - reflection_threshold) / (1.0f - reflection_threshold); // normalize
				}

				outFresnelReflection = vec4(glass_reflection.rgb, reflection);
				return;
#endif

			}

#ifdef GLASS_REFLECTION_PASS
			return;
#endif

			if ((materials.material_properties[state.material].flags & MaterialFlag_Transparent) == 0)
			{
				if (pre_step_along_normal)
					state.origin += vec3(normal) * rt_setup.initial_face_start_distance;

				// this is initial sample and this is only place where for now we apply roughness
				#if 1
				state.dir = reflect(state.dir, vec3(state.normal));
				#else
				// NOTE: There is still something broken here:(
				state.dir = rt_reflect_dir_for_roughness_sample_ggx(
					ivec2(native_sample_pos),
					state.dir,
					state.normal,
					materials.material_properties[state.material].roughness,
					state.final_color_factor
				);
				#endif

				if (state.final_color_factor <= 0.0)
					return;
			}
			else // if ((materials.material_properties[state.material].flags & MaterialFlag_Transparent) == 0)
			{
				// pre-step inside the shape
				if (pre_step_along_normal)
					state.origin += vec3(-normal) * rt_setup.initial_face_start_distance;

				state.dir                = glass_refract(state.dir, -state.normal);
				state.inside_transparent = true;
			}

			#if 1
			{
				// NOTE: This is REALLY costly when divergens goes to hell, so for now because we don't cluster
				// rays just try to limit the roughness...
				if (materials.material_properties[state.material].roughness > 0.0f)
				{
					state.dir = rt_randomize_dir_for_roughness(ivec2(native_sample_pos), state.dir, state.normal, materials.material_properties[state.material].roughness, state.final_color_factor);
				}
			}
			#endif

			{
				//vec3 color = TurboColormap(float(state.tests) / 256.0f);
				//outAlbedo.rgb = color;
				//return;
			}

			dir = state.dir;

			// pre-step using reflected normal
			if (pre_step_along_ray)
				state.origin += state.dir * rt_setup.initial_face_start_distance;

			findClosestDDAMultibounce(traversal_params, state, closest_fi, closest_fi, closest_it, MAX_BOUNCES);

			// Sample env map only if we didn't hit anything after first bounce. first hit is handled by normal deferred pipeline
			if (state.hit == false && state.bounces >= int16_t(2))
			{
				float current_roughness = materials.material_properties[state.material].roughness;
				if ((materials.material_properties[state.material].flags & MaterialFlag_Transparent) != 0)
				{
					vec3 env_map_color = sample_env_map(state.dir, current_roughness).rgb;
					//state.color = vec4(state.transparency); //mix(state.color, sample_env_map(state.dir), mix(factor, 1.0f, state.transparency));
					//state.color = state.color;//mix(state.color, sample_env_map(state.dir), mix(factor, 1.0f, state.transparency));
					//state.color = mix(state.color, mix(sample_env_map(state.dir), state.color, materials.material_properties[material].roughness), state.transparency);
					state.color = mix(state.color, env_map_color, 1.0 - state.transparency);
				}
				else
				{
					//state.color = vec4(state.transparency); //mix(state.color, sample_env_map(state.dir), mix(factor, 1.0f, state.transparency));

					f16vec3 mat_color = f16vec3(mix(
						materials.material_properties[material].diffuse.rgb * sample_env_map(dir, current_roughness).rgb,
						materials.material_properties[material].diffuse.rgb,
						materials.material_properties[material].roughness
					));
					//state.color = mix(state.color, state.color * sample_env_map(state.dir, current_roughness).rgb, 1.0f - state.transparency);
					state.color = f16vec3(mix(state.color, state.color * sample_env_map(state.dir, current_roughness).rgb, state.transparency));
					state.color = f16vec3(mix(mat_color, state.color, state.transparency));
				}
			}
			else if (state.hit == false && state.bounces < int16_t(2))
			{
				// if we didn't hit anything then output black as the reflection
				state.color.rgb = vec3(0.0f, 0.0f, 0.0f);
				//state.color = mix(state.color, materials.material_properties[material].diffuse, 1.0f - materials.material_properties[material].transparency);
			}

			// state.color = vec4(float(state.bounces) / 10.0f);
			//state.color = mix(state.color, materials.material_properties[material].diffuse, 1.0f - materials.material_properties[material].transparency);

		}

		color.rgb = state.color * state.final_color_factor;
		//color.rgb = vec3(state.transparency);
		//color.rgb = state.normal;// * 0.5 + 0.5;
		//color.rgb = TurboColormap(fract(float(state.material) / 10.0f));

		//#ifdef VISUALIZE_HEATMAP
		
		//color.rgb = TurboColormap(float(state.face_tests) / 1024.0f);
		//color.rgb = TurboColormap(float(state.tests) / 256.0f);
		//color.rgb = TurboColormap(float(ballot_count(state.hit)) /64.0f);
		if (state.hit)
		{
			//color.rgb = vec3(0.0f, 1.0f, 0.0f);
		}
		
		//#endif

		//color.rgb = TurboColormap(float(state.active_threads_factor) / (state.active_threads_samples * 64));
		//if (state.left)
		//	color.rgb = vec3(0.0, 1.0, 0.0);
#if 0
		int bi = state.bounces;
		if (bi == 0)
			color.rgb = vec3(1.0f);
		if (bi == 1)
			color.rgb = vec3(1.0f, 0.0f, 0.0f);
		if (bi == 2)
			color.rgb = vec3(0.0f, 1.0f, 0.0f);
		if (bi == 3)
			color.rgb = vec3(0.0f, 0.0f, 1.0f);
		if (bi > 3)
			color.rgb = vec3(0.5f, 0.0f, 1.0f);
#endif

		//color.rgb = vec3(state.transparency);
		#ifndef GLASS_REFLECTION_PASS

		if (materials.material_properties[material].transparency == 0.0f)
			outAlbedo.a = 0.0f;
		else
			outAlbedo.a = state.transparency;

#ifndef GLASS_REFLECTION_PASS
		outHitInfo.r = state.bounces;
#endif

		#endif
	}

#ifndef GLASS_REFLECTION_PASS

	vec4 glass_reflection = texelFetch(sFresnelReflection, native_sample_pos, 0);
	color.rgb = mix(color.rgb, glass_reflection.rgb, vec3(glass_reflection.a));
	//color.rgb = color.rgb + glass_reflection.rgb * vec3(glass_reflection.a);
	outAlbedo.rgb = color.rgb;
	//outAlbedo.rgb = glass_reflection.rgb;
	//outAlbedo.rgb = glass_reflection.aaa;
#endif

#endif

}

