# convert2digi
# a tool to read wav or mp3 files, compressing them and writing a binary file
# that can be used with my 8-bit digi player for the C64
# by Wil
# V1.0 2023 March

from __future__ import print_function
import os, sys, re
import wave
import math
import numpy as np 
from sklearn.cluster import KMeans  # import KMeans from scikit-learn
import argparse

# volume tables from Pex Mahoney Tufvesson (https://livet.se/mahoney/)
volume_table_common_8580_256_of_256=[159,159,223,223,223,223,223,158,158,158,158,158,222,222,222,222,222,157,157,157,157,157,157,221,221,221,221,221,156,156,156,156,156,220,220,220,220,220,155,155,155,155,155,155,219,219,219,219,219,154,154,154,154,154,218,218,218,218,218,153,153,153,153,153,153,217,217,217,217,217,152,152,152,152,152,216,216,216,216,216,216,63,63,63,63,151,151,151,215,215,62,62,126,126,126,61,61,125,125,125,60,60,124,124,124,59,59,123,123,123,58,58,122,122,122,57,57,57,121,121,24,24,88,88,88,120,23,23,55,87,119,22,22,22,86,86,86,21,21,85,85,85,20,20,20,84,84,19,19,19,83,83,83,18,18,18,82,82,17,17,17,81,81,240,144,128,128,128,128,1,1,1,65,65,65,2,2,66,66,66,3,3,3,3,67,67,4,4,4,4,68,68,68,5,5,5,69,69,6,6,6,6,70,70,7,7,7,7,71,71,71,40,8,8,8,72,72,41,9,9,9,73,73,10,10,10,10,74,74,11,11,11,75,75,75,44,12,12,12,76,76,45,13,13,77,77,77,46,14,14,78,78,78,47,15,15]
volume_table_common_6581_256_of_256=[159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,158,158,158,158,158,157,157,157,157,157,157,156,156,156,156,156,156,155,155,155,155,155,220,220,154,154,154,154,219,153,153,153,153,153,153,218,152,152,152,152,152,217,186,186,151,151,151,151,151,151,151,151,215,215,150,150,150,150,214,214,214,149,149,149,149,149,213,213,213,148,148,148,148,181,212,212,212,180,180,147,147,244,211,211,179,179,179,146,146,146,210,210,178,178,242,242,145,145,145,209,177,241,241,18,81,81,113,144,48,0,161,162,163,164,164,165,65,97,33,33,1,1,66,66,66,98,98,34,34,34,34,67,67,99,99,99,35,35,35,35,68,100,100,100,69,69,69,36,36,36,36,101,70,70,70,70,37,37,37,71,71,71,103,103,38,38,38,38,72,72,72,104,104,39,39,39,73,73,105,105,74,74,74,74,40,40,40,40,75,75,107,41,41,41,41,41,108,108,42,42,42,42,42,42,78,78,43,43,43,43,43,43,44,44,44,44,44,44,44,45,45,45,45,45,45,45,46,46,46,46,46,46,14,47,47,47,47]


def error_exit(s):
    print("Error: "+s)
    sys.exit(1)

def use_tool(toolname, args):
    if args != "":
        args=" "+args
    return os.system(toolname+' '+args)     
    
def read8bitwav(filename):
    # open .wav file and read its header structure
    with wave.open(filename, 'rb') as f:
    
        # get info on wave file
        num_frames = f.getnframes()
    
        if f.getsampwidth()!=1:
            error_exit("Sample width of .wav file must be 1 byte!");

        if f.getnchannels()!=1:
            error_exit(".wav file must be mono!");
        print("reading",filename,"with",num_frames,"samples")
        # read  and return frames
        return f.readframes(num_frames)    
        
def vector_quantize(samples,vector_length):
    # calculate length of cropped array
    new_length = int(len(samples) // vector_length) * vector_length
    A=np.frombuffer(samples[:new_length], dtype=np.uint8) 
    # reshape array into vectors of len vector_length
    reshaped_array=np.array(A).reshape(len(A)//vector_length, vector_length)
    # define KMeans model with 256 clusters
    kmeans = KMeans(n_clusters=256, init='k-means++', max_iter=300, n_init=10, random_state=0).fit(reshaped_array)    
    # get the codebook
    codebook = kmeans.cluster_centers_.astype(int)
    # use the codebook to vector quantize the array of 4000 values
    quantized_values = kmeans.predict(reshaped_array)                
    return codebook,quantized_values

parser = argparse.ArgumentParser() 
parser.add_argument("-f", "--samplefrequency", type=int, default = 7800, help="sample frequency") 
parser.add_argument("-c", "--compression", type=int, default = 2, help="compression factor") 
parser.add_argument("-r", "--review", type=int, nargs='?', const = 20, help="review quantization result for first values") 
parser.add_argument("-n", "--nopad", action='store_true', help="don't pad quantized values to next multiple of 256")
parser.add_argument("-t", "--table", type=str, default = "6581", help="select volume table for target: 6581 (=default), 8580)") 
parser.add_argument("-o", "--outfile", type=str, help="name of the output file") 
parser.add_argument("filename", help="enter filename") 
args = parser.parse_args() 

samplefrequency = args.samplefrequency 
vector_length = args.compression 
filename = args.filename
outfilename = args.outfile
if args.outfile:
    outfilename = args.outfile
else:
    outfilename = ".".join(filename.split(".")[:-1]) + ".digidata"
    print (outfilename)

if args.table=="6581":
    volume_table=volume_table_common_6581_256_of_256
elif args.table=="8580":
    volume_table=volume_table_common_8580_256_of_256
elif args.table.lower()=="vice":
    error_exit("Unknown selection for volume table!")

if filename is None:
    error_exit("Filename was not provided!")

newfilename=filename[:-4]+'.tmp.wav'
if os.path.exists(newfilename):
    os.remove(newfilename)
use_tool('ffmpeg','-i '+filename+'  -ar '+str(samplefrequency)+' -ac 1 -acodec pcm_u8 '+newfilename)
filename=newfilename
        
frames=read8bitwav(filename)
originalsize=len(frames)

n_quantized = (len(frames)+vector_length-1) // vector_length

padval=128
if args.nopad:
    target_n = vector_length * n_quantized
else:    
    target_n = vector_length * (math.ceil(n_quantized/256)*256 - 1) 
    #-1 because we need 2 more bytes, but we save the first quantized value which is always 0
    #so 1-2 = -1

if target_n>len(frames):
    print("padded to",target_n)
    frames=frames + bytearray([padval]*(target_n-len(frames)))


codebook,quantized_values=vector_quantize(frames,vector_length)

#make it so that the first quantized_value is a 0
new0=quantized_values[0]
if new0!=0:   
    #switch 0 <-> new0
    for i in range(len(quantized_values)):
        if quantized_values[i]==0:
            quantized_values[i]=new0
        elif quantized_values[i]==new0:
            quantized_values[i]=0
    #switch codebook entries
    codebook[[0,new0]] = codebook[[new0,0]]

if args.review:
    print("Reviewing quality of quantization")
    print("Original:  ",end="")
    comma=""
    for i in range(args.review):
        print(comma,str(frames[i]),end="")
        comma=","
    print()
    print("Compressed:",end="")    
    comma=""    
    for i in range(args.review):
        x=codebook[quantized_values[i // vector_length]][i % vector_length]
        print(comma,x,end="")
        comma=","
    print()
    e2=0
    for i in range(len(frames)):
        x=codebook[quantized_values[i // vector_length]][i % vector_length]
        e2 += (x - frames[i])**2
    print("Mean squared error over all samples:",e2/len(frames))

#rewrite codebook to suit nonlinear values of C64 $d418 register

for i in range(len(codebook)):
    for j in range(len(codebook[i])):
        codebook[i][j] = volume_table[codebook[i][j]]

#writing results to file

pages = (len(quantized_values)+255) // 256
if pages > 192:
    error_exit("file too large!")
with open(outfilename, 'wb') as f:
    f.write(vector_length.to_bytes(1, byteorder='big'))
    f.write(np.transpose(codebook).astype("uint8").tobytes()[1:])   #all codebook values but the first one
    f.write(pages.to_bytes(1, byteorder='big'))    
    f.write(codebook[0][0].astype("uint8").tobytes())           #add first codebook value here
    f.write(quantized_values.astype("uint8")[1:])
print("written",256*vector_length,"codebook bytes and",len(quantized_values),"quantized values to output file",outfilename)
outputlen=256*vector_length+len(quantized_values)
print("compression",originalsize,"to",outputlen,"(down by",str(round(100-100*outputlen/originalsize))+"%)")

