"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _chalk = _interopRequireDefault(require("chalk"));
var _os = _interopRequireDefault(require("os"));
var _settings = require("../../../api/settings");
var _signManifest = require("../../../api/signManifest");
var _userSettings = _interopRequireDefault(require("../../../api/user/UserSettings"));
var _user = require("../../../api/user/user");
var Log = _interopRequireWildcard(require("../../../log"));
var _rudderstackClient = require("../../../utils/analytics/rudderstackClient");
var _fn = require("../../../utils/fn");
var _link = require("../../../utils/link");
var _url = require("../../../utils/url");
var _manifestMiddleware = require("./ManifestMiddleware");
var _resolvePlatform = require("./resolvePlatform");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
const debug = require("debug")("expo:start:server:middleware:ClassicManifestMiddleware");
class ClassicManifestMiddleware extends _manifestMiddleware.ManifestMiddleware {
    getParsedHeaders(req) {
        const platform = (0, _resolvePlatform).parsePlatformHeader(req) || "ios";
        (0, _resolvePlatform).assertRuntimePlatform(platform);
        return {
            platform,
            acceptSignature: Boolean(req.headers["exponent-accept-signature"]),
            hostname: (0, _url).stripPort(req.headers["host"])
        };
    }
    async _getManifestResponseAsync({ acceptSignature , ...requestOptions }) {
        const { exp , hostUri , expoGoConfig , bundleUrl  } = await this._resolveProjectSettingsAsync(requestOptions);
        const manifest = {
            ...exp,
            ...expoGoConfig,
            hostUri,
            bundleUrl
        };
        // Gather packager and host info
        const hostInfo = await createHostInfoAsync();
        const headers = new Map();
        headers.set("Exponent-Server", JSON.stringify(hostInfo));
        // Create the final string
        const body = await this._fetchComputedManifestStringAsync({
            manifest,
            hostId: hostInfo.host,
            acceptSignature
        });
        return {
            body,
            version: manifest.sdkVersion,
            headers
        };
    }
    trackManifest(version) {
        // Log analytics
        (0, _rudderstackClient).logEventAsync("Serve Manifest", {
            sdkVersion: version != null ? version : null
        });
    }
    /** Exposed for testing. */ async _getManifestStringAsync({ manifest , hostId , acceptSignature  }) {
        const currentSession = await (0, _user).getUserAsync();
        if (!currentSession || _settings.APISettings.isOffline) {
            manifest.id = `@${_user.ANONYMOUS_USERNAME}/${manifest.slug}-${hostId}`;
        }
        if (!acceptSignature) {
            return JSON.stringify(manifest);
        } else if (!currentSession || _settings.APISettings.isOffline) {
            return getUnsignedManifestString(manifest);
        } else {
            return this.getSignedManifestStringAsync(manifest);
        }
    }
    getSignedManifestStringAsync = (0, _fn).memoize(_signManifest.signClassicExpoGoManifestAsync);
    /** Exposed for testing. */ async _fetchComputedManifestStringAsync(props) {
        try {
            return await this._getManifestStringAsync(props);
        } catch (error) {
            debug(`Error getting manifest:`, error);
            if (error.code === "UNAUTHORIZED" && props.manifest.owner) {
                // Don't have permissions for siging, warn and enable offline mode.
                this.addSigningDisabledWarning(`This project belongs to ${_chalk.default.bold(`@${props.manifest.owner}`)} and you have not been granted the appropriate permissions.\n` + `Please request access from an admin of @${props.manifest.owner} or change the "owner" field to an account you belong to.\n` + (0, _link).learnMore("https://docs.expo.dev/versions/latest/config/app/#owner"));
                _settings.APISettings.isOffline = true;
                return await this._getManifestStringAsync(props);
            } else if (error.code === "ENOTFOUND") {
                // Got a DNS error, i.e. can't access exp.host, warn and enable offline mode.
                this.addSigningDisabledWarning(`Could not reach Expo servers, please check if you can access ${error.hostname || "exp.host"}.`);
                _settings.APISettings.isOffline = true;
                return await this._getManifestStringAsync(props);
            } else {
                throw error;
            }
        }
    }
    addSigningDisabledWarning = (0, _fn).memoize((reason)=>{
        Log.warn(`${reason}\nFalling back to offline mode.`);
        // For the memo
        return reason;
    });
}
exports.ClassicManifestMiddleware = ClassicManifestMiddleware;
// Passed to Expo Go and registered as telemetry.
// TODO: it's unclear why we don't just send it from the CLI.
async function createHostInfoAsync() {
    return {
        host: await _userSettings.default.getAnonymousIdentifierAsync(),
        server: "expo",
        // Defined in the build step
        serverVersion: "0.6.2",
        serverDriver: _manifestMiddleware.DEVELOPER_TOOL,
        serverOS: _os.default.platform(),
        serverOSVersion: _os.default.release()
    };
}
function getUnsignedManifestString(manifest) {
    const unsignedManifest = {
        manifestString: JSON.stringify(manifest),
        signature: "UNSIGNED"
    };
    return JSON.stringify(unsignedManifest);
}

//# sourceMappingURL=ClassicManifestMiddleware.js.map