//***************************************************************************
// "title_card.c"
// Code for title card screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright 2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdio.h>
#include "scene.h"
#include "video.h"

// Where the sprites are stored
#define MAX_CHARS 0x80
static GraphicsSet *gfxset_chars = NULL;
static Sprite *spr_chars[0x80];
static Sprite *spr_red_rect;
static Sprite *spr_blue_rect;

//***************************************************************************
// load_title_card
// Loads the title card graphics.
//***************************************************************************

void load_title_card(void) {
   // Load graphics set
   gfxset_chars = load_graphics_set("graphics/title_card");

   // Reset charset
   for (unsigned i = 0; i < MAX_CHARS; i++)
      spr_chars[i] = NULL;

   // Get letter sprites
   for (unsigned i = 'a'; i < 'z'; i++) {
      char buffer[0x10];
      sprintf(buffer, "char_%c", i);
      spr_chars[i] = get_sprite(gfxset_chars, buffer);
      spr_chars[i-0x20] = spr_chars[i];
   }

   // Get number sprites
   for (unsigned i = '0'; i < '9'; i++) {
      char buffer[0x10];
      sprintf(buffer, "char_%c", i);
      spr_chars[i] = get_sprite(gfxset_chars, buffer);
   }

   // Get symbol sprites
   spr_chars['.'] = get_sprite(gfxset_chars, "char_point");
   spr_chars['-'] = get_sprite(gfxset_chars, "char_minus");

   // Get rest of the sprites
   spr_red_rect = get_sprite(gfxset_chars, "rectangle_red");
   spr_blue_rect = get_sprite(gfxset_chars, "rectangle_blue");
}

//***************************************************************************
// draw_title_card
// Draws the title card.
//---------------------------------------------------------------------------
// param offset: used to animate the title card
//***************************************************************************

void draw_title_card(unsigned offset) {
   // Draw rectangles
   draw_sprite(spr_red_rect, screen_w - 128, -(offset << 4), 0);
   draw_sprite(spr_blue_rect, screen_w - 312 - (offset << 4), screen_cy, 0);

   // Get name of the level
   const char *name = get_scene_name();

   // Calculate start position
   int32_t x = screen_w - 0x10 + (offset << 4);
   for (const char *ptr = name; *ptr != '\0'; ptr++) {
      // Valid character?
      int ch = *ptr;
      if (ch & 0x80) ch = ' ';
      if (spr_chars[ch] == NULL) ch = ' ';
      if (ch == ' ') {
         x -= 8;
         continue;
      }

      // Count pixels
      x -= spr_chars[ch]->width;
   }

   // Now draw name
   for (; *name != '\0'; name++) {
      // Valid character?
      int ch = *name;
      if (ch & 0x80) ch = ' ';
      if (spr_chars[ch] == NULL) ch = ' ';
      if (ch == ' ') {
         x += 8;
         continue;
      }

      // Draw character
      draw_sprite(spr_chars[ch], x, screen_cy - 0x10, 0);
      x += spr_chars[ch]->width;
   }
}

//***************************************************************************
// unload_title_card
// Unloads the title card graphics.
//***************************************************************************

void unload_title_card(void) {
   // Get rid of graphics set
   destroy_graphics_set(gfxset_chars);
}
