//***************************************************************************
// "tally.c"
// Code for handling the score tally at the end of a level.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <stdio.h>
#include "editor.h"
#include "ingame.h"
#include "scene.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// Text positions
#define TEXT_TX (screen_cx)            // Level complete
#define TEXT_X1 (screen_cx - 96)       // Left column
#define TEXT_X2 (screen_cx + 96)       // Right column

#define TEXT_TY (screen_cy - 0x30)     // Level complete
#define TEXT_Y1 (screen_cy - 0x10)     // Enemy bonus
#define TEXT_Y2 (screen_cy + 0x00)     // Item bonus
#define TEXT_Y3 (screen_cy + 0x10)     // Special bonus
#define TEXT_YT (screen_cy + 0x30)     // Total
#define TEXT_YT2 (screen_cy + 0x40)    // Time spent

// Used to know when we're tallying
static unsigned tallying;

// Used to keep track of the score
static struct {
   uint64_t enemy;
   uint64_t items;
   uint64_t special;
   uint64_t total;
} bonus;
static struct {
   unsigned enemy;
   unsigned items;
   unsigned special;
} combo;

// Used to keep track of the time spent in the level
// If it's UINT32_MAX, this is diasbled
static uint32_t level_time;

// Used to animate the lines when they enter
// Indicates the shift to the right in pixels
#define SHIFT_BASE 0x120
#define SHIFT_STEP 0x10
static struct {
   int32_t title;
   int32_t enemy;
   int32_t items;
   int32_t special;
   int32_t total;
   int32_t time;
} shift;

// How long to delay at the beginning
static unsigned start_timer;
static unsigned end_timer;

// Private function prototypes
static uint64_t inflate_bonus(uint64_t, unsigned);
static void count_bonus(uint64_t *);

//***************************************************************************
// reset_tally
// Resets the status of the score tally, including all the bonus.
//***************************************************************************

void reset_tally(void) {
   // Reset bonuses
   bonus.enemy = 0;
   bonus.items = 0;
   bonus.special = 0;
   combo.enemy = 0;
   combo.items = 0;
   combo.special = 0;

   // Reset level time
   level_time = 0;

   // Not tallying yet
   tallying = 0;
}

//***************************************************************************
// disable_level_time
// Disables counting how long it took to beat the level.
//***************************************************************************

void disable_level_time(void) {
   level_time = UINT32_MAX;
}

//***************************************************************************
// update_level_time
// Increments the time it's taking to beat the level, assuming we're still
// keeping track of that (otherwise it won't do anything).
//***************************************************************************

void update_level_time(void) {
   // Stop counting once we beat the level!
   if (tallying)
      return;

   // Update time spent if we're keeping track of it
   if (level_time != UINT32_MAX)
      level_time++;
}

//***************************************************************************
// draw_level_time
// Draws the time currently spent in the level. Used in debug mode.
//***************************************************************************

void draw_level_time(void) {
   // Keeping track of it?
   if (level_time != UINT32_MAX) {
      char buffer[0x20];
      sprintf(buffer, "%u:%02u.%02u",
         level_time / 3600,
         level_time / 60 % 60,
         level_time % 60 * 100 / 60);
      draw_text(buffer, screen_w - 0x10, screen_h - 0x08,
         FONT_LIT, ALIGN_BOTTOMRIGHT);
   }

   // Not keeping track of it?
   else {
      draw_text("-:--.--", screen_w - 0x10, screen_h - 0x08,
         FONT_DIM, ALIGN_BOTTOMRIGHT);
   }
}

//***************************************************************************
// start_tally
// Starts the score tally when the level is finished.
//***************************************************************************

void start_tally(void) {
   // Start playing the tally BGM
   play_bgm(BGM_FINISH);

   // Reset timers
   start_timer = 120;
   end_timer = 180;

   // Inflate bonuses
   bonus.enemy = inflate_bonus(bonus.enemy, (combo.enemy+1)/2) * 10;
   bonus.items = inflate_bonus(bonus.items, combo.items) * 100;
   bonus.special = inflate_bonus(bonus.special, combo.special) * 250;

   // Reset total
   bonus.total = 0;

   // Reset the shifting
   shift.title = SHIFT_BASE;
   shift.enemy = SHIFT_BASE + 0x20;
   shift.items = SHIFT_BASE + 0x30;
   shift.special = SHIFT_BASE + 0x40;
   shift.total = SHIFT_BASE + 0x60;
   shift.time = SHIFT_BASE + 0x70;

   // Start tallying now!
   tallying = 1;
}

//***************************************************************************
// inflate_bonus [internal]
// Inflates the value of a bonus based on how many times it was obtained.
//***************************************************************************

static uint64_t inflate_bonus(uint64_t count, unsigned combo) {
   return count * (combo * (combo + 1) / 2);
}

//***************************************************************************
// run_tally
// Performs the actual tallying.
//***************************************************************************

void run_tally(void) {
   // Do nothing if not tallying yet
   if (!tallying)
      return;

   // Stop if we're quitting
   if (end_timer == 0)
      return;

   // Reduce shifts if needed
   if (shift.title > 0)    shift.title -= SHIFT_STEP;
   if (shift.enemy > 0)    shift.enemy -= SHIFT_STEP;
   if (shift.items > 0)    shift.items -= SHIFT_STEP;
   if (shift.special > 0)  shift.special -= SHIFT_STEP;
   if (shift.total > 0)    shift.total -= SHIFT_STEP;
   if (shift.time > 0)     shift.time -= SHIFT_STEP;

   // Wait at the beginning
   if (start_timer > 0) {
      start_timer--;
      return;
   }

   // Add enemy bonus if needed
   if (bonus.enemy || bonus.items || bonus.special) {
      count_bonus(&bonus.enemy);
      count_bonus(&bonus.items);
      count_bonus(&bonus.special);
      return;
   }

   // Wait some more at the end
   end_timer--;

   // Quit level
   if (end_timer == 0) {
      if (get_editor_map() != NULL)
         fade_off_and_switch(GAMEMODE_EDITOR);
      else
         switch_to_next_scene();
   }
}

//***************************************************************************
// count_bonus [internal]
// Tallies some points from the given bonus and adds them to the total.
//***************************************************************************

static void count_bonus(uint64_t *what) {
   // Calculate how much to substract
   uint64_t step = (*what >> 6) + 80 + get_random(40);
   if (step > *what) step = *what;

   // Tally those points
   *what -= step;
   bonus.total += step;
}

//***************************************************************************
// draw_tally
// Draws the score tally.
//***************************************************************************

void draw_tally(void) {
   // Draw nothing if not tallying yet
   if (!tallying)
      return;

   // Draw title
   draw_text(text.tally.title,
      TEXT_TX + shift.title, TEXT_TY, FONT_LIT, ALIGN_CENTER);

   // Draw labels
   draw_text(text.tally.enemy,
      TEXT_X1 + shift.enemy, TEXT_Y1, FONT_LIT, ALIGN_LEFT);
   draw_text(text.tally.items,
      TEXT_X1 + shift.items, TEXT_Y2, FONT_LIT, ALIGN_LEFT);
   draw_text(text.tally.special,
      TEXT_X1 + shift.special, TEXT_Y3, FONT_LIT, ALIGN_LEFT);
   draw_text(text.tally.total,
      TEXT_X1 + shift.total, TEXT_YT, FONT_LIT, ALIGN_LEFT);

   // Draw scores
   draw_text_int("{param}", bonus.enemy,
      TEXT_X2 + shift.enemy, TEXT_Y1, FONT_LIT, ALIGN_RIGHT);
   draw_text_int("{param}", bonus.items,
      TEXT_X2 + shift.items, TEXT_Y2, FONT_LIT, ALIGN_RIGHT);
   draw_text_int("{param}", bonus.special,
      TEXT_X2 + shift.special, TEXT_Y3, FONT_LIT, ALIGN_RIGHT);
   draw_text_int("{param}", bonus.total,
      TEXT_X2 + shift.total, TEXT_YT, FONT_LIT, ALIGN_RIGHT);

   // Draw time spent, if counted
   if (level_time != UINT32_MAX) {
      // Render how much time was spent beating the level
      char buffer[0x20];
      sprintf(buffer, "%u:%02u.%02u",
         level_time / 3600,
         level_time / 60 % 60,
         level_time % 60 * 100 / 60);

      // Draw text now
      draw_text(text.tally.time, TEXT_X1 + shift.time,
         TEXT_YT2, FONT_LIT, ALIGN_LEFT);
      draw_text(buffer, TEXT_X2 + shift.time,
         TEXT_YT2, FONT_LIT, ALIGN_RIGHT);
   }
}

//***************************************************************************
// add_enemy_bonus
// Increments the enemy bonus
//---------------------------------------------------------------------------
// param value: value of enemy
//***************************************************************************

void add_enemy_bonus(unsigned value) {
   bonus.enemy += value;
   combo.enemy++;
}

//***************************************************************************
// add_item_bonus
// Increments the item bonus
//---------------------------------------------------------------------------
// param value: value of item
//***************************************************************************

void add_item_bonus(unsigned value) {
   bonus.items += value;
   combo.items++;
}


//***************************************************************************
// add_special_bonus
// Increments the special bonus
//***************************************************************************

void add_special_bonus(void) {
   bonus.special++;
   combo.special++;
}
