//***************************************************************************
// "sound.c"
// Sound playback subsystem.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <errno.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <SDL2/SDL.h>
#include <physfs.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#include "main.h"
#include "audiovideo.h"
#include "file.h"
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "recording.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"
#include "video.h"

// Parameters for tweaking how audio stream processing is done. Changing
// these values will affect playback performance.
#define STREAM_FREQ     44100          // Playback sample rate
#define STREAM_LEN      (0x400*2)      // Samples processed each time
#define DECBUFSIZE      0x1000         // Size of decoding buffer (in bytes)

// Where the BGM data is stored
// Hint: we load the OGG files as-is in memory
// We get vorbisfile to decode them on the fly
typedef struct {
   uint8_t *data;          // OGG file data
   size_t size;            // OGG file size
} BGMData;
static BGMData bgm_data[NUM_BGM] = {
   { NULL, 0 }
};

// Current status of BGM playback
static struct {
   // Some general data
   BgmID id;                        // ID of BGM playing
   unsigned playing: 1;             // Set if playing
   unsigned restart: 1;             // Set if file should restart
   unsigned paused: 1;              // Set if playing but paused
   unsigned stereo: 1;              // Set if stereo, clear if mono
   uint32_t rate;                   // Sample rate (0x10000 = 44100Hz)

   // OGG stream info
   struct {
      uint8_t *data;                // Where OGG stream is stored
      size_t size;                  // Size of OGG stream
      size_t pos;                   // Position of OGG stream
   } ogg;

   // Decoder info
   struct {
      OggVorbis_File file;          // The "file" used by vorbisfile
      uint8_t buffer[DECBUFSIZE];   // Decoding buffer
      size_t buflen;                // Size of decoded buffer in SAMPLES
      size_t bufpos;                // Position of decoded buffer in SAMPLES
      uint32_t advance;             // How much to advance pointer (16.16)
   } vorbis;
} bgm_state;

// Where the sound effects' data is stored
// I should try compressing it or something... (ADPCM?)
typedef struct {
   int16_t *data;          // Sample data itself
   size_t length;          // Length in samples
   uint32_t rate;          // Sample rate (0x10000 = 44100Hz)
   unsigned stereo: 1;     // Set if stereo, clear if mono
} SFXData;
static SFXData sfx_data[NUM_SFX] = {
   { NULL, 0, 0, 0 }
};

// Data used to keep track of sound effects
#define NUM_CHAN 0x100
typedef struct {
   SFXData *sfx;           // Sound effect being played (NULL if none)
   uint64_t speed;         // Playbak speed (48.16)
   uint64_t where_l;       // Which sample to play next (48.16)
   uint64_t where_r;          // One value for each speaker
   uint64_t delay_l;       // How many samples to delay playback
   uint64_t delay_r;          // One value for each speaker
   uint16_t vol_l;         // Volume for left speaker (0..0x100)
   uint16_t vol_r;         // Volume for right speaker (0..0x100)
   unsigned active_l: 1;   // Set if left speaker is active
   unsigned active_r: 1;   // Set if right speaker is active
} Channel;
static Channel channels[NUM_CHAN];
static size_t next_chan = 0;

// Names for each BGM
static const char *bgm_names[NUM_BGM] = {
   "virtual",
   "park",
   "sewer",
   "harbor",
   "desert",
   "cave",
   "factory",
   "carnival",

   "boss",
   "final_boss",

   "cutscene",
   "cutscene_2",
   "mrevil",

   "title",
   "menu",
   "ending",
   "invincibility",
   "finish",
   "trailer"
};

// Some information about each sound effect
static const struct {
   SfxGroupID group;
   const char *name;
} const sfx_info[] = {
   { SFXGROUP_PLAYER, "jump" },
   { SFXGROUP_PLAYER, "fall" },
   { SFXGROUP_PLAYER, "skid" },
   { SFXGROUP_PLAYER, "hurt" },
   { SFXGROUP_PLAYER, "fly" },
   { SFXGROUP_PLAYER, "hammer" },
   { SFXGROUP_PLAYER, "footstep" },
   { SFXGROUP_PLAYER, "land" },
   { SFXGROUP_PLAYER, "wall" },
   { SFXGROUP_PLAYER, "ceiling" },
   { SFXGROUP_PLAYER, "pinball" },
   { SFXGROUP_NOISE, "destroy" },
   { SFXGROUP_ALERT, "warning" },
   { SFXGROUP_ALERT, "warning_alt" },
   { SFXGROUP_NOISE, "robot_footstep" },
   { SFXGROUP_NOISE, "robot_climb_up" },
   { SFXGROUP_NOISE, "robot_climb_down" },
   { SFXGROUP_QUIET, "robot_drive" },
   { SFXGROUP_QUIET, "robot_roll" },
   { SFXGROUP_QUIET, "robot_idle" },
   { SFXGROUP_QUIET, "robot_heli" },
   { SFXGROUP_NOISE, "explosion" },
   { SFXGROUP_NOISE, "big_explosion" },
   { SFXGROUP_MISC, "crash" },
   { SFXGROUP_MISC, "crush" },
   { SFXGROUP_NOISE, "break" },
   { SFXGROUP_NOISE, "crack" },
   { SFXGROUP_ITEM, "spring" },
   { SFXGROUP_QUIET, "clue_spring" },
   { SFXGROUP_ITEM, "balloon" },
   { SFXGROUP_QUIET, "clue_balloon" },
   { SFXGROUP_ITEM, "heart" },
   { SFXGROUP_QUIET, "clue_heart" },
   { SFXGROUP_ITEM, "shield" },
   { SFXGROUP_QUIET, "clue_shield" },
   { SFXGROUP_QUIET, "clue_invincibility" },
   { SFXGROUP_ITEM, "powerup" },
   { SFXGROUP_QUIET, "clue_powerup" },
   { SFXGROUP_ITEM, "yupee" },
   { SFXGROUP_ITEM, "checkpoint" },
   { SFXGROUP_QUIET, "clue_checkpoint" },
   { SFXGROUP_ITEM, "goal" },
   { SFXGROUP_QUIET, "clue_goal" },
   { SFXGROUP_ITEM, "blue_star" },
   { SFXGROUP_QUIET, "clue_blue_star" },
   { SFXGROUP_ITEM, "switch" },
   { SFXGROUP_QUIET, "clue_switch" },
   { SFXGROUP_MISC, "drop" },
   { SFXGROUP_MISC, "shoot" },
   { SFXGROUP_MISC, "missile" },
   { SFXGROUP_QUIET, "fire" },
   { SFXGROUP_QUIET, "shock" },
   { SFXGROUP_QUIET, "clue_stalactite" },
   { SFXGROUP_QUIET, "clue_projectile" },
   { SFXGROUP_QUIET, "clue_bouncing" },
   { SFXGROUP_QUIET, "clue_liquid" },
   { SFXGROUP_QUIET, "clue_coil" },
   { SFXGROUP_QUIET, "clue_buzzsaw" },
   { SFXGROUP_NOISE, "static" },
   { SFXGROUP_UI, "sonar_left" },
   { SFXGROUP_UI, "sonar_right" },
   { SFXGROUP_UI, "sonar_up" },
   { SFXGROUP_UI, "sonar_down" },
   { SFXGROUP_UI, "beep" },
   { SFXGROUP_UI, "ok" },
   { SFXGROUP_UI, "cancel" },
   { SFXGROUP_UI, "speak_robot" },
   { SFXGROUP_UI, "speak_sol" },
   { SFXGROUP_UI, "speak_ruby" },
   { SFXGROUP_UI, "speak_evil" },
   { SFXGROUP_UI, "logo" },
   { SFXGROUP_UI, "cheat" }
};

// Private function prototypes
static void render_channels(int32_t *, int32_t *);
static void open_bgm(void);
void decode_bgm(int32_t *, int32_t *);
static void load_sfx(const char *, SFXData *);
static size_t read_callback_bgm(void *, size_t, size_t, void *);
static size_t read_callback_sfx(void *, size_t, size_t, void *);
static void pan_channel(Channel *, int32_t, int32_t, uint16_t);

//***************************************************************************
// init_sound
// Initializes the sound playback subsystem.
//***************************************************************************

void init_sound(void) {
   // Whoops, the way these are defined result in a compiler warning
   // triggering because they're unused (we use our own callbacks so we
   // don't really need them).
   (void) OV_CALLBACKS_DEFAULT;
   (void) OV_CALLBACKS_NOCLOSE;
   (void) OV_CALLBACKS_STREAMONLY;
   (void) OV_CALLBACKS_STREAMONLY_NOCLOSE;

   // Audio subsystem disabled on request?
   if (settings.no_sound)
      return;

#ifdef DEBUG
   fputs("Initializing sound\n", stderr);
#endif

   // Initialize the audio subsystem
   if (SDL_InitSubSystem(SDL_INIT_AUDIO)) {
      settings.no_sound = 1;
      return;
   }

   // Set up the properties for the audio stream we want
   SDL_AudioSpec spec;
   spec.freq = STREAM_FREQ;
   spec.format = AUDIO_U16LSB;
   spec.channels = 2;
   spec.samples = STREAM_LEN;
   spec.callback = sound_render_callback;
   spec.userdata = NULL;

   // Open SDL audio device
   if (SDL_OpenAudio(&spec, NULL)) {
      settings.no_sound = 1;
      return;
   }

   // Reset BGM
   SDL_LockAudio();
   bgm_state.playing = 0;

   // Reset all sound channels
   next_chan = 0;
   for (size_t i = 0; i < NUM_CHAN; i++)
      channels[i].sfx = NULL;
   SDL_UnlockAudio();

   // Start playing sound
   SDL_PauseAudio(0);

   // Load all sound effects
   for (SfxID i = 0; i < NUM_SFX; i++) {
      // Get filename
      char filename[0x80];
      sprintf(filename, "sounds/%s.ogg", sfx_info[i].name);

      // Load sound effect from file if possible
      // Note that if this fails it's going to be ignored, the game will just
      // not play this sound effect when requested in such case
      load_sfx(filename, &sfx_data[i]);
   }
}

//***************************************************************************
// sound_render_callback
// Callback called by SDL to render the audio stream. Here is where all the
// mixing happens.
//---------------------------------------------------------------------------
// param dummy: because SDL wants this...
// param out: pointer to output buffer
// param outlen: size of output buffer in bytes
//***************************************************************************

void sound_render_callback(void *dummy, uint8_t *out, int outlen) {
   // When I find an use for this parameter I'll rename it, remove this line
   // and make something productive. Meanwhile, this will stick here around
   // because SDL needs the parameter and the compiler complains if I don't
   // use the variable.
   (void) dummy;

   // Convert buffer length so it's measured in samples instead
   // There are four bytes per sample (stereo, 16-bit per channel)
   outlen >>= 2;

   // Process all the samples needed to fill the buffer
   // Yes, we gave a specific length to SDL, but it can allocate a different
   // amount so we better go by what the parameter says (in fact, as I write
   // this the documentation seems to not be clear enough and I had to add a
   // *2 to STREAM_LEN to account for stereo, so I better not trust it...)
   for (int i = 0; i < outlen; i++) {
      // Where we'll store the output samples as we're processing
      // Using larger integer types so it can overflow without breaking.
      // We'll clamp the result after we mixed all the sounds. And no, we
      // can't clamp as we mix, the outcome would be wrong if we tried that
      // (not to mention it'd be slower).
      int32_t sample_l = 0;      // Left speaker
      int32_t sample_r = 0;      // Right speaker

      // Play sound channels
      render_channels(&sample_l, &sample_r);

      // Play BGM if needed
      if (bgm_state.playing)
         decode_bgm(&sample_l, &sample_r);

      // Downsample to mono?
      if (!settings.stereo) {
         int32_t mono = (sample_l * 707 + sample_r * 707) / 1000;
         sample_l = mono;
         sample_r = mono;
      }

      // Make output samples to be in the 0x0000..0xFFFF range
      sample_l += 0x8000;
      sample_r += 0x8000;

      // Clamp the output to 16-bit
      if (sample_l < 0x0000) sample_l = 0x0000;
      if (sample_l > 0xFFFF) sample_l = 0xFFFF;
      if (sample_r < 0x0000) sample_r = 0x0000;
      if (sample_r > 0xFFFF) sample_r = 0xFFFF;

      // Write output samples to the buffer
      if (!settings.reverse) {
         *out++ = sample_l;
         *out++ = sample_l >> 8;
         *out++ = sample_r;
         *out++ = sample_r >> 8;
      } else {
         *out++ = sample_r;
         *out++ = sample_r >> 8;
         *out++ = sample_l;
         *out++ = sample_l >> 8;
      }
   }
}

//***************************************************************************
// render_channels [internal]
// Updates all the sound channels.
//---------------------------------------------------------------------------
// param output_l: left speaker output
// param output_r: right speaker output
//***************************************************************************

static void render_channels(int32_t *output_l, int32_t *output_r) {
   // Where the channels get mixed
   int32_t mix_l = 0;
   int32_t mix_r = 0;

   // Process every channel
   for (unsigned i = 0; i < NUM_CHAN; i++) {
      Channel *chan = &channels[i];
      SFXData *sfx = chan->sfx;
      if (sfx == NULL)
         continue;

      // Retrieve samples
      int32_t sample_l;
      int32_t sample_r;

      if (!sfx->stereo) {
         sample_l = chan->active_l ?
            sfx->data[chan->where_l >> 16] : 0;
         sample_r = chan->active_r ?
            sfx->data[chan->where_r >> 16] : 0;
      } else {
         sample_l = chan->active_l ?
            sfx->data[chan->where_l >> 16 << 1] : 0;
         sample_r = chan->active_r ?
            sfx->data[(chan->where_r >> 16 << 1) + 1] : 0;
      }

      // Update playback
      chan->where_l += !chan->delay_l ? chan->speed : 0;
      chan->where_r += !chan->delay_r ? chan->speed : 0;
      chan->delay_l -= !chan->delay_l ? 0 : 1;
      chan->delay_r -= !chan->delay_r ? 0 : 1;

      if (chan->where_l >> 16 >= sfx->length)
         chan->active_l = 0;
      if (chan->where_r >> 16 >= sfx->length)
         chan->active_r = 0;
      if (!chan->active_l && !chan->active_r)
         chan->sfx = NULL;

      // Apply volume
      sample_l = (sample_l * chan->vol_l) >> 8;
      sample_r = (sample_r * chan->vol_r) >> 8;

      // Mix in sample
      mix_l += sample_l;
      mix_r += sample_r;
   }

   // Mix in the channels with everything else
   *output_l += mix_l * settings.sfx_volume / 100;
   *output_r += mix_r * settings.sfx_volume / 100;
}

//***************************************************************************
// deinit_sound
// Deinitializes the sound playback subsystem.
//***************************************************************************

void deinit_sound(void) {
   // Ensure the callback isn't running
   // Yes, this can cause race conditions if not done
   SDL_LockAudio();
   SDL_PauseAudio(1);
   SDL_UnlockAudio();

   // Deinitialize sound system
   SDL_QuitSubSystem(SDL_INIT_AUDIO);

   // Unload background music
   unload_all_bgm();

   // Unload sound effects
   for (SfxID i = 0; i < NUM_SFX; i++) {
      if (sfx_data[i].data != NULL) {
         free(sfx_data[i].data);
         sfx_data[i].data = NULL;
         sfx_data[i].length = 0;
         sfx_data[i].rate = 0;
      }
   }
}

//***************************************************************************
// load_bgm
// Loads a BGM into memory.
//---------------------------------------------------------------------------
// param id: ID of BGM to load
//***************************************************************************

void load_bgm(BgmID id) {
   // *ahem*
   if (settings.no_sound || settings.beeper)
      return;
   if (id == BGM_NONE)
      return;

   // Demo version lacks the final boss theme, so we have this quick hack
   // in place... Ugly but hey :P
   if (settings.demo && id == BGM_FINALBOSS)
      id = BGM_BOSS;

   // Is BGM already loaded?
   if (bgm_data[id].data != NULL)
      return;

   // Try to open file
   char filename[0x80];
   sprintf(filename, "music/%s.ogg", bgm_names[id]);
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      return;

   // Keep reading data until we reach the end
   uint8_t *buffer = NULL;
   size_t bufpos = 0;
   size_t bufsize = 0;
   size_t total = 0;
   for (;;) {
      // Allocate more memory
      bufsize += 0x1000;
      buffer = realloc(buffer, bufsize);
      if (buffer == NULL)
         abort_program(ERR_NOMEMORY, NULL);

      // Read more data into memory
      size_t readsize = read_file_ex(file, &buffer[bufpos], 0x1000);
      total += readsize;
      if (end_of_file(file))
         break;
      if (readsize != 0x1000) {
         free(buffer);
         close_file(file);
         return;
      }

      // Advance pointer
      bufpos += 0x1000;
   }

   // Store data in the BGM structure
   bgm_data[id].data = buffer;
   bgm_data[id].size = total;

   // Done!
   close_file(file);
}

//***************************************************************************
// play_bgm
// Starts playback of the specified BGM. If the BGM isn't available, then
// nothing plays instead (behaves like stop_bgm).
//---------------------------------------------------------------------------
// param id: ID of BGM to play
//***************************************************************************

void play_bgm(BgmID id) {
   // Stop previously playing BGM if needed
   stop_bgm();

   // *ahem*
   if (id == BGM_NONE)
      return;

   // Demo version lacks the final boss theme, so we have this quick hack
   // in place... Ugly but hey :P
   if (settings.demo && id == BGM_FINALBOSS)
      id = BGM_BOSS;

   // Because we'll clash with the sound renderer...
   SDL_LockAudio();

   // Check if BGM is even available for starters
   if (bgm_data[id].data == NULL) {
      SDL_UnlockAudio();
      return;
   }

   // Store BGM information
   bgm_state.id = id;
   bgm_state.playing = 1;
   bgm_state.restart = 1;
   bgm_state.paused = 0;

   // Done with the sound stuff
   SDL_UnlockAudio();

   // Add BGM to the movie if we're recording
   if (is_recording())
      record_bgm(id);
}

//***************************************************************************
// pause_bgm
// Pauses the BGM currently playing.
//***************************************************************************

void pause_bgm(void) {
   SDL_LockAudio();
   bgm_state.paused = 1;
   SDL_UnlockAudio();
}

//***************************************************************************
// resume_bgm
// Unpauses the BGM that was playing.
//***************************************************************************

void resume_bgm(void) {
   SDL_LockAudio();
   bgm_state.paused = 0;
   SDL_UnlockAudio();
}

//***************************************************************************
// stop_bgm
// Stops BGM playback. Does nothing if BGM isn't playing.
//***************************************************************************

void stop_bgm(void) {
   // Mark BGM as not playing
   SDL_LockAudio();
   if (bgm_state.playing) {
      ov_clear(&bgm_state.vorbis.file);
      bgm_state.playing = 0;
   }
   SDL_UnlockAudio();
}

//***************************************************************************
// get_current_bgm
// Retrieves which BGM is currently playing.
//---------------------------------------------------------------------------
// return: playing BGM, or BGM_NONE if not playing anything
//***************************************************************************

BgmID get_current_bgm(void) {
   return bgm_state.playing ? bgm_state.id : BGM_NONE;
}

//***************************************************************************
// unload_bgm
// Unloads a BGM from memory. Does nothing if not loaded.
//---------------------------------------------------------------------------
// param id: ID of BGM to unload
//***************************************************************************

void unload_bgm(BgmID id) {
   // Demo version lacks the final boss theme, so we have this quick hack
   // in place... Ugly but hey :P
   if (settings.demo && id == BGM_FINALBOSS)
      id = BGM_BOSS;

   // Stop BGM if it's the one being currently played!
   if (id == bgm_state.id)
      stop_bgm();

   // Check if BGM is loaded
   if (bgm_data[id].data != NULL) {
      // Unload data
      free(bgm_data[id].data);

      // Reset BGM info
      bgm_data[id].data = NULL;
      bgm_data[id].size = 0;
   }
}

//***************************************************************************
// unload_all_bgm
// Unloads *all* BGM from memory.
//***************************************************************************

void unload_all_bgm(void) {
   // Just unload all BGMs
   for (BgmID i = 0; i < NUM_BGM; i++)
      unload_bgm(i);
}

//***************************************************************************
// open_bgm [internal]
// Opens the "file" used by vorbisfile to decode the BGM as it plays.
//***************************************************************************

static void open_bgm(void) {
   // Make sure BGM is playing
   if (!bgm_state.playing)
      return;

   // Reset OGG stream
   BgmID id = bgm_state.id;
   bgm_state.ogg.data = bgm_data[id].data;
   bgm_state.ogg.size = bgm_data[id].size;
   bgm_state.ogg.pos = 0;

   // Set up the functions we want vorbisfile to use
   // ...in other words, the read function :P
   ov_callbacks callbacks;
   callbacks.read_func = read_callback_bgm;
   callbacks.seek_func = NULL;
   callbacks.tell_func = NULL;
   callbacks.close_func = NULL;

   // Open the file
   if (ov_open_callbacks(bgm_state.ogg.data, &bgm_state.vorbis.file,
   NULL, 0, callbacks)) {
      bgm_state.playing = 0;
#ifdef DEBUG
      fputs("Warning: couldn't start BGM!\n", stderr);
#endif
      return;
   }

   // Get file information
   vorbis_info *info = ov_info(&bgm_state.vorbis.file, -1);
   if (info == NULL) {
      bgm_state.playing = 0;
      ov_clear(&bgm_state.vorbis.file);
      return;
   }

   // Check number of channels
   switch (info->channels) {
      // Mono
      case 1:
         bgm_state.stereo = 0;
         break;

      // Stereo
      case 2:
         bgm_state.stereo = 1;
         break;

      // We won't bother handling this...
      default:
         bgm_state.playing = 0;
         ov_clear(&bgm_state.vorbis.file);
         return;
   }

   // Get playback rate
   // <Sik> The check here is to prevent overflow, but... is it even possible
   // to have sample rates this high? (that's 16777215Hz...) The check is
   // still needed anyway because otherwise somebody could make a mod that
   // hangs the system on purpose and we really don't want that.
   if (info->rate > 0xFFFFFF) {
      bgm_state.playing = 0;
      ov_clear(&bgm_state.vorbis.file);
      return;
   }
   bgm_state.rate = (uint64_t)(info->rate) * 0x10000 / 44100;

   // Reset playback data
   bgm_state.vorbis.bufpos = 0;
   bgm_state.vorbis.buflen = 0;
   bgm_state.vorbis.advance = 0;

   // There, opened
   bgm_state.restart = 0;
}

//***************************************************************************
// decode_bgm [internal]
// Decodes a sample from the BGM and mixes it with the output.
//---------------------------------------------------------------------------
// param output_l: left speaker output
// param output_r: right speaker output
//***************************************************************************

void decode_bgm(int32_t *output_l, int32_t *output_r) {
   // BGM needs to restart?
   if (bgm_state.playing && bgm_state.restart)
      open_bgm();

   // BGM not playing? (don't mix anything)
   if (!bgm_state.playing || bgm_state.paused)
      return;

   // Advance pointer
   bgm_state.vorbis.advance += bgm_state.rate;
   size_t delta = bgm_state.vorbis.advance >> 16;
   bgm_state.vorbis.advance &= 0xFFFF;
   bgm_state.vorbis.bufpos += delta;

   // We need to decode more?
   while (bgm_state.vorbis.bufpos >= bgm_state.vorbis.buflen) {
      // Reset buffer pointer
      bgm_state.vorbis.bufpos -= bgm_state.vorbis.buflen;

      // Fetch more samples
      int bitstream = 0;
      long total = ov_read(&bgm_state.vorbis.file,
         (char*) &bgm_state.vorbis.buffer,
         sizeof(bgm_state.vorbis.buffer),
         1, 2, 0, &bitstream);

      // Yikes...
      if (total < 0) {
         bgm_state.playing = 0;
         return;
      }

      // End of BGM? (loop it! - if it should, that is)
      if (total == 0 || bitstream != 0) {
         if (bgm_state.id == BGM_ENDING ||
         bgm_state.id == BGM_TITLE ||
         bgm_state.id == BGM_FINISH ||
         bgm_state.id == BGM_TRAILER)
            bgm_state.playing = 0;
         else {
            ov_clear(&bgm_state.vorbis.file);
            bgm_state.restart = 1;
            decode_bgm(output_l, output_r);
         }
         return;
      }

      // Update total of samples read
      bgm_state.vorbis.buflen = (size_t) total;
      bgm_state.vorbis.buflen >>= bgm_state.stereo ? 2 : 1;
   }

   // Decode samples
   int32_t sample_l = 0, sample_r = 0;
   if (bgm_state.stereo) {
      size_t pos = bgm_state.vorbis.bufpos << 2;
      sample_l = -0x8000 +
         (bgm_state.vorbis.buffer[pos+0] << 8 |
         bgm_state.vorbis.buffer[pos+1]);
      sample_r = -0x8000 +
         (bgm_state.vorbis.buffer[pos+2] << 8 |
         bgm_state.vorbis.buffer[pos+3]);
   } else {
      size_t pos = bgm_state.vorbis.bufpos << 1;
      sample_l = sample_r = -0x8000 +
         (bgm_state.vorbis.buffer[pos+0] << 8 |
         bgm_state.vorbis.buffer[pos+1]);
   }

   // Mix BGM output
   *output_l += sample_l * settings.bgm_volume / 100;
   *output_r += sample_r * settings.bgm_volume / 100;
}

//***************************************************************************
// load_sfx [internal]
// Loads a sound effect from an OGG file into the passed structure. If it
// fails then the sound effect is left empty (game will continue to work if
// it's unable to play a sound effect).
//---------------------------------------------------------------------------
// param filename: name of input file
// param sfx: where to store sound effect
//***************************************************************************

static void load_sfx(const char *filename, SFXData *sfx) {
   // Initialize structure
   sfx->data = NULL;
   sfx->length = 0;
   sfx->rate = 0;

   // Open OGG file
   File *file = open_file(filename, FILE_READ);
   if (file == NULL) {
#ifdef DEBUG
      fprintf(stderr, "Warning: can't load sound effect \"%s\"\n",
         filename);
#endif
      return;
   }

   // Set up the functions we want vorbisfile to use
   // ...in other words, the read function :P
   ov_callbacks callbacks;
   callbacks.read_func = read_callback_sfx;
   callbacks.seek_func = NULL;
   callbacks.tell_func = NULL;
   callbacks.close_func = NULL;

   // Open vorbis stream
   OggVorbis_File vf;
   if (ov_open_callbacks(file, &vf, NULL, 0, callbacks)) {
#ifdef DEBUG
      fprintf(stderr, "Warning: can't load sound effect \"%s\"\n",
         filename);
#endif
      close_file(file);
      return;
   }

   // Get the information for the first bitstream
   vorbis_info *info = ov_info(&vf, -1);

   // Ugh x_x
   if (info->channels > 2) {
      ov_clear(&vf);
      close_file(file);
      return;
   }

   // OK, no many-channel shenanigans, store whether it's stereo or not
   sfx->stereo = info->channels == 2 ? 1 : 0;

   // Calculate rate at which samples advance
   // <Sik> The check here is to prevent overflow, but... is it even possible
   // to have sample rates this high? (that's 16777215Hz...) The check is
   // still needed anyway because otherwise somebody could make a mod that
   // hangs the system on purpose and we really don't want that.
   if (info->rate > 0xFFFFFF) {
      ov_clear(&vf);
      close_file(file);
      return;
   }
   sfx->rate = (uint64_t)(info->rate) * 0x10000 / 44100;

   // Scan the entire file
   size_t outpos = 0;
   for (;;) {
      // Decode the next chunk of samples
      uint8_t decbuf[DECBUFSIZE];
      int bitstream;
      long len = ov_read(&vf, (char *) decbuf, sizeof(decbuf), 1, 2, 0,
         &bitstream);

      // Stop here?
      if (len == 0 || bitstream != 0)
         break;

      // Error while reading?
      if (len < 0) {
#ifdef DEBUG
         fprintf(stderr, "Warning: can't load sound effect \"%s\"\n",
            filename);
#endif
         if (sfx->data) free(sfx->data);
         ov_clear(&vf);
         close_file(file);
         sfx->data = NULL;
         sfx->length = 0;
         return;
      }

      // Get number of samples it actually returned
      // The division by 2 is because the length is returned in bytes and
      // each sample is 2 bytes long (samples are 16-bit). Remove that
      // division for some funny distortion (as gaps are introduced)...
      unsigned num_samp = len/2 / info->channels;
      sfx->length += num_samp;

      // Make room for the extra samples
      {
         int16_t *temp = (int16_t *) realloc(sfx->data,
            sizeof(int16_t) * sfx->length * info->channels);
         if (temp == NULL) {
#ifdef DEBUG
            fprintf(stderr, "Warning: can't fit sound effect \"%s\"\n",
               filename);
#endif
            if (sfx->data) free(sfx->data);
            ov_clear(&vf);
            close_file(file);
            sfx->data = NULL;
            sfx->length = 0;
            return;
         } else
            sfx->data = temp;
      }

      // Process samples depending on the number of samples
      uint8_t *decptr = decbuf;

      switch (info->channels) {
         // Mono
         case 1:
            for (; num_samp > 0; num_samp--) {
               sfx->data[outpos] = -0x8000 +
                  (decptr[0] << 8) + decptr[1];
               outpos++;
               decptr += 2;
            }
            break;

         // Stereo
         case 2:
            for (; num_samp > 0; num_samp--) {
               sfx->data[outpos] = -0x8000 +
                  (decptr[0] << 8) + decptr[1];
               sfx->data[outpos+1] = -0x8000 +
                  (decptr[2] << 8) + decptr[3];
               outpos += 2;
               decptr += 4;
            }
            break;

         // This should not happen...
         // (really here just to make the compiler not complain)
         default:
            abort_program(ERR_UNKNOWN, NULL);
            return;
      }
   }

   // If playing using the beeper mode then make sound effects sound like
   // they're being output through a 1-bit speaker
   if (settings.beeper) {
      // Determine how many frames are in this SFX
      // Yes, we actually mean game frames (as in 60Hz)
      size_t framelen = info->rate * info->channels / 60;
      size_t framecount = sfx->length / framelen;

      // Truncate any excess
      sfx->length /= framecount;
      sfx->length *= framecount;

      // Process all frames
      int16_t min = 0;
      int16_t max = 0;
      size_t where = 0;
      for (size_t frame = 0; frame < framecount; frame++) {
         // Try to figure out the overall frequency in this frame
         int16_t framemin = 0;
         int16_t framemax = 0;
         unsigned count = 0;
         unsigned pos = 0;
         for (size_t i = 0; i < framelen; i++) {
            if (sfx->data[where+i] < framemin) framemin = sfx->data[where+i];
            if (sfx->data[where+i] > framemax) framemax = sfx->data[where+i];
            if (framemin < min) min = framemin;
            if (framemax > max) max = framemax;
            if (sfx->data[where+i] > 0x100) {
               if (!pos) count++;
               pos = 1;
            } else {
               //if (pos) count--;
               pos = 0;
            }
         }

         // Trying to keep some sanity around...
         if (framemax - framemin < 0x5000)
            count = 0;

         // Now fill the frame with that frequency
         uint64_t beeper = 0;
         uint64_t delta = count * 0x10000 / framelen;
         if (sfx->stereo) {
            for (size_t i = 0; i < framelen; i++) {
               sfx->data[where+i] =
               sfx->data[where+i+1] = (beeper & 0x8000) ? 0x4000 : 0x0000;
               beeper += delta + delta;
            }
         } else {
            for (size_t i = 0; i < framelen; i++) {
               sfx->data[where+i] = (beeper & 0x8000) ? 0x4000 : 0x0000;
               beeper += delta;
            }
         }

         // Get where next frame starts
         where += framelen;
      }

      // To prevent trailing chunks of sound at the end
      sfx->length -= sfx->length % framelen;

      // If the sound effect isn't loud enough just don't play it
      // Used to prevent small sound effects from drowning everything
      if (max - min < 0x4000) {
         free(sfx->data);
         sfx->data = NULL;
         sfx->length = 0;
         sfx->rate = 0;
      }
   }

   // Done with stream
   ov_clear(&vf);
   close_file(file);

#ifdef DEBUG
   // Report data about the sound effect was loaded
   /*
   printf("Sound effect \"%s\", length %lu, rate %uHz\n",
      filename, sfx->length, (unsigned) sfx->rate * 44100 >> 16);
   */
#endif
}

//***************************************************************************
// play_sfx
// Plays the sound effect with the specified ID (if loaded...)
//---------------------------------------------------------------------------
// param id: ID of sound effect to play
//***************************************************************************

void play_sfx(SfxID id) {
   play_sfx_ex(id, 0, 0, 0x100, 0, SFLAG_ABSOLUTE);
}

//***************************************************************************
// play_2d_sfx
// Like play_sfx, but applies 2D positioning to the sound effect. If in-game,
// the position will be adjusted to take the player location into account.
//---------------------------------------------------------------------------
// param id: ID of sound effect to play
// param x: X coordinate
// param y: Y coordinate
//***************************************************************************

void play_2d_sfx(SfxID id, int32_t x, int32_t y) {
   play_sfx_ex(id, x, y, 0x100, 0, SFLAG_REVERBOK);
}

//***************************************************************************
// play_sfx_ex
// The real function that takes care of playing sound effects.
//---------------------------------------------------------------------------
// param id: ID of sound effect to play
// param x: X coordinate
// param y: Y coordinate
// param vol: volume (0..0x100)
// param delay: how many samples to delay
// param flags: see SFLAG_*
//***************************************************************************

void play_sfx_ex(SfxID id, int32_t x, int32_t y, uint16_t vol,
uint64_t delay, uint32_t flags) {
   // *ahem*
   if (settings.no_sound)
      return;
   if (id == SFX_NONE)
      return;

   SFXData *sfx = &sfx_data[id];
   if (sfx->data == NULL)
      return;

   // Beeper mode? We gotta restrict ourselves a bit
   if (settings.beeper) {
      stop_sfx();
      x = 0;
      y = 0;
      flags = SFLAG_ABSOLUTE;
   }

   // This stuff is only alright in audio mode
   // It distracts too much in video mode
   if (!settings.audiovideo) {
      flags &= ~SFLAG_REVERBOK;
      flags |= SFLAG_NOWALLS;
   }

   // Reverberation from the ceiling?
   if ((flags & SFLAG_REVERBOK) && game_mode == GAMEMODE_INGAME) {
      // First of all, play the base sound
      flags &= ~SFLAG_REVERBOK;
      play_sfx_ex(id, x, y, vol, delay, flags);

      // Determine how deep should the reverberation be
      // If the ceiling is too high then it means no reverberation
      unsigned depth = 1;
      for (int32_t yr = y; depth < 4; depth++) {
         yr -= TILE_SIZE;
         TileType type = get_tile_by_pixel(x, yr)->collision;

         if (type == TILE_SOLID ||
         type == TILE_NESW_1 || type == TILE_NESW_1_BG ||
         type == TILE_NESW_2 || type == TILE_NESW_2_BG ||
         type == TILE_NWSE_1 || type == TILE_NWSE_1_BG ||
         type == TILE_NWSE_2 || type == TILE_NWSE_2_BG ||
         type == TILE_BELT_LEFT || type == TILE_BELT_RIGHT)
            break;
      }

      // Play echoes
      uint16_t faint = depth * (settings.audiovideo ? 0x20 : 0x40);
      while (vol > faint) {
         play_sfx_ex(id, x, y, vol - faint, delay, flags);
         delay += 3300;
         faint += 0x40;
      }

      // Already issued enough sounds
      return;
   }

   // These two can't happen at once
   // Since the logical thing is for them to neutralize each other (one
   // raises 2 semitones, the other lowers 2 semitones), may as well remove
   // them to avoid needless rounding errors
   if ((flags & SFLAG_HIPITCH) && (flags & SFLAG_LOPITCH)) {
      flags &= ~SFLAG_HIPITCH;
      flags &= ~SFLAG_LOPITCH;
   }

   // When in-game, adjust positions so that they follow the player
   if (!(flags & SFLAG_ABSOLUTE) && game_mode == GAMEMODE_INGAME) {
      Object *player = get_first_object(OBJGROUP_PLAYER);
      if (player != NULL) {
         if (player->dead) return;
         x -= player->x;
         y -= player->y;
      }

      // For consistency
      flags |= SFLAG_ABSOLUTE;
   }

   // Sounds fade out too quickly, so let's increase the coverage
   // We don't do this in audiovideo mode though because we need tight
   // location precision in that case (at the expense of a reduced
   // viewport, however)
   if ((!settings.audiovideo || game_mode != GAMEMODE_INGAME) &&
   !(flags & SFLAG_NOADJUST)) {
      x >>= 1;
      y >>= 1;
   }

   // Walls and such obstruct sound
   if (!(flags & SFLAG_NOWALLS) && !(flags & SFLAG_ABSOLUTE) &&
   game_mode == GAMEMODE_INGAME) {
      int32_t xr = x;
      int32_t yr = y;

      // Scan vertically
      while ((yr & ~TILE_SIZE_MASK) != (y & ~TILE_SIZE_MASK)) {
         LevelTile *tile = get_tile_by_pixel(xr, yr);
         uint8_t type = tile->collision;

         if ((type != TILE_EMPTY && type != TILE_EMPTY_BG &&
         type != TILE_NWSE_2 && type != TILE_NWSE_2_BG &&
         type != TILE_NESW_2 && type != TILE_NESW_2_BG &&
         type != TILE_HIDDEN) || !tile->obstruction)
            vol >>= 1;

         yr += yr > y ? TILE_SIZE : -TILE_SIZE;
      }

      // Scan horizontally
      while ((xr & ~TILE_SIZE_MASK) != (x & ~TILE_SIZE_MASK)) {
         LevelTile *tile = get_tile_by_pixel(xr, yr);
         uint8_t type = tile->collision;

         if ((type != TILE_EMPTY && type != TILE_EMPTY_BG &&
         type != TILE_NWSE_2 && type != TILE_NWSE_2_BG &&
         type != TILE_NESW_2 && type != TILE_NESW_2_BG &&
         type != TILE_HIDDEN) || !tile->obstruction)
            vol >>= 1;

         xr += xr > x ? TILE_SIZE : -TILE_SIZE;
      }

      // For consistency
      flags |= SFLAG_NOWALLS;
   }

   // Log sound effect
   if (is_recording())
      record_sfx(id, x, y, vol, delay, flags);

   // Don't waste time with obviously out-of-range sounds
   if (x < -0xC0 || x > 0xC0 || y < -0x80 || y > 0x80)
      return;

   // Determine playback speed
   uint64_t rate = sfx->rate;
   if (flags & SFLAG_HIPITCH)
      rate = rate * 1122 / 1000;
   if (flags & SFLAG_LOPITCH)
      rate = rate * 891 / 1000;

   // Get a new channel
   Channel *chan = &channels[next_chan];
   next_chan = (next_chan + 1) % NUM_CHAN;

   SDL_LockAudio();
   chan->sfx = sfx;
   chan->speed = rate;
   chan->where_l = 0;
   chan->where_r = 0;
   chan->delay_l = delay;
   chan->delay_r = delay;
   chan->active_l = 1;
   chan->active_r = 1;

   // Calculate 2D positioning
   pan_channel(chan, x, y, vol);
   SDL_UnlockAudio();
}

//***************************************************************************
// stop_sfx
// Stops all sound effects.
//***************************************************************************

void stop_sfx(void) {
   // Mark all channels as playing nothing
   SDL_LockAudio();
   for (size_t i = 0; i < NUM_CHAN; i++)
      channels[i].sfx = NULL;
   SDL_UnlockAudio();
}

//***************************************************************************
// read_callback_bgm [internal]
// Callback for reading BGM data for vorbisfile
//---------------------------------------------------------------------------
// param file: file to read from
// param size: size of each block
// param count: amount of blocks to read
// param buffer: where to store the data
// return: number of bytes read
//***************************************************************************

static size_t read_callback_bgm(void *buffer, size_t size, size_t count,
void *file) {
   // We don't need this actually...
   // (see: bgm_state)
   (void) file;

   // Because errno could contain some error from beforehand... (remember the
   // standard library likes to set errno on its own despite returning error
   // codes anyway)
   errno = 0;

   // Because we don't use two separate parameters...
   // To-do: check for overflow?
   size *= count;

   // See how much we can read
   size_t len = size;
   if (len > bgm_state.ogg.size - bgm_state.ogg.pos)
      len = bgm_state.ogg.size - bgm_state.ogg.pos;

   // Eh, end of data?
   if (len == 0)
      return 0;

   // Proceed to copy the data
   memcpy(buffer, &bgm_state.ogg.data[bgm_state.ogg.pos], len);
   bgm_state.ogg.pos += len;

   // Return amount of data we've read
   return len;
}

//***************************************************************************
// read_callback_sfx [internal]
// Callback for reading SFX data for vorbisfile
//---------------------------------------------------------------------------
// param file: file to read from
// param size: size of each block
// param count: amount of blocks to read
// param buffer: where to store the data
// return: number of bytes read
//***************************************************************************

static size_t read_callback_sfx(void *buffer, size_t size, size_t count,
void *file) {
   // Because errno could contain some error from beforehand... (remember the
   // standard library likes to set errno on its own despite returning error
   // codes anyway)
   errno = 0;

   // Because we don't use two separate parameters...
   // To-do: check for overflow?
   size *= count;

   // Try to read from the file
   size_t len = read_file_ex((File *) file, buffer, size);

   // If we didn't read enough data make sure it wasn't an error...
   if (len < size && !end_of_file((File *) file)) {
      len = 0;
      errno = -1;
   }

   // Return amount of data we've read
   return len;
}

//***************************************************************************
// get_bgm_by_name
// Gets the ID of a background music by its name
//---------------------------------------------------------------------------
// param name: name of background music
// return: ID of background music
//***************************************************************************

BgmID get_bgm_by_name(const char *name) {
   // Scan all BGMs
   for (size_t i = 0; i < NUM_BGM; i++)
      if (strcmp(bgm_names[i], name) == 0)
         return i;

   // Unknown BGM...
   return BGM_NONE;
}

//***************************************************************************
// get_sfx_by_name
// Gets the ID of a sound effect by its name
//---------------------------------------------------------------------------
// param name: name of sound effect
// return: ID of sound effect
//***************************************************************************

SfxID get_sfx_by_name(const char *name) {
   // Scan all SFXs
   for (size_t i = 0; i < NUM_SFX; i++)
      if (strcmp(sfx_info[i].name, name) == 0)
         return i;

   // Unknown SFX...
   return SFX_NONE;
}

//***************************************************************************
// bgm_exists
// Checks if a specified BGM exists (i.e. its file is available)
//---------------------------------------------------------------------------
// param id: ID of BGM
// return: non-zero if it exists, zero otherwise
//***************************************************************************

int bgm_exists(BgmID id) {
   // This technically is always available...
   // (when played it behaves as a BGM stop command)
   if (id == BGM_NONE)
      return 1;

   // ...and outbounds not
   if (id < 0 || id >= NUM_BGM)
      return 0;

   // Check if the file exists
   char buffer[0x40];
   sprintf(buffer, "gamefs/music/%s.ogg", bgm_names[id]);
   return PHYSFS_exists(buffer);
}

//***************************************************************************
// pan_channel [intenral]
// Calculates the volume of a channel based on its 2D position.
//---------------------------------------------------------------------------
// param chan: channel
// param x: X coordinate (relative to listener)
// param y: Y coordinate (relative to listener)
// param vol: volume (0..0x100)
//***************************************************************************

static void pan_channel(Channel *chan, int32_t x, int32_t y, uint16_t vol)
{
   int32_t vol_l = 0;
   int32_t vol_r = 0;

   // Calculate horizontal panning
   if (x >= -0xBF && x <= 0x3F)
      vol_l = sines[(x + 0xC0) >> 1];
   if (x >= -0x3F && x <= 0xBF)
      vol_r = sines[(x + 0x40) >> 1];

   // Calculate vertical panning
   if (y >= -0x7F && y <= 0x7F) {
      int ypan = sines[(y + 0x80) >> 1];
      vol_l *= ypan;
      vol_r *= ypan;
   } else {
      vol_l = 0;
      vol_r = 0;
   }

   // Calculate overall volume
   vol_l *= vol;
   vol_r *= vol;

   // Update channel
   chan->vol_l = vol_l >> 16;
   chan->vol_r = vol_r >> 16;
   chan->delay_l += x > 0 ? x : 0;
   chan->delay_r += x < 0 ? -x : 0;
}
