//***************************************************************************
// "scene.c"
// Handles the scene list. No, I really couldn't come up with a good place to
// store this thing...
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "file.h"
#include "level.h"
#include "parser.h"
#include "scene.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Definition of a scene
typedef struct {
   SceneType type;      // Type of scene
   char *id;            // ID of resource used by scene (if needed)
   char *name;          // Name of the scene
   unsigned number;     // Number (for the level counter)
} Scene;

// Where scene data is stored
static size_t num_scenes = 0;
static Scene *scenes = NULL;

// To keep track of which is the current scene
size_t curr_scene;

// To count how many levels are there
// Used to enumerate the levels
unsigned num_levels;

//***************************************************************************
// load_scene_list
// Loads and parses the scene list.
//***************************************************************************

void load_scene_list(void) {
   // Can't load scene list without a language since some stuff is localized,
   // come back when the scene list is not localized
   //if (settings.language == -1)
      //return;

#ifdef DEBUG
   fputs("Loading scene list\n", stderr);
#endif

   // Make sure the scene list is reset
   if (num_scenes != 0)
      unload_scene_list();
   num_levels = 0;

   // Open script
   File *file = open_file("system/scenes", FILE_READ);
   if (file == NULL)
      abort_program(ERR_LOADSCENES, NULL);

   // Go through all lines
   while (!end_of_file(file)) {
      // Get parameters for this line
      char *line = read_line(file);
      if (line == NULL)
         abort_program(ERR_LOADSCENES, NULL);
      Args *args = parse_args(line);
      if (args == NULL)
         abort_program(ERR_LOADSCENES, NULL);
      free(line);

      // Empty line?
      if (args->count == 0) {
         free_args(args);
         continue;
      }

      // Too many scenes?
      if (num_scenes == 0xFF)
         abort_program(ERR_NUMSCENES, NULL);

      // Allocate room for next scene
      num_scenes++;
      scenes = (Scene *) realloc(scenes, sizeof(Scene) * num_scenes);
      if (scenes == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      Scene *new_scene = &scenes[num_scenes-1];

      // End of game?
      if (!strcmp(args->list[0], "end")) {
         // Store scene properties
         new_scene->type = SCENE_END;
         new_scene->id = NULL;
         new_scene->name = NULL;
      }

      // Level?
      else if (!strcmp(args->list[0], "level") ||
      !strcmp(args->list[0], "bonus")) {
         // Mark scene type
         new_scene->type = (args->list[0][0] == 'l' ?
            SCENE_LEVEL : SCENE_BONUS);

         // Set level number if needed
         if (new_scene->type == SCENE_LEVEL) {
            num_levels++;
            new_scene->number = num_levels;
         }

         // Check parameters
         // To-do: explain the error?
         if (args->count != 3)
            abort_program(ERR_LOADSCENES, NULL);
         if (!is_valid_id(args->list[1]))
            abort_program(ERR_LOADSCENES, NULL);

         // We need to keep these strings...
         new_scene->id = malloc(strlen(args->list[1]) + 1);
         if (new_scene->id == NULL)
            abort_program(ERR_LOADSCENES, NULL);
         strcpy(new_scene->id, args->list[1]);

         new_scene->name = malloc(strlen(args->list[2]) + 1);
         if (new_scene->name == NULL)
            abort_program(ERR_LOADSCENES, NULL);
         strcpy(new_scene->name, args->list[2]);
      }

      // Cutscene?
      else if (!strcmp(args->list[0], "cutscene")) {
         // Mark scene type
         new_scene->type = SCENE_CUTSCENE;

         // Check parameters
         // To-do: explain the error?
         if (args->count != 2 && args->count != 3)
            abort_program(ERR_LOADSCENES, NULL);
         if (!is_valid_id(args->list[1]))
            abort_program(ERR_LOADSCENES, NULL);

         // We can't load the cutscene name yet
         if (settings.language == -1) {
            new_scene->name = NULL;
         }

         // Try to get cutscene name from its data
         // (allows for localization of its name)
         else if (args->count == 2) {
            // Pointer to whatever we're gonna use as the cutscene name
            char *name = NULL;

            // Try to open file with the cutscene script
            // Use its last line as the name of the cutscene
            char buffer[0x80];
            sprintf(buffer, "cutscenes/%s.%s", args->list[1],
               get_language_id());
            File *script = open_file(buffer, FILE_READ);
            if (script != NULL) {
               // Keep reading lines until we find the last one
               // Only take into account non-blank lines (this is done to
               // account for files that end in a newline, we don't want that
               // spurious extra line to be considered)
               while (!end_of_file(script)) {
                  char *line = read_line(script);
                  if (line == NULL) continue;
                  if (*line == '\0') { free(line); continue; }
                  if (name != NULL) free(name);
                  name = line;
               }

               // Done
               close_file(script);
            }

            // Got a name? Use it then
            if (name != NULL)
               new_scene->name = name;

            // Can't find it, just use the cutscene ID
            else {
               new_scene->name = malloc(strlen(args->list[1]) + 1);
               if (new_scene->name == NULL)
                  abort_program(ERR_LOADSCENES, NULL);
               strcpy(new_scene->name, args->list[1]);
            }
         }

         // Hardcoded cutscene name
         else {
            // Just store it as-is
            new_scene->name = malloc(strlen(args->list[2]) + 1);
            if (new_scene->name == NULL)
               abort_program(ERR_LOADSCENES, NULL);
            strcpy(new_scene->name, args->list[2]);
         }

         // We need to keep this string
         new_scene->id = malloc(strlen(args->list[1]) + 1);
         if (new_scene->id == NULL)
            abort_program(ERR_LOADSCENES, NULL);
         strcpy(new_scene->id, args->list[1]);
      }

      // Invalid scene! :(
      else {
#ifdef DEBUG
         fputs("wtf?\n", stderr);
#endif
         abort_program(ERR_LOADSCENES, NULL);
      }

      // Go for next line
      free_args(args);
   }

   // Done with script
   close_file(file);
}

//***************************************************************************
// switch_to_scene
// Switches the game mode to the specified scene.
//---------------------------------------------------------------------------
// param id: scene to switch to
//***************************************************************************

void switch_to_scene(size_t id) {
   // Assume we're already switching if fading off
   if (is_fading())
      return;

   // No replay plz
   settings.replay = 0;

   // Keep track of which scene we're on
   curr_scene = id;

   // Invalid scene?
   if (id >= num_scenes) {
      fade_off_and_switch(GAMEMODE_TITLE);
      return;
   }

   // Determine what to do based on the selected scene
   switch (scenes[id].type) {
      // End of game?
      case SCENE_END:
         fade_off_and_switch(GAMEMODE_TITLE);
         break;

      // Level?
      case SCENE_LEVEL:
      case SCENE_BONUS:
         reset_spawn_point();
         fade_off_and_switch(GAMEMODE_INGAME);
         break;

      // Cutscene?
      case SCENE_CUTSCENE:
         fade_off_and_switch(GAMEMODE_CUTSCENE);
         break;
   }
}

//***************************************************************************
// switch_to_next_scene
// Switches the game mode to the next scene.
//***************************************************************************

void switch_to_next_scene(void) {
   switch_to_scene(curr_scene + 1);
}

//***************************************************************************
// switch_to_first_level
// Switches to the first level in the game.
//***************************************************************************

void switch_to_first_level(void) {
   // Look for the first level scene
   unsigned id;
   for (id = 0; id < num_scenes; id++) {
      if (scenes[id].type == SCENE_LEVEL ||
      scenes[id].type == SCENE_BONUS)
         break;
   }

   // Huuuuuuuh
   if (id == num_scenes)
      fade_off_and_switch(GAMEMODE_TITLE);

   // Switch to this scene
   switch_to_scene(id);
}

//***************************************************************************
// get_scene_type
// Retrieves the type of the current scene.
//---------------------------------------------------------------------------
// returns: scene type
//***************************************************************************

SceneType get_scene_type(void) {
   return get_scene_type_by_id(curr_scene);
}

//***************************************************************************
// get_scene_type_by_id
// Retrieves the type of the specified scene.
//---------------------------------------------------------------------------
// param id: scene ID
// returns: scene type
//***************************************************************************

SceneType get_scene_type_by_id(size_t id) {
   if (id >= num_scenes)
      return SCENE_END;
   else
      return scenes[id].type;
}

//***************************************************************************
// get_scene_id
// Retrieves the ID associated with the current scene.
//---------------------------------------------------------------------------
// returns: scene ID value
//***************************************************************************

const char *get_scene_id(void) {
   if (curr_scene >= num_scenes)
      return NULL;
   else
      return scenes[curr_scene].id;
}

//***************************************************************************
// get_scene_name
// Retrieves the name associated with the current scene.
//---------------------------------------------------------------------------
// returns: scene name
//***************************************************************************

const char *get_scene_name(void) {
   if (curr_scene >= num_scenes)
      return NULL;
   else
      return scenes[curr_scene].name;
}

//***************************************************************************
// get_scene_number
// Retrieves the number associated with the current scene.
//---------------------------------------------------------------------------
// returns: scene number
//***************************************************************************

unsigned get_scene_number(void) {
   if (curr_scene >= num_scenes)
      return 0;
   else
      return scenes[curr_scene].number;
}

//***************************************************************************
// get_curr_scene
// Returns the current scene.
//---------------------------------------------------------------------------
// returns: current scene
//***************************************************************************

unsigned get_curr_scene(void) {
   return curr_scene;
}

//***************************************************************************
// get_num_scenes
// Returns how many scenes are in the game.
//---------------------------------------------------------------------------
// returns: number of scenes
//***************************************************************************

size_t get_num_scenes(void) {
   return num_scenes;
}

//***************************************************************************
// unload_scene_list
// Unloads the scene list (freeing up the resources it was using).
//***************************************************************************

void unload_scene_list(void) {
   // Deallocate scenes list
   for (size_t i = 0; i < num_scenes; i++) {
      if (scenes[i].id != NULL)
         free(scenes[i].id);
      if (scenes[i].name != NULL)
         free(scenes[i].name);
   }
   free(scenes);

   // Reset scene list
   scenes = NULL;
   num_scenes = 0;
}
