//***************************************************************************
// "saveselect.c"
// Code for the save select screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "main.h"
#include "background.h"
#include "input.h"
#include "loading.h"
#include "menu.h"
#include "reader.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// List of menu options
enum {
   OPT_PLAY,               // Start playing
   OPT_LEFT,               // Left arrow
   OPT_RIGHT,              // Right arrow
   OPT_CANCEL,             // Don't play
   MAX_OPT                 // Number of menu items
};

// Locations for the left and right buttons
#define LEFT_X1 (screen_cx-128)     // Left button: left
#define LEFT_Y1 (screen_cy-16)      // Left button: top
#define LEFT_X2 (LEFT_X1+31)        // Left button: right
#define LEFT_Y2 (LEFT_Y1+31)        // Left button: bottom

#define RIGHT_X1 (screen_cx+96)     // Right button: left
#define RIGHT_Y1 (screen_cy-16)     // Right button: top
#define RIGHT_X2 (RIGHT_X1+31)      // Right button: right
#define RIGHT_Y2 (RIGHT_Y1+31)      // Right button: bottom

// Location for the play button
#define PLAY_X1 (LEFT_X2 + 8)       // Left
#define PLAY_Y1 (screen_cy - 32)    // Top
#define PLAY_X2 (RIGHT_X1 - 8)      // Right
#define PLAY_Y2 (screen_cy + 31)    // Bottom

// Location for the cancel button
#define CANCEL_Y1 (screen_cy + 48)  // Top
#define CANCEL_Y2 (screen_cy + 63)  // Bottom

// Last scene we can visit
static size_t max_scene;

// Where graphics are stored
static GraphicsSet *gfxset_menu = NULL;
static Sprite *spr_left[2];
static Sprite *spr_right[2];

// Private function prototypes
static void init_saveselect_menu(void);
static void load_saveselect(void);
static void button_prev(void);
static void button_next(void);
static void button_start(void);
static void button_oneswitch(void);
static void button_back(void);

// One-switch stuff
static struct {
   unsigned selected;
   unsigned timer;
} oneswitch;

//***************************************************************************
// init_saveselect
// Initializes the save select screen.
//***************************************************************************

void init_saveselect(void) {
   // Get data from the savegame
   SaveGame savedata;
   get_savegame(&savedata);

   // Haven't even started the game yet?
   if (savedata.last_scene == 0) {
      switch_to_scene(0);
      return;
   }

   // Get last scene ever visited
   max_scene = savedata.last_scene;
   if (max_scene >= get_num_scenes())
      max_scene = get_num_scenes() - 1;

   // Default to the last visited scene
   curr_scene = savedata.curr_scene;
   if (curr_scene >= max_scene)
      curr_scene = max_scene;
   if (curr_scene >= get_num_scenes())
      curr_scene = 0;

   // Make sure we aren't in an "end" scene
   if (get_scene_type() == SCENE_END)
      button_next();

   // Load graphics
   loading_screen(load_saveselect);
   spr_left[0] = get_sprite(gfxset_menu, "arrow_left_off");
   spr_left[1] = get_sprite(gfxset_menu, "arrow_left");
   spr_right[0] = get_sprite(gfxset_menu, "arrow_right_off");
   spr_right[1] = get_sprite(gfxset_menu, "arrow_right");

   // Initialize menu data
   init_saveselect_menu();

   // Reset one-switch stuff
   oneswitch.selected = 0;

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();

   // Start playing music
   play_bgm(BGM_MENU);
}

//***************************************************************************
// init_saveselect_menu [internal]
// Sets up the menu in the level select.
//***************************************************************************

static void init_saveselect_menu(void) {
   init_menu();
   set_reinit_menu_func(init_saveselect_menu);

   menu.cancel = button_back;
   menu.defoption.up = OPT_CANCEL;
   menu.defoption.down = OPT_PLAY;
   menu.defoption.left = OPT_PLAY;
   menu.defoption.right = OPT_PLAY;

   if (!settings.audiovideo) {
      menu.options[OPT_LEFT].box.x1 = LEFT_X1;
      menu.options[OPT_LEFT].box.y1 = LEFT_Y1;
      menu.options[OPT_LEFT].box.x2 = LEFT_X2;
      menu.options[OPT_LEFT].box.y2 = LEFT_Y2;
      menu.options[OPT_LEFT].move.up = OPT_CANCEL;
      menu.options[OPT_LEFT].move.down = OPT_CANCEL;
      menu.options[OPT_LEFT].move.left = OPT_PLAY;
      menu.options[OPT_LEFT].move.right = OPT_PLAY;
      menu.options[OPT_LEFT].action.accept = button_prev;

      menu.options[OPT_RIGHT].box.x1 = RIGHT_X1;
      menu.options[OPT_RIGHT].box.y1 = RIGHT_Y1;
      menu.options[OPT_RIGHT].box.x2 = RIGHT_X2;
      menu.options[OPT_RIGHT].box.y2 = RIGHT_Y2;
      menu.options[OPT_RIGHT].move.up = OPT_CANCEL;
      menu.options[OPT_RIGHT].move.down = OPT_CANCEL;
      menu.options[OPT_RIGHT].move.left = OPT_PLAY;
      menu.options[OPT_RIGHT].move.right = OPT_PLAY;
      menu.options[OPT_RIGHT].action.accept = button_next;
   }

   menu.options[OPT_PLAY].box.x1 = PLAY_X1;
   menu.options[OPT_PLAY].box.y1 = PLAY_Y1;
   menu.options[OPT_PLAY].box.x2 = PLAY_X2;
   menu.options[OPT_PLAY].box.y2 = PLAY_Y2;
   menu.options[OPT_PLAY].move.up = OPT_CANCEL;
   menu.options[OPT_PLAY].move.down = OPT_CANCEL;
   menu.options[OPT_PLAY].move.oneswitch = OPT_CANCEL;
   menu.options[OPT_PLAY].action.left = button_prev;
   menu.options[OPT_PLAY].action.right = button_next;
   menu.options[OPT_PLAY].action.accept = button_start;
   menu.options[OPT_PLAY].action.oneswitch = button_oneswitch;

   menu.options[OPT_CANCEL].box.x1 = screen_cx -
      calc_text_len(text.saveselect.cancel) / 2 - 8;
   menu.options[OPT_CANCEL].box.y1 = CANCEL_Y1;
   menu.options[OPT_CANCEL].box.x2 = screen_cx +
      calc_text_len(text.saveselect.cancel) / 2 + 7;
   menu.options[OPT_CANCEL].box.y2 = CANCEL_Y2;
   menu.options[OPT_CANCEL].move.up = OPT_PLAY;
   menu.options[OPT_CANCEL].move.down = OPT_PLAY;
   menu.options[OPT_CANCEL].move.oneswitch = OPT_PLAY;
   menu.options[OPT_CANCEL].action.accept = button_back;
   menu.options[OPT_CANCEL].sfx = SFX_CANCEL;
}

//***************************************************************************
// load_saveselect [internal]
// Loads the assets for the save select screen. Run during the loading screen
//***************************************************************************

static void load_saveselect(void) {
   gfxset_menu = load_graphics_set("graphics/menus");
   set_loading_total(1, 3);
   load_background("graphics/menus/background", gfxset_menu);
   set_loading_total(2, 3);
   load_bgm(BGM_MENU);
   set_loading_total(3, 3);
}

//***************************************************************************
// run_saveselect
// Processes the logic for the saveslot select screen.
//***************************************************************************

void run_saveselect(void) {
   // Update the menu
   if (!oneswitch.selected)
      update_menu();

   // Update background animation
   update_background();

   // Selecting in one-switch mode?
   if (oneswitch.selected) {
      // Cycle levels
      oneswitch.timer--;
      if (oneswitch.timer == 0) {
         play_sfx(SFX_BEEP);
         oneswitch.timer = settings.menu_delay;
         button_next();
      }

      // Select level?
      if (input.oneswitch.tap || input.oneswitch.tap2) {
         play_sfx(SFX_OK);
         button_start();
      }
   }
}

//***************************************************************************
// draw_saveselect
// Draws the save select screen.
//***************************************************************************

void draw_saveselect(void) {
   // Draw background
   draw_background();

   // Draw arrow buttons
   draw_sprite(spr_left[menu.selected == OPT_LEFT ? 1 : 0],
      LEFT_X1, LEFT_Y1, SPR_NOFLIP);
   draw_sprite(spr_right[menu.selected == OPT_RIGHT ? 1 : 0],
      RIGHT_X1, RIGHT_Y1, SPR_NOFLIP);

   // Draw play button
   draw_rectangle(PLAY_X1, PLAY_Y1, PLAY_X2, PLAY_Y2,
      menu.selected == OPT_PLAY ? settings.box_lit[1] : settings.box_dim[1]);
   draw_rectangle(PLAY_X1+1, PLAY_Y1+1, PLAY_X2-1, PLAY_Y2-1,
      menu.selected == OPT_PLAY ? settings.box_lit[0] : settings.box_dim[0]);
   draw_rectangle(PLAY_X1+2, PLAY_Y1+2, PLAY_X2-2, PLAY_Y2-2,
      menu.selected == OPT_PLAY ? settings.box_lit[1] : settings.box_dim[1]);
   draw_hline(PLAY_X1+3, PLAY_Y1+3, PLAY_X2-3,
      menu.selected == OPT_PLAY ? settings.box_lit[2] : settings.box_dim[2]);
   draw_hline(PLAY_X1, PLAY_Y2+1, PLAY_X2,
      menu.selected == OPT_PLAY ? settings.box_lit[2] : settings.box_dim[2]);

   // Draw type of scene
   const char *str;
   switch (get_scene_type()) {
      case SCENE_LEVEL: str = text.saveselect.level; break;
      case SCENE_BONUS: str = text.saveselect.bonus; break;
      case SCENE_CUTSCENE: str = text.saveselect.cutscene; break;
      default: str = "???"; break;
   }
   draw_text_int(str, get_scene_number(), screen_cx, screen_cy,
      menu.selected >= OPT_PLAY && menu.selected <= OPT_RIGHT ?
      FONT_LIT : FONT_DIM, ALIGN_BOTTOM);

   // Draw name of current scene
   draw_text(get_scene_name(), screen_cx, screen_cy,
      menu.selected >= OPT_PLAY && menu.selected <= OPT_RIGHT ?
      FONT_LIT : FONT_DIM, ALIGN_TOP);

   // Draw cancel button
   draw_text(text.saveselect.cancel, screen_cx, CANCEL_Y1,
      menu.selected == OPT_CANCEL ? FONT_LIT : FONT_DIM, ALIGN_TOP);

   // Draw title
   draw_text(text.saveselect.title, screen_cx, 0x20,
      FONT_LIT, ALIGN_CENTER);

   // Show selected option in the screen reader
   switch (menu.selected) {
      // Show current level
      case OPT_PLAY:
      case OPT_LEFT:
      case OPT_RIGHT: {
         // For now...
         set_reader_text(get_scene_name());
      } break;

      // Cancel button
      case OPT_CANCEL:
         set_reader_text(text.saveselect.cancel);
         break;

      // Nothing to show...
      default:
         break;
   }
}

//***************************************************************************
// deinit_saveselect
// Deinitializes the save select screen.
//***************************************************************************

void deinit_saveselect(void) {
   unload_all_bgm();
   unload_background();
   destroy_graphics_set(gfxset_menu);
}

//***************************************************************************
// button_prev [internal]
// Selects the previous scene.
//***************************************************************************

static void button_prev(void) {
   // Select the previous available scene
   do {
      if (curr_scene == 0)
         curr_scene = max_scene+1;
      curr_scene--;
   } while (get_scene_type() == SCENE_END);
}

//***************************************************************************
// button_next [internal]
// Selects the next scene.
//***************************************************************************

static void button_next(void) {
   // Select the next available scene
   do {
      curr_scene++;
      if (curr_scene > max_scene)
         curr_scene = 0;
   } while (get_scene_type() == SCENE_END);
}

//***************************************************************************
// button_start [internal]
// Starts the game.
//***************************************************************************

static void button_start(void) {
   // Go to the ingame mode!
   switch_to_scene(curr_scene);
}

//***************************************************************************
// button_oneswitch [internal]
// Starts level rotation in one-switch mode.
//***************************************************************************

static void button_oneswitch(void) {
   // Go into one-switch level select
   oneswitch.selected = 1;
   oneswitch.timer = settings.menu_delay * 3 / 2;

   // Don't select yet...
   input.oneswitch.tap = 0;
   input.oneswitch.tap2 = 0;
}

//***************************************************************************
// button_back [internal]
// Function for the "Back" button. Returns back to the title screen.
//***************************************************************************

static void button_back(void) {
   // If we were going to record, this means we decided to give up instead,
   // so quit the game in that case
   if (settings.record)
      fade_off_and_switch(GAMEMODE_QUIT);

   // Return back to the title screen otherwise
   else
      fade_off_and_switch(GAMEMODE_TITLE);
}
