//***************************************************************************
// "replay.c"
// Code for handling replays
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <stdio.h>
#include "main.h"
#include "file.h"
#include "input.h"
#include "replay.h"
#include "scene.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Bits for each input
#define IN_UP        0x01
#define IN_DOWN      0x02
#define IN_LEFT      0x04
#define IN_RIGHT     0x08
#define IN_ACTION    0x10

// Where frames are stored
#define MAX_FRAMES (30*60)
#define END_FRAME (MAX_FRAMES-30)
static uint8_t replay[MAX_FRAMES];
static unsigned current_frame;
static int loaded = 0;

// Determine which replay to load next
static unsigned next_replay = 0;

// Valid header for a replay
static const uint8_t header[] = { 0x1A, 0x73, 0x4F, 0x6C, 0x01, 0x00 };

// Scene and difficulty used in the replay
static unsigned scene;
static Difficulty difficulty;

// Private function prototypes
static void fake_input(unsigned, unsigned);

//***************************************************************************
// load_replay
// Tries to load the demo replay.
//***************************************************************************

void load_replay(void) {
   // Nope
   loaded = 0;

   // Open file
   next_replay++;
   char filename[0x60];
   sprintf(filename, "system/demo%u.solr", next_replay);
   File *file = open_file(filename, FILE_READ);
   if (file == NULL) {
      // It's possible the file doesn't exist, in which case we ran out of
      // replays and we should try again the first... although if it's that
      // one the one that failed we should give up then
      if (next_replay != 1) {
         next_replay = 0;
         load_replay();
      }
      return;
   }

   // Get header and check if it's valid
   uint8_t buffer[8];
   if (read_file(file, buffer, 8)) {
      close_file(file);
      return;
   }
   if (memcmp(buffer, header, 6) != 0) {
      close_file(file);
      return;
   }
   if (buffer[6] > get_num_scenes() ||
   (get_scene_type_by_id(buffer[6]) != SCENE_LEVEL &&
   get_scene_type_by_id(buffer[6]) != SCENE_BONUS)) {
      close_file(file);
      return;
   }
   if (buffer[7] > 0x02) {
      close_file(file);
      return;
   }

   // Get scene and difficulty
   scene = buffer[6];
   difficulty = buffer[7];

   // Read replay
   if (read_file(file, replay, MAX_FRAMES)) {
      close_file(file);
      return;
   }

   // Done
   close_file(file);
   loaded = 1;
}

//***************************************************************************
// start_replay_playback
// Starts playing the demo replay
//***************************************************************************

void start_replay_playback(void) {
   // Er...
   if (!loaded)
      return;

   // Start game with replay turned on
   //switch_to_first_level();
   switch_to_scene(scene);
   settings.replay = 1;

   // Start from the beginning :P
   current_frame = 0;
}

//***************************************************************************
// update_replay_playback
// Fakes player input from the replay
//***************************************************************************

void update_replay_playback(void) {
   // Get current frame
   uint8_t data = replay[current_frame];
   current_frame++;

   // Fake button presses
   fake_input(PL_INPUT_UP, (data & IN_UP) ? 1 : 0);
   fake_input(PL_INPUT_DOWN, (data & IN_DOWN) ? 1 : 0);
   fake_input(PL_INPUT_LEFT, (data & IN_LEFT) ? 1 : 0);
   fake_input(PL_INPUT_RIGHT, (data & IN_RIGHT) ? 1 : 0);
   fake_input(PL_INPUT_ACTION, (data & IN_ACTION) ? 1 : 0);

   // Time to quit?
   if (current_frame == END_FRAME)
      fade_off_and_switch(GAMEMODE_TITLE);
   if (!settings.attract &&
   (input.rawpress.keyboard || input.rawpress.joystick))
      fade_off_and_switch(GAMEMODE_TITLE);
}

//***************************************************************************
// fake_input [internal]
// Fakes player input during a replay.
//---------------------------------------------------------------------------
// param id: ID of input (see PL_INPUT_*)
// param value: set if pressed, clear if released
//***************************************************************************

static void fake_input(unsigned id, unsigned value) {
   // Pressed?
   if (value) {
      input.player.press[id] = !input.player.hold[id];
      input.player.hold[id] = 1;
   }

   // Released?
   else {
      input.player.press[id] = 0;
      input.player.hold[id] = 0;
   }
}

//***************************************************************************
// get_replay_difficulty
// Returns the difficulty used for this replay.
//---------------------------------------------------------------------------
// return: difficulty
//***************************************************************************

Difficulty get_replay_difficulty(void) {
   return difficulty;
}

//***************************************************************************
// start_replay_recording
// Starts recording a replay
//***************************************************************************

void start_replay_recording(void) {
   // Start from the beginning :P
   current_frame = 0;
}

//***************************************************************************
// update_replay_recording
// Updates a recording replay
//***************************************************************************

void update_replay_recording(void) {
   // Generate new frame data
   uint8_t data = 0;

   if (input.player.hold[PL_INPUT_UP])
      data |= IN_UP;
   if (input.player.hold[PL_INPUT_DOWN])
      data |= IN_DOWN;
   if (input.player.hold[PL_INPUT_LEFT])
      data |= IN_LEFT;
   if (input.player.hold[PL_INPUT_RIGHT])
      data |= IN_RIGHT;
   if (input.player.hold[PL_INPUT_ACTION])
      data |= IN_ACTION;

   // Store frame
   replay[current_frame] = data;
   current_frame++;

   // Time to quit?
   if (current_frame == END_FRAME)
      fade_off_and_switch(GAMEMODE_QUIT);
}

//***************************************************************************
// draw_replay_hud
// Draws the replay timer
//***************************************************************************

void draw_replay_hud(void) {
   // Generate string
   char buffer[0x10];
   sprintf(buffer, "%u.%02u", current_frame / 60, current_frame % 60);

   // Draw it
   draw_text(buffer, screen_w - 0x10, 0x08, FONT_LIT, ALIGN_TOPRIGHT);
}

//***************************************************************************
// save_replay
// Writes recorded replay to disk
//***************************************************************************

void save_replay(void) {
   // Open file
   // To-do: check for errors
   File *file = open_save_file("savedata/replay.solr", FILE_WRITE);

   // Write header
   // To-do: check for errors
   uint8_t buffer[2];
   buffer[0] = get_curr_scene();
   buffer[1] = settings.difficulty;

   write_file(file, header, 6);
   write_file(file, buffer, 2);

   // Write frame data
   // To-do: check for errors
   write_file(file, replay, MAX_FRAMES);

   // Done
   close_file(file);
}
